##################################################################################
#   __   __      _                       ______                _____            
#   \ \ / /     | |                      |  _  \              |  __ \           
#    \ V / _ __ | | ___  _ __ __ _ ______| | | |___  ___ _ __ | |  \/ ___  ___  
#    /   \| '_ \| |/ _ \| '__/ _` |______| | | / _ \/ _ \ '_ \| | __ / _ \/ _ \ 
#   / /^\ \ |_) | | (_) | | | (_| |      | |/ /  __/  __/ |_) | |_\ \  __/ (_) |
#   \/   \/ .__/|_|\___/|_|  \__,_|      |___/ \___|\___| .__/ \____/\___|\___/ 
#         | |                                           | |                     
#         |_|                                           |_|                     
##################################################################################

import os
import subprocess
import shutil
import re
import time
import sys

# Modalità di esecuzione: 'binary' (default) oppure 'tiff'
MODE = 'binary'

# NEW: Funzione per convertire i percorsi Windows in percorsi WSL
def convert_to_wsl_path(win_path):
    formatted_path = win_path.replace("\\", "/")
    try:
        wsl_path = subprocess.check_output(['wsl', 'wslpath', '-a', formatted_path]).decode().strip()
        # print(f"Percorso WSL convertito: {wsl_path}")  # Debug
        return wsl_path
    except subprocess.CalledProcessError as e:
        # print(f"Errore nella conversione del percorso per WSL: {e}")
        return None


# Funzione per assicurarsi che CMake venga eseguito solo una volta
def ensure_cmake(build_dir_wsl, source_dir_wsl):
    makefile_path = os.path.join(build_dir, 'Makefile')
    if not os.path.exists(makefile_path):
        print("Makefile non trovato. Generazione con CMake...")
        run_cmake(build_dir_wsl, source_dir_wsl)
    else:
        print("Makefile già esistente. Salto la generazione con CMake.")

def run_cmake(build_dir_wsl, source_dir_wsl):
    print("Generating Makefile with CMake...")
    try:
        # Esegui CMake
        subprocess.run([
            'wsl', 'cmake', '-S', source_dir_wsl, '-B', build_dir_wsl
        ], check=True)
        print("CMake completed successfully.")
    except subprocess.CalledProcessError as e:
        print(f"Errore durante l'esecuzione di CMake: {e}")




# Funzione per eliminare tutto il contenuto di una cartella
def clear_folder(folder_path):
    if os.path.exists(folder_path):
        for filename in os.listdir(folder_path):
            file_path = os.path.join(folder_path, filename)
            try:
                if os.path.isdir(file_path):
                    shutil.rmtree(file_path)  # Elimina la cartella
                else:
                    os.remove(file_path)  # Elimina il file
            except Exception as e:
                print(f"Errore nell'eliminare {file_path}: {e}")


# Funzione per eseguire il make e l'eseguibile
def run_make_and_executable(folder_path_wsl, resampled_subfolder_wsl, json_full_path_wsl, extra_args: str = ""):
    print("Make...")
    try:
        # Esegui make
        subprocess.run(['wsl', 'make', '-C', build_dir_wsl], check=True)
        print("Running...")

        output_dir_wsl = convert_to_wsl_path(resampled_folder)

        cmd = f'{executable_wsl} "{folder_path_wsl}" "{resampled_subfolder_wsl}" "{json_full_path_wsl}" {extra_args}'.strip()
        subprocess.run(['wsl', 'bash', '-c', cmd], check=True)

    except subprocess.CalledProcessError as e:
        print(f"Errore durante l'esecuzione: {e}")

# Funzione per copiare le immagini nella nuova directory
def copy_images(start_idx, end_idx, input_subfolder):
    folder_name = f"rilievo{images[start_idx][-10:-5]}-{images[end_idx][-10:-5]}"
    folder_path = os.path.join(input_dir, folder_name)
    os.makedirs(input_subfolder, exist_ok=True)

    for i in range(start_idx, end_idx + 1):
        src_image = os.path.join(input_dir, images[i])
        dst_image = os.path.join(input_subfolder, images[i])
        shutil.copy(src_image, dst_image)


# NEW: directory base
base_dir = os.path.abspath(os.path.join(os.path.dirname(__file__), '..'))

# Variabili
start_time = time.perf_counter()
"""
Modalità BINARIA: scansione batch di TempData/GuiTest per tutte le Swath*/VV/SubSwath*/tomography
Modalità TIFF: usa input_dir come cartella singola di immagini TIFF
"""
# Percorso base di scansione (puoi sostituire "GuiTest" con il progetto corrente)
scan_root = os.path.abspath(os.path.join(base_dir, '..', 'TempData', 'GuiTest'))
if not os.path.isdir(scan_root):
    # Fallback: usa TempData generico e poi cerca Swath* dentro i progetti
    scan_root = os.path.abspath(os.path.join(base_dir, '..', 'TempData'))
# Fallback per modalità TIFF singola (non usato in binary)
input_dir = os.path.join(base_dir, '..', 'TempData', 'GuiTest', 'Swath001', 'VV', 'SubSwath00', 'tomography')
config_file = os.path.join(base_dir, 'C', 'src', 'utils', 'configurations', 'configSTD.hh')
build_dir = os.path.join(base_dir, 'C', 'build')
# print(f"build_dir: {build_dir}")
executable = os.path.join(build_dir, 'compiledVersion')
resampled_folder = os.path.join(base_dir, 'Cartelle', 'Resampled_C')  # usato solo in modalità TIFF
output_json = os.path.join(base_dir, 'Cartelle', 'Json_C')
source_dir = os.path.join(base_dir, 'C')  # La directory con CMakeLists.txt

# NEW: conversione percorsi per WSL
build_dir_wsl = convert_to_wsl_path(build_dir)
executable_wsl = convert_to_wsl_path(executable)
source_dir_wsl = convert_to_wsl_path(source_dir)


overlap = 30  # Numero di immagini da sovrapporre
batch_size = 40  # Numero di immagini per batch

os.makedirs(output_json, exist_ok=True)
if MODE == 'tiff':
    # Clear solo in modalità TIFF
    clear_folder(resampled_folder)
    clear_folder(output_json)

ensure_cmake(build_dir_wsl, source_dir_wsl)

# Branch per modalità BINARIA vs TIFF
if MODE == 'binary':
    # Scansione Swath -> VV -> tutte le SubSwath
    swath_re = re.compile(r'^Swath(\d+)$', re.IGNORECASE)

    def swath_num(name: str) -> int:
        m = swath_re.match(name)
        return int(m.group(1)) if m else -1

    if not os.path.isdir(scan_root):
        raise FileNotFoundError(f"Scan root non trovato: {scan_root}")

    # Raccogli tutte le cartelle Swath* sia direttamente sotto scan_root
    # sia un livello sotto (TempData/<Project>/Swath*)
    swath_paths = []  # lista di (swath_name, swath_abs_path)
    direct = [d for d in os.listdir(scan_root) if swath_re.match(d) and os.path.isdir(os.path.join(scan_root, d))]
    for d in direct:
        swath_paths.append((d, os.path.join(scan_root, d)))
    # Cerca un livello sotto
    projects = [p for p in os.listdir(scan_root) if os.path.isdir(os.path.join(scan_root, p)) and not swath_re.match(p)]
    for proj in projects:
        proj_path = os.path.join(scan_root, proj)
        for d in os.listdir(proj_path):
            if swath_re.match(d) and os.path.isdir(os.path.join(proj_path, d)):
                swath_paths.append((d, os.path.join(proj_path, d)))

    if not swath_paths:
        print(f"Nessuna Swath* trovata sotto {scan_root}")
        swaths = []
    else:
        # Ordina per numero
        swath_paths.sort(key=lambda t: swath_num(t[0]))
        swaths = swath_paths

    converter_py = os.path.join(base_dir, 'C', 'numpy_to_binary.py')
    binary_folder = os.path.join(base_dir, 'C', 'binary_tomography')
    os.makedirs(binary_folder, exist_ok=True)

    env = os.environ.copy()
    env['PYTHONIOENCODING'] = 'utf-8'
    py = sys.executable or 'python'

    for swath_name, swath_path in swaths:
        vv_dir = os.path.join(swath_path, 'VV')
        if not os.path.isdir(vv_dir):
            print(f"Skip {swath_name}: VV mancante")
            continue

        subs = [d for d in os.listdir(vv_dir) if d.lower().startswith('subswath') and os.path.isdir(os.path.join(vv_dir, d))]
        subs.sort()
        if not subs:
            print(f"Skip {swath_name}: nessuna SubSwath in VV")
            continue

        for sub in subs:
            tomography_dir = os.path.join(vv_dir, sub, 'tomography')
            if not os.path.isdir(tomography_dir):
                print(f"Skip {swath_name}/{sub}: tomography mancante")
                continue

            # 1) Conversione NumPy -> Bin
            print(f"[MODE=binary] Conversione: {swath_name}/VV/{sub}")
            subprocess.run([py, converter_py, '--input', tomography_dir, '--output', binary_folder], check=True, env=env)

            # 2) JSON per SubSwath
            json_filename = f"{swath_name}_VV_{sub}.json"
            json_path = os.path.join(output_json, json_filename)
            os.makedirs(output_json, exist_ok=True)

            # 3) Esecuzione compiledVersion
            input_dir_wsl = convert_to_wsl_path(tomography_dir)
            json_path_wsl = convert_to_wsl_path(json_path)
            binary_file_wsl = convert_to_wsl_path(os.path.join(binary_folder, 'tomography_data.bin'))
            resampled_dummy_wsl = '/tmp/resampled'
            extra_args = f"--mode=binary --bin={binary_file_wsl}"
            run_make_and_executable(input_dir_wsl, resampled_dummy_wsl, json_path_wsl, extra_args=extra_args)

else:
    # Modalità TIFF: batching come prima
    image_pattern = re.compile(r'.*_([0-9]{3})cm\.tiff$')
    images = [f for f in os.listdir(input_dir) if image_pattern.match(f)]
    images.sort()
    total_images = len(images)
    print(f"Numero totale di immagini: {total_images}")

    # CICLO SUI BATCH
    for start_idx in range(0, total_images - batch_size + 1, overlap):
        end_idx = start_idx + batch_size - 1

        folder_name = f"rilievo{images[start_idx][-10:-5]}-{images[end_idx][-10:-5]}"
        input_subfolder = os.path.join(input_dir, folder_name)
        os.makedirs(input_subfolder, exist_ok=True)
        copy_images(start_idx, end_idx, input_subfolder)

        resampled_subfolder = os.path.join(resampled_folder, folder_name)
        os.makedirs(resampled_subfolder, exist_ok=True)

        input_subfolder_wsl = convert_to_wsl_path(input_subfolder)
        resampled_subfolder_wsl = convert_to_wsl_path(resampled_subfolder)

        json_filename = f"rilievo{start_idx:03d}-{end_idx:03d}.json"
        json_path = os.path.join(output_json, json_filename)
        json_path_wsl = convert_to_wsl_path(json_path)

        run_make_and_executable(input_subfolder_wsl, resampled_subfolder_wsl, json_path_wsl, extra_args="--mode=tiff")

end_time = time.perf_counter()
execution_time = end_time - start_time# Calcola ore, minuti e secondi
hours = int(execution_time // 3600)
minutes = int((execution_time % 3600) // 60)
seconds = int(execution_time % 60)

print(f"Tempo di esecuzione: {hours} ore, {minutes} minuti, {seconds} secondi")