#!/usr/bin/env python3
"""
Script per convertire file NumPy (.npy) in formato binario semplice per C++
Formato binario:
- Header: [width:uint32][height:uint32][depth:uint32][data_type_size:uint32]
- Data: array di float32 consecutivi [width*height*depth]
"""

import numpy as np
import os
import time
import struct
import argparse
from pathlib import Path

def convert_numpy_to_binary(numpy_dir, output_dir):
    """
    Converte tutti i file .npy in una directory in formato binario semplice
    
    Args:
        numpy_dir: Directory contenente i file .npy
        output_dir: Directory dove salvare i file binari
    """
    start_time = time.perf_counter()
    
    # Crea directory di output se non esiste
    os.makedirs(output_dir, exist_ok=True)
    
    # Trova tutti i file .npy
    numpy_files = []
    for file in os.listdir(numpy_dir):
        if file.endswith('.npy'):
            numpy_files.append(file)
    
    # Ordina per profondità
    numpy_files.sort(key=lambda x: int(x.split('_')[1].split('cm')[0]))
    
    print(f"🔄 Conversione di {len(numpy_files)} file NumPy...")
    
    # Carica tutti i file per determinare le dimensioni
    all_data = []
    shapes = []
    
    load_start = time.perf_counter()
    for i, filename in enumerate(numpy_files):
        filepath = os.path.join(numpy_dir, filename)
        data = np.load(filepath)
        
        # Converti in float32 per compatibilità C++
        data = data.astype(np.float32)
        all_data.append(data)
        shapes.append(data.shape)
        
        if i % 50 == 0:
            print(f"  📁 Caricato {i+1}/{len(numpy_files)}: {filename} - Shape: {data.shape}")
    
    load_time = time.perf_counter() - load_start
    print(f"⏱️  Tempo caricamento NumPy: {load_time:.3f}s")
    
    # Verifica che tutti i file abbiano la stessa forma
    if len(set(shapes)) > 1:
        raise ValueError(f"Tutti i file devono avere la stessa forma. Trovate: {set(shapes)}")
    
    width, height = shapes[0]
    depth = len(numpy_files)
    
    print(f"📊 Dimensioni volume: {width} x {height} x {depth}")
    print(f"💾 Dimensione totale: {width * height * depth * 4 / (1024*1024):.2f} MB")
    
    # Crea file binario unico
    binary_start = time.perf_counter()
    binary_filename = os.path.join(output_dir, "tomography_data.bin")
    
    with open(binary_filename, 'wb') as f:
        # Header: width, height, depth, data_type_size (4 bytes per float32)
        header = struct.pack('IIII', width, height, depth, 4)
        f.write(header)
        
        # Scrivi tutti i dati consecutivamente
        for data in all_data:
            f.write(data.tobytes())
    
    binary_time = time.perf_counter() - binary_start
    
    # Crea anche file di metadati per debug
    metadata_filename = os.path.join(output_dir, "metadata.txt")
    with open(metadata_filename, 'w', encoding='utf-8') as f:
        f.write(f"# Metadati conversione NumPy -> Binario\n")
        f.write(f"# Generato: {time.strftime('%Y-%m-%d %H:%M:%S')}\n")
        f.write(f"# File originali: {len(numpy_files)}\n")
        f.write(f"# Dimensioni: {width} x {height} x {depth}\n")
        f.write(f"# Tipo dati: float32\n")
        f.write(f"# Tempo caricamento: {load_time:.3f}s\n")
        f.write(f"# Tempo scrittura: {binary_time:.3f}s\n")
        f.write(f"# Tempo totale: {time.perf_counter() - start_time:.3f}s\n")
        f.write(f"\n# File originali:\n")
        for filename in numpy_files:
            f.write(f"# {filename}\n")
    
    total_time = time.perf_counter() - start_time
    
    print(f"✅ Conversione completata!")
    print(f"📁 File binario: {binary_filename}")
    print(f"📄 Metadati: {metadata_filename}")
    print(f"⏱️  Tempi:")
    print(f"   - Caricamento NumPy: {load_time:.3f}s")
    print(f"   - Scrittura binario: {binary_time:.3f}s")
    print(f"   - Totale: {total_time:.3f}s")
    print(f"🚀 Velocità: {len(numpy_files) / total_time:.1f} file/s")
    
    return binary_filename, metadata_filename

def verify_binary_file(binary_file):
    """Verifica che il file binario sia stato creato correttamente"""
    print(f"\n🔍 Verifica file binario: {binary_file}")
    
    if not os.path.exists(binary_file):
        print("❌ File binario non trovato!")
        return False
    
    with open(binary_file, 'rb') as f:
        # Leggi header
        header = f.read(16)  # 4 * uint32
        width, height, depth, data_size = struct.unpack('IIII', header)
        
        print(f"📊 Header: {width} x {height} x {depth}, data_size={data_size}")
        
        # Calcola dimensione attesa
        expected_size = 16 + (width * height * depth * data_size)
        actual_size = os.path.getsize(binary_file)
        
        print(f"📏 Dimensione file: {actual_size} bytes (attesa: {expected_size})")
        
        if actual_size == expected_size:
            print("✅ File binario corretto!")
            return True
        else:
            print("❌ File binario corrotto!")
            return False

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Converti tomografie NumPy (.npy) in binario unico")
    parser.add_argument("--input", dest="input_dir", default="../../TempData/GuiTest/Swath001/HH/SubSwath00/tomography", help="Cartella con i file .npy")
    parser.add_argument("--output", dest="output_dir", default="./binary_tomography", help="Cartella di output per il file binario")
    args = parser.parse_args()

    numpy_dir = args.input_dir
    output_dir = args.output_dir

    print("Conversione NumPy -> Binario")
    print(f"Input: {numpy_dir}")
    print(f"Output: {output_dir}")
    print("-" * 50)

    try:
        binary_file, metadata_file = convert_numpy_to_binary(numpy_dir, output_dir)
        verify_binary_file(binary_file)
    except Exception as e:
        print(f"Errore: {e}")
        import traceback
        traceback.print_exc()
