/**
 * __   __      _                       ______                _____
 * \ \ / /     | |                      |  _  \              |  __ \
 *  \ V / _ __ | | ___  _ __ __ _ ______| | | |___  ___ _ __ | |  \/ ___  ___
 *  /   \| '_ \| |/ _ \| '__/ _` |______| | | / _ \/ _ \ '_ \| | __ / _ \/ _ \
 * / /^\ \ |_) | | (_) | | | (_| |      | |/ /  __/  __/ |_) | |_\ \  __/ (_) |
 * \/   \/ .__/|_|\___/|_|  \__,_|      |___/ \___|\___| .__/ \____/\___|\___/
 *       | |                                           | |
 *       |_|                                           |_|
 */

#include <iostream>
#include <filesystem>  // Per directory_iterator
#include <fstream>     // Per testFile
#include "./utils/Matrix3D.hh"
#include "./utils/Matrix3DAnalysis.hh"
#include "./utils/Point3D.hh"
#include "./utils/PCAResult.hh"
#include "./utils/ImageLoader.hh"
#include "./utils/graph.hh"
#include "compiledVersion.hh"

#ifdef USE_STD
#include "utils/configurations/configSTD.hh"
#endif
#ifdef USE_ALT1
#include "utils/configurations/configALT1.hh"
#endif

int main(int argc, char* argv[]){

    if (argc < 4)
    {
        std::cerr << "Usage: " << argv[0]
                  << " <inputDir> <outputDir> <jsonPath> [--mode=binary|tiff] [--bin=/path/to/tomography_data.bin]" << std::endl;
        return 1;
    }

    // Ricava i path dal comando lanciato da Python
    std::string inputDir = argv[1];
    std::string outputDir = argv[2];
    std::string jsonPath = argv[3];

    ImageLoader loader;

    // Default: use binary mode unless user forces TIFF
    std::string mode = "binary"; // default
    std::string binaryFileOverride = "";

    // Parse optional flags
    for (int i = 4; i < argc; ++i) {
        std::string arg = argv[i];
        if (arg.rfind("--mode=", 0) == 0) {
            mode = arg.substr(7);
        } else if (arg.rfind("--bin=", 0) == 0) {
            binaryFileOverride = arg.substr(6);
        }
    }

    if (mode == "binary") {
        // Solo binario: nessun fallback
        std::string binaryFile = binaryFileOverride.empty()
            ? std::string("/mnt/c/Users/Valer/Desktop/lavoro/GUI_DEEPGEO_SW/Algorithm/C/binary_tomography/tomography_data.bin")
            : binaryFileOverride;
        std::ifstream testFile(binaryFile);
        if (!testFile.good()) {
            std::cerr << "[MODE=binary] ERRORE: file binario non trovato: " << binaryFile << std::endl;
            return 2;
        }
        std::cout << "[MODE=binary] Caricamento file binario: " << binaryFile << std::endl;
        loader.loadBinaryTomography(binaryFile);
        testFile.close();
    } else {
        // Carica immagini TIFF tradizionali
        if (PREPROCESSING) {
            // Esegui il resize georeferenziato da inputDir a outputDir
            loader.processGeoreferencedImages(inputDir, outputDir, 0.02);
            // Poi carica le immagini già ridimensionate
            loader.loadImages(outputDir);
        } else {
            // Carica direttamente le immagini dalla cartella di input
            loader.loadImages(inputDir);
        }
    }
    loader.applyGauss();


    Matrix3D<float>& imageMatrix = loader.getMatrix();

    cout << "Pre flatten ----" << endl;
    imageMatrix.flatten();
    cout << "Post flatten ---" << endl;

    // CREAZIONE DEL VISUALIZZATORE | GRANDEZZA DATA DALLA DIMENSIONE DELLA MATRICE
    DeepGeo dg;
    //Passaggio dei parametri di offset
    dg.offsetUp=loader.top;
    dg.offsetDown=loader.bottom;
    dg.offsetDx=loader.right;
    dg.offsetSx=loader.left;
    dg.offsetZ=loader.z;

    dg.addMatrix(&imageMatrix);
    dg.matrix=imageMatrix;
    try {
        // Implementa il tuo algoritmo qui
        cout << "Esecuzione dell'Analisi..." << endl;

        Matrix3DAnalysis<float> analysis(dg.matrix);
        analysis.maxAnalysis();

        dg.matrix.setCodici(analysis.getCodici());
        dg.matrix.codFlatten();

        dg.matrix.setFlag(analysis.getFlag());
        dg.matrix.flagFlatten();


        vector<PCAResult> pcaResults = analysis.getPCAResults();
        vector<Line> stats = analysis.getStats();
        vector<Line> statsBar = analysis.getStatsBarycenter();

        dg.addPcaResult(pcaResults);


        dg.localMaxList = analysis.localMaxList;
        dg.localMaxSize = analysis.localMaxSize;
        dg.statCoppie = analysis.getStatsCoppieBar();

        //visualizer.runShortestPath1 = true;

        cout << "Analisi completata con successo." << endl;

        if(dg.getPcaResult().empty()){
            return 0;
        }
        cout<< "------------------------------------"<<endl;
        cout<<"Graph Analysis..."<<endl;

        Graph g = Graph(dg.getPcaResult(), dg.statCoppie, jsonPath);
        //offset
        g.offsetUp=dg.offsetUp;
        g.offsetDown=dg.offsetDown;
        g.offsetLeft=dg.offsetSx;
        g.offsetRight=dg.offsetDx;
        g.offsetZ=dg.offsetZ;

        g.addMatrix(dg.matrix);
        g.addMaxLocals(dg.getMaxPoints());
        g.startShortestPath();
        cout<<"Execution done!"<<endl;
    } catch (const std::out_of_range& e) {
        std::cerr << "Errore: Indice della matrice fuori dai limiti: " << e.what() << std::endl;
    } catch (const std::exception& e) {
        std::cerr << "Errore durante l'esecuzione dell'algoritmo: " << e.what() << std::endl;
    } catch (...) { // Catch super generico
        std::cerr << "Errore sconosciuto durante l'esecuzione dell'algoritmo." << std::endl;
    }

    return 0;
}