/**
 * __   __      _                       ______                _____            
 * \ \ / /     | |                      |  _  \              |  __ \           
 *  \ V / _ __ | | ___  _ __ __ _ ______| | | |___  ___ _ __ | |  \/ ___  ___  
 *  /   \| '_ \| |/ _ \| '__/ _` |______| | | / _ \/ _ \ '_ \| | __ / _ \/ _ \ 
 * / /^\ \ |_) | | (_) | | | (_| |      | |/ /  __/  __/ |_) | |_\ \  __/ (_) |
 * \/   \/ .__/|_|\___/|_|  \__,_|      |___/ \___|\___| .__/ \____/\___|\___/ 
 *       | |                                           | |                     
 *       |_|                                           |_|                     
 *
 * @file Evaluation.hh
 * @brief Contiene una serie di funzioni template per il calcolo di statistiche descrittive su vettori di dati numerici.
 * 
 * Questo file fornisce implementazioni di funzioni per calcolare le seguenti statistiche:
 * - Media
 * - Mediana
 * - Varianza
 * - Deviazione standard
 * - Minimo e Massimo
 * - Range
 * - Asimmetria (Skewness)
 * - Kurtosi
 * - Coefficiente di variazione
 * - Covarianza tra due vettori
 * - Quartili (Q1 e Q3)
 * - Intervallo interquartile (IQR)
 * - Percentile
 * - Moda
 * 
 * Le funzioni sono implementate come template generici 
 * per consentire l'utilizzo di vector con qualsiasi tipo numerico 
 * (ad esempio, `int`, `float`, `double`).
 * 
 * 
 * @authors I favolosi ragazzi del Georadar (Akira), Xplora srl
 * 
 */

#ifndef EVALUATION_HH
#define EVALUATION_HH
#include <iostream>
#include <vector>
#include <cmath>
#include<algorithm>
#include <float.h>

using namespace std;

/**
 * @brief Calcola la media di un vector
 */
template <typename T>
inline T calculateMean(const vector<T> &data){
    if (data.empty()) {
        throw std::domain_error("Il vettore è vuoto, impossibile calcolare la media.");
    }
    int n=data.size();
    // Calcolo della media
    T somma = 0;
    for (T valore : data) {
        somma += valore;
    }
    return somma / n;
}
/**
 * @brief Calcola la mediana di un vector
 */
template <typename T>
inline T calculateMedian(const std::vector<T> &data) {
    if (data.empty()) {
        throw std::domain_error("Il vettore è vuoto, impossibile calcolare la mediana.");
    }
    std::vector<float> dataSort=data;
    // Ordina il vettore
    std::sort(dataSort.begin(), dataSort.end());

    size_t size = dataSort.size();
    if (size % 2 == 0) {
        // Se il numero di elementi è pari, la mediana è la media dei due valori centrali
        return (dataSort[size / 2 - 1] + dataSort[size / 2]) / 2.0f;
    } else {
        // Se il numero di elementi è dispari, la mediana è il valore centrale
        return dataSort[size / 2];
    }
}
/**
 * @brief calcola la varianza di un vector
 * 
 * @details La varianza è il valore medio del valore di cui
 * un campione si discosta dalla media
 */
template <typename T>
inline T calculateVariance(const vector<T>& data) {
    int n = data.size();
    if (n == 0) {
        std::cerr << "Il vettore è vuoto!" << std::endl;
        return -1;  // Errore se il vettore è vuoto
    }
    //Calcolo della media
    T media=calculateMean(data);

    // Calcolo della varianza
    T varianza = 0;
    for (T valore : data) {
        varianza += (valore - media) * (valore - media);
    }
    varianza /= n;  // Varianza per la popolazione (usare n-1 per il campione)

    return varianza;
}

/**
 * @brief Calcola deviazione standard di un vector
 * 
 * @details La deviazione standard è la radice quadrata della varianza 
 */
template <typename T>
inline T calculateStdDeviation(const std::vector<T>& data){
    T variance=calculateVariance(data);
    return sqrt(variance);
}

/**
 * @brief Calcola Minimo e Massimo di un vector
 */
template <typename T>
inline void calculateMinMax(const vector<T>& data, T *minimo, T *massimo) {
    *minimo = std::numeric_limits<T>::max();  // Inizializzazione con il massimo valore possibile
    *massimo = std::numeric_limits<T>::lowest();  // Inizializzazione con il minimo valore possibile

    for (int i = 0; i < data.size(); i++) {
        if (data[i] < *minimo) {
            *minimo = data[i];
        }
        if (data[i] > *massimo) {
            *massimo = data[i];
        }
    }
}

/**
 * @brief Calcola Minimo di un vector
 */
template <typename T>
inline T calculateMin(const vector<T> &data) {
    T minimo = std::numeric_limits<T>::lowest();  // Inizializzazione con il massimo valore possibile
    for (int i = 0; i < data.size(); i++) {
        if (data[i] < minimo) {
            minimo = data[i];
        }
    }
    return minimo;
}
/**
 * @brief Calcola Massimo di un vector
 */
template <typename T>
inline T calculateMax(const vector<T> &data) {
    T max = std::numeric_limits<T>::max();  // Inizializzazione con il massimo valore possibile
    for (int i = 0; i < data.size(); i++) {
        if (data[i] > max) {
            max = data[i];
        }
    }
    return max;
}

//Calcola il range di un vector
template <typename T>
inline T calcolaRange(const vector<T> &data) {
    T minimo, massimo;
    calculateMinMax(data, &minimo, &massimo);
    return massimo - minimo;
}

/**
 * @brief Calcola Skewness (Asimmetria) di un vector
 * 
 * @details 
 * La skewness misura la simmetria del dataset. Un valore di skewness:
 *  -Positivo indica che la distribuzione è asimmetrica verso destra (la coda della distribuzione è più lunga a destra).
 *  -Negativo indica che la distribuzione è asimmetrica verso sinistra (la coda è più lunga a sinistra).
 *  -Zero indica una distribuzione simmetrica.
*/
template <typename T>
inline T calculateSkewness(const vector<T>& data) {
    T media= calculateMean(data);
    T deviazioneStandard= calculateStdDeviation(data);
    T skewness = 0.0;
    int n= data.size();
    for (int i = 0; i < n; i++) {
        skewness += pow((data[i] - media) / deviazioneStandard, 3);
    }
    skewness *= (n / ((n - 1) * (n - 2)));
    return skewness;
}
/**
 * @brief Calcola la Kurtosi di un vector
 * 
 * @details 
 * La kurtosis misura la "puntezza" della distribuzione dei dati. Può essere interpretata come una misura della "concentrazione" dei dati attorno alla media.
 *  -Kurtosi alta (positivo): la distribuzione ha code più pesanti, quindi ci sono più valori estremi.
 *  -Kurtosi bassa (negativo): la distribuzione ha code più leggere, quindi ci sono pochi valori estremi.
 *  -Una distribuzione normale ha una kurtosi di circa 3 (valore mesokurtico). Se si sottrae 3 dal valore ottenuto, si ottiene una misura di "excess kurtosis".
 */
template <typename T>
inline T calculateKurtosis(const vector<T>& data) {
    T kurtosis = 0.0;
    T media=calculateMean(data);
    T deviazioneStandard=calculateStdDeviation(data);
    int n= data.size();
    for (int i = 0; i < n; i++) {
        kurtosis += pow((data[i] - media) / deviazioneStandard, 4);
    }
    kurtosis *= (n * (n + 1)) / ((n - 1) * (n - 2) * (n - 3));
    kurtosis -= (3 * pow(n - 1, 2)) / ((n - 2) * (n - 3));
    return kurtosis;
}
/**
 * @brief Calcola il coefficiente di variazione in un vector
 * 
 * @details il coefficiente percentuale di rapporto tra 
 * deviazione standard e media
 */
template <typename T>
inline T calculateCOV(const vector<T>& data){
    return ((calculateStdDeviation(data)/calculateMean(data)) *100);
}

/**
 * @brief Calcola la covarianza tra due vector
 * 
 * @param dataX Primo vettore di dati
 * @param dataY Secondo vettore di dati
 * @return La covarianza tra i due vettori
 * 
 * @details La covarianza misura quanto due variabili cambiano insieme.
 */
template <typename T>
inline T calculateCovariance(const vector<T>& dataX, const vector<T>& dataY) {
    if (dataX.size() != dataY.size()) {
        throw std::invalid_argument("I due vettori devono avere la stessa lunghezza.");
    }

    int n = dataX.size();
    T meanX = calculateMean(dataX);
    T meanY = calculateMean(dataY);
    T covariance = 0.0f;

    for (int i = 0; i < n; i++) {
        covariance += (dataX[i] - meanX) * (dataY[i] - meanY);
    }

    return covariance / n;
}

/**
 * @brief Calcola il primo (Q1) e il terzo quartile (Q3) di un vector
 * 
 * @details Il 25% dei dati è inferiore al primo quartile Q1
 *          Il 75% dei dati è inferiore al terzo quartile Q3
 * 
 * @param data Vettore di dati
 * @param Q1 Primo quartile (25%)
 * @param Q3 Terzo quartile (75%)
 */
template <typename T>
inline void calculateQuartiles(const vector<T>& data, T &Q1, T &Q3) {
    if (data.empty()) {
        throw std::domain_error("Il vettore è vuoto, impossibile calcolare i quartili.");
    }

    vector<T> sortedData = data;
    std::sort(sortedData.begin(), sortedData.end());

    int n = sortedData.size();
    
    // Calcolo del primo quartile (Q1)
    int Q1Index = n / 4;
    Q1 = sortedData[Q1Index];

    // Calcolo del terzo quartile (Q3)
    int Q3Index = 3 * n / 4;
    Q3 = sortedData[Q3Index];
}

/**
 * @brief Calcola l'Intervallo Interquartile (IQR) di un vector
 * 
 * @param data Vettore di dati
 * @return L'intervallo interquartile (IQR)
 */
template <typename T>
inline T calculateIQR(const vector<T>& data) {
    T Q1, Q3;
    calculateQuartiles(data, Q1, Q3);
    return Q3 - Q1;
}

/**
 * @brief Calcola un percentile di un vector
 * 
 * @param data Vettore di dati ordinato
 * @param percentile Il percentile da calcolare (tra 0 e 100)
 * @return Il valore corrispondente al percentile nel vettore
 */
template <typename T>
inline T calculatePercentile(const vector<T>& data, T percentile) {
    if (data.empty()) {
        throw std::domain_error("Il vettore è vuoto, impossibile calcolare il percentile.");
    }
    if (percentile < 0 || percentile > 100) {
        throw std::invalid_argument("Il valore del percentile deve essere tra 0 e 100.");
    }

    vector<T> sortedData = data;
    std::sort(sortedData.begin(), sortedData.end());

    T rank = (percentile / 100) * (sortedData.size() - 1);  // Calcola il rank (indice) del percentile
    int lowerIndex = static_cast<int>(rank);  // Indice del valore inferiore
    int upperIndex = lowerIndex + 1;  // Indice del valore superiore

    if (upperIndex >= sortedData.size()) {
        // Se l'indice superiore è fuori dai limiti, restituisci l'ultimo valore
        return sortedData[lowerIndex];
    }

    // Interpolazione lineare tra i due valori
    T lowerValue = sortedData[lowerIndex];
    T upperValue = sortedData[upperIndex];
    T fractionalPart = rank - lowerIndex;
    return lowerValue + fractionalPart * (upperValue - lowerValue);
}

/**
 * @brief Calcola la moda di un vector
 * @details La moda è il valore con frequenza massima nel vettore
 */
template <typename T>
inline float calculateMode(const vector<T>& vec) {
    unordered_map<T, int> frequenze;

    // Conta le frequenze di ciascun elemento
    for (T num : vec) {
        frequenze[num]++;
    }

    // Trova l'elemento con la frequenza massima
    T moda = vec[0];
    int maxFreq = frequenze[moda];
    
    for (const auto& [valore, freq] : frequenze) {
        if (freq > maxFreq) {
            maxFreq = freq;
            moda = valore;
        }
    }
    
    return moda;
}


#endif