#include "Graph.hh"
#include "Matrix3D.hh"
#include "Point3D.hh"
#include "Line.hh"
#include "RadMatrixElement.hh"
#include "Evaluation.hh"
#include "pixelParams.hh"
#include <cmath>
#include <errno.h>
#include <fstream>

#ifdef USE_STD
#include "configurations/configSTD.hh"
#endif
#ifdef USE_ALT1
#include "configurations/configALT1.hh"
#endif

//Scala per tarare rapporto tra X/Y e Z
#define SCALA M_PER_PIXEL_Z/M_PER_PIXEL_X


float Graph::norm(Point3D& P){
    return sqrt(P.x * P.x + P.y * P.y + (P.z*SCALA * P.z*SCALA));
}


/**
 * Calcola una regressione lineare tridimensionale per un insieme di punti specificati dagli indici `from` a `to` nell'array `elenco`.
 * Utilizza la formula dei minimi quadrati per determinare il piano che meglio approssima i punti dati in termini di errore quadratico medio.
 *
 * @param elenco Vettore di indici dei nodi nel grafo, che indica i punti per i quali eseguire la regressione.
 * @param from Indice iniziale dell'intervallo nel vettore `elenco` da considerare per la regressione.
 * @param to Indice finale (esclusivo) dell'intervallo nel vettore `elenco` da considerare.
 *
 * @return Point3D Oggetto Point3D dove le coordinate x e y rappresentano i coefficienti del piano di regressione, e z è impostato a -1.
 *                 Il piano di regressione è espresso dalla forma: z = ax + by + c. I coefficienti a e b sono calcolati e c è implicitamente zero.
 *
 * @details I punti sono presi dal vettore `nodi` usando gli indici forniti in `elenco`, accumulando i valori necessari per il calcolo della regressione.
 * La regressione lineare tridimensionale è approssimata calcolando i parametri di un piano che minimizza la distanza dai punti dati.
 */

Point3D Graph::linearRegression(vector<int> &elenco, int from, int to)
{
    double sumX =0, sumY = 0, sumZ = 0;
    double sumXY = 0, sumXZ = 0, sumYZ = 0;
    double sumXX = 0, sumYY = 0, sumZZ = 0;

    for (int i= from; i< to; i++)
    {
        Point3D a = nodi[elenco[i]].baricentro;
        sumX += a.getX();
        sumY += a.getY();
        sumZ += a.getZ();

        sumXY += a.getX() * a.getY();
        sumXZ += a.getX() * a.getZ();
        sumYZ += a.getY() * a.getZ();

        sumXX += a.getX() * a.getX();
        sumYY += a.getY() * a.getY();
        sumZZ += a.getZ() * a.getZ();
    }

    double d = ((sumXX * sumYY) - (sumXY * sumXY));
    double a = 0;
    double b = 0;
    if (d != 0.0)
    {
        a = ((sumYZ * sumXX) - (sumXY * sumXZ)) / d;
        b = ((sumYZ * sumXY) - (sumYY * sumXZ)) / d;
    }

    return Point3D(a, b, -1.0f);
}


/**
 * @brief Calcola la distanza massima tra i baricentri dei nodi.
 *
 * La funzione scorre attraverso tutti i nodi e calcola la distanza euclidea tra i baricentri
 * di ogni coppia di nodi. La distanza massima trovata viene restituita come risultato.
 *
 * @return float La distanza massima tra i baricentri di tutti i nodi.
 *
 * @details
 * - La funzione utilizza due cicli annidati per confrontare ogni coppia di nodi e calcolare la distanza
 *   euclidea tra i loro baricentri.
 * - Viene mantenuta una variabile `maxDist` per tracciare la distanza massima trovata durante l'esecuzione.
 * - La distanza tra due baricentri `nodi[i]` e `nodi[j]` viene calcolata come:
 *   dist = sqrt{(dx^2 + dy^2 + dz^2)}, dove `dx`, `dy`, e `dz` sono le differenze delle coordinate x, y e z.
 * - La funzione restituisce `maxDist` al termine del ciclo, che rappresenta la maggiore distanza trovata tra i nodi.
 */
float Graph::getMaxDistance()
{
    float maxDist = 0.0;
    for (int i = 0; i < nodi.size(); i++) {
        for (int j = i + 1; j < nodi.size(); j++) {
            float dx = nodi[i].baricentro.x - nodi[j].baricentro.x;
            float dy = nodi[i].baricentro.y - nodi[j].baricentro.y;
            float dz = (nodi[i].baricentro.z - nodi[j].baricentro.z);
            float dist = sqrt(dx * dx + dy * dy + dz*SCALA * dz*SCALA);
            if (dist > maxDist) {
                maxDist = dist;
            }
        }
    }
    return maxDist;
}

/**
 * @brief Calcola la distanza euclidea tra due punti nello spazio tridimensionale.
 *
 * La funzione prende come input le differenze delle coordinate `dx`, `dy`, e `dz` tra due punti
 * e calcola la loro distanza euclidea utilizzando la formula:
 * \f$ \text{distanza} = \sqrt{dx^2 + dy^2 + dz^2} \f$.
 *
 * @param dx Differenza tra le coordinate x dei due punti.
 * @param dy Differenza tra le coordinate y dei due punti.
 * @param dz Differenza tra le coordinate z dei due punti.
 * @return float La distanza euclidea calcolata tra i due punti.
 *
 * @details
 * - La funzione utilizza la formula standard della distanza euclidea nello spazio 3D.
 * - Restituisce un valore `float` che rappresenta la distanza tra i due punti.
 * - È utile per calcolare la distanza quando si conoscono direttamente le differenze delle coordinate.
 */
float Graph::distance(float dx, float dy, float dz)
{
    return sqrt(dx * dx + dy * dy + dz * dz);
}

/**
 * @brief Calcola la distanza euclidea tra due punti 3D.
 *
 * La funzione prende come input due oggetti `Point3D`, calcola la differenza delle loro coordinate x, y e z,
 * e restituisce la distanza euclidea tra i due punti utilizzando la formula:
 * \f$ \text{distanza} = \sqrt{(x_1 - x_2)^2 + (y_1 - y_2)^2 + (z_1 - z_2)^2} \f$.
 *
 * @param p1 Primo punto 3D.
 * @param p2 Secondo punto 3D.
 * @return float La distanza euclidea calcolata tra `p1` e `p2`.
 *
 * @details
 * - La funzione utilizza la formula della distanza euclidea nello spazio tridimensionale.
 * - Restituisce un valore `float` che rappresenta la distanza tra i due punti.
 * - È utile quando si desidera calcolare la distanza tra due oggetti `Point3D` specificati.
 */
float Graph::distance(const Point3D &p1, const Point3D &p2)
{
    return sqrt(pow(p1.x - p2.x, 2) + pow(p1.y - p2.y, 2) + pow((p1.z - p2.z)*SCALA, 2));
}



/////////////////////////////////////////
//////////////// PUBLIC /////////////////
/////////////////////////////////////////

/**
 * @brief Costruttore per creare un grafo utilizzando risultati PCA e statistiche di connessione.
 *
 * La funzione crea un grafo utilizzando un vettore di risultati PCA (`pcaResults`) per rappresentare i nodi
 * e una matrice di statistiche (`stats`) per determinare le connessioni tra i nodi. Il costruttore inizializza
 * i nodi e crea le connessioni (archi) basate sui punteggi presenti nella matrice di statistiche.
 *
 * @param pcaResults Vettore di oggetti `PCAResult` che rappresentano i nodi del grafo.
 * @param stats Matrice 2D di oggetti `Stats` che contengono le statistiche per determinare i punteggi delle connessioni tra i nodi.
 *
 * @details
 * - Il costruttore inizializza i nodi del grafo copiando i risultati PCA (`pcaResults`) in `this->nodi`.
 * - Le strutture `edges` e `vertex` vengono ridimensionate per adattarsi al numero di nodi.
 * - Per ogni coppia di nodi `(i, j)`, se il punteggio `stats[i][j].score` è maggiore di 0, viene creato un arco da `i` a `j`.
 * - Per garantire la simmetria delle connessioni, se non esiste un arco da `j` a `i` (ossia `stats[j][i].score <= 0`), viene aggiunto un arco simmetrico da `j` a `i`.
 * - Ogni arco è rappresentato da un oggetto `Edge`, contenente il nodo di destinazione, il punteggio dell'arco e un peso di default (1).
 */
Graph::Graph(vector<PCAResult> pcaResults, vector<vector<Stats>> stats, const string& jsonOutputPath){
    this->nodi = pcaResults;
    edges.resize(pcaResults.size());
    vertex.resize(pcaResults.size());
    this->m_jsonPath = jsonOutputPath;

    for(int i = 0; i < pcaResults.size(); i++){
        for(int j = 0; j < pcaResults.size(); j++) {
            if( i!=j && stats[i][j].score > 0 && (stats[j][i].score <= 0 || (stats[j][i].score > 0 && i < j))) {
                // Aggiungi un edge da i a j e tra j
                edges[i].push_back(Edge(j, stats[i][j].score, 1));
                edges[j].push_back(Edge(i, stats[i][j].score, 1));
            }
        }
    }
}

/**
 * @brief Aggiunge una matrice 3D al grafo.
 *
 * La funzione assegna una matrice 3D di tipo `float` al grafo, memorizzandola come membro della classe.
 * Questo permette al grafo di utilizzare la matrice come parte della sua struttura dati per analisi o calcoli futuri.
 *
 * @param matrix Riferimento a un oggetto `Matrix3D<float>` che rappresenta la matrice 3D da aggiungere al grafo.
 *
 * @details
 * - La funzione semplicemente assegna l'oggetto `matrix` passato come argomento alla variabile membro `this->matrix`.
 * - Questo metodo permette di aggiornare o aggiungere una matrice 3D associata al grafo, rendendola accessibile per ulteriori operazioni.
 */
void Graph::addMatrix(Matrix3D<float>& matrix){
    this->matrix=matrix;
}

/**
 * @brief Aggiunge una lista di punti locali massimi al grafo.
 *
 * La funzione accetta un vettore di oggetti `Point3D` che rappresentano i punti locali massimi e
 * li memorizza come variabile membro della classe. Questa lista può essere utilizzata per
 * identificare o analizzare i punti di interesse nel contesto del grafo.
 *
 * @param maxList Vettore di oggetti `Point3D` che rappresentano i punti locali massimi da aggiungere al grafo.
 *
 * @details
 * - La funzione copia i valori del vettore `maxList` passato come argomento nella variabile membro `this->maxList`.
 * - Questo metodo consente di aggiornare o assegnare una nuova lista di punti massimi locali al grafo,
 *   rendendoli disponibili per operazioni di analisi e calcolo.
 */
void Graph::addMaxLocals(vector<Point3D> maxList){
    this->maxList=maxList;
}

/**
 * @brief Calcola il percorso più breve in un grafo da un nodo iniziale specificato (`from`) utilizzando una
 *        struttura MinHeap come coda di priorità per gestire i nodi visitati.
 *
 * La funzione determina la distanza minima da `from` agli altri nodi non esclusi, sfruttando il valore della direzione
 * e la distanza euclidea tra nodi. Si distingue per il calcolo di un "cilindro" intorno al percorso trovato, usato
 * per identificare e marcare i nodi da escludere in base alla loro posizione rispetto al percorso ottimale.
 *
 * @param from Indice del nodo di partenza.
 * @param heap Struttura dati MinHeap per mantenere l'ordine dei nodi non ancora visitati in base alla distanza.
 */

// Bar
void Graph::shortestPath(int from, MinHeap &heap)
{
    //cout << "from " << from << endl;

    float val = 0;
    //cout << "calcolo val: " << endl;
    if (nodi[from].baricentro.x < matrix.getX() && nodi[from].baricentro.y < matrix.getY() && nodi[from].baricentro.z < matrix.getZ())
    {
        val = matrix((int)nodi[from].baricentro.x, (int)nodi[from].baricentro.y, (int)nodi[from].baricentro.z);
    }
    else
    {
        cout << "X: " << nodi[from].baricentro.x << " XMatrix: " << matrix.getX() << endl;
        cout <<"Errore: indice della matrice fuori dai limiti" << endl;
        val = 0.0;
    }
    //cout << "Val: " << val << endl;

    vector<float> migliorAttualeV;

    // inizializzo i nodi
    for (int i = 0; i < nodi.size(); i++){
        Vertex &ver = vertex[i]; // Vertex è un vettore che contiene tutti i nodi del grafo
        ver.distance = heap.INF; // inizializzato con distanza infinita
        ver.visited = false;
        ver.prev = -1; // nessun predecessore

        if (ver.flagEscluso)
            continue;

        // calcolo distanza euclidea tra nodo corrente e nodo di partenza
        float dx = nodi[i].baricentro.x - nodi[from].baricentro.x;
        float dy = nodi[i].baricentro.y - nodi[from].baricentro.y;
        float dz = (nodi[i].baricentro.z - nodi[from].baricentro.z)*SCALA;
        ver.distEuclidea = sqrt(dx * dx + dy * dy + dz * dz);
    }
    vertex[from].distance = 0; // distanza nodo da se stesso

    heap.reset();
    heap.insert(from, 0);

    while (!heap.isEmpty())
    {
        int debug = 0;
        //Estraggo il nodo con valore minimo
        long idU = heap.extractMinID();

        Vertex &u = vertex[idU];
        if (u.visited && !u.flagEscluso){
            continue;
        }
        u.visited = true;

        // direzione rappresentata dall'autovettore massimo delle PCA nelle tre coordinate xyz se non è presente un nodo precedente
        Point3D direzione = Point3D(nodi[from].autovetMax[0],nodi[from].autovetMax[1],nodi[from].autovetMax[2]);

        // se c'è il predecessore calcolo facendo la differenza tra coordinate dell'attuale e coordinate del predecessore, la differenza rappresenta
        // il vettore che punta dal predecessore al corrente
        if (u.prev != -1){
            direzione = Point3D(nodi[idU].baricentro-nodi[u.prev].baricentro);
            direzione.normalize();
        }

        // itero sugli archi del nodo corrente, per ogni arco analizzo il nodo di destinazione
        for(const Edge& e : edges[idU]){

            int id = e.to; // nodo destinazione
            Vertex &v = vertex[id]; // nodo destinazione nel grafo

            float val_id = matrix((int)nodi[id].baricentro.x, (int)nodi[id].baricentro.y, (int)nodi[id].baricentro.z);

            float penalty = pow(1 - fabs(val - val_id), EXP_PENALTY);

            if (debug)
                cout << "entrato " <<"idU: " << idU <<"id: " << id << "direction: " << e.direction << endl;

            if (v.flagEscluso)
            {
                continue;
            }
            Point3D b = Point3D(nodi[id].baricentro-nodi[idU].baricentro); // vettore che punta dal nodo corrente idU al nodo destinazione id

            float d = norm(b); //distanza tra idU e id
            b.normalize(); // normalizzo b in un vettore unitario (lunghezza 1) per tenere conto solo della direzione

            float cosTheta = direzione * b; // prodotto scalare tra vettore direzione e arco per vedere l'angolo
            if(u.prev == -1){
                cosTheta = fabs(cosTheta); // se non c'è predecessore prendo valore assoluto per tenere in considerazione entrambe le direzioni
            }
            //cosTheta=(cosTheta-THR_ANG)/(1-THR_ANG);
            //float thr_ang = 0.5;

            //Se l'angolo non ha almeno coseno thr_ang lo scarto
            if (cosTheta < THR_ANG)
            {
                continue;
            }
            float velocity = e.direction * pow(cosTheta,EXP_COS_THETA) + 0.5*pow(nodi[id].baricentro.value, EXP_VALUE_BAR_SP);

            // Normalizza la distanza nel calcolo del costo
            float d_norm = d / maxDistance;

            //Calcolo del costo
            float cost = u.distance + d_norm / ((velocity * penalty) + 0.001); // (velocity + 0.001);


            // verifico se il nodo è presente nell'heap, se presente si controlla se il nuovo costo è inferiore alla distanza già registrata per quel nodo
            if (v.distance > cost)
            {
                if (heap.containsID(id))
                {
                       if (debug)
                            cout << "Nodo " << id << " trovato nell'heap, aggiornamento distanza da " << v.distance << " a " << cost << endl;
                        v.distance = cost;
                        v.prev = idU;
                        heap.decreaseKey(id, cost);

                }
                else
                {
                    if (debug)
                        cout << "Nodo " << id << " non trovato nell'heap, lo inserisco con distanza " << cost << endl;
                    v.distance = cost;
                    v.prev = idU;
                    heap.insert(id, cost);
                }

            }
        }
    // FINE espansione archi
}
// fine SP

if (1)
// QUI FARE IL MERGE PER VEDERE DA ENTRAMBI I LATI
for (int i = 1; i >= -1; i -= 2)
{
    Point3D direzione_from = Point3D(nodi[from].autovetMax[0], nodi[from].autovetMax[1], nodi[from].autovetMax[2]);
    direzione_from.x *= i;
    direzione_from.y *= i;
    direzione_from.z *= i;
    vector<int> stackPunti; // memorizzo i punti attraversati nel cammino
    float bestScore = std::numeric_limits<float>::max(); // punteggio iniziale massimo
    vector<int> migliorAttuale; // memorizzo miglior cammino
    ricercaRicorsiva(from, stackPunti, bestScore, migliorAttuale, direzione_from, 0);
    //inizializzo l'altezza del cilindro
    float altezzaCilindro=0.0f;
    if(migliorAttuale.size()>1)
    for (int i = 0; i < migliorAttuale.size() - 1; i++)
        {
            int start = migliorAttuale[i];
            int end = migliorAttuale[i + 1];
            //incremento l'altezza del cilindro
            altezzaCilindro += distance(nodi[start].baricentro, nodi[end].baricentro);
        }
    if(migliorAttuale.size()>1 && altezzaCilindro>diametro*FATTORE_LUNGHEZZA_MINIMA_SP)
    {
        
        //Inizializzo l'oggetto TuboJSON per il salvataggio dei dati
        TuboJSON tubo=TuboJSON("Tubo "+to_string(++contaTubi));
        tubo.epoca=epoca;
        tubo.diametro=diametro;
        tubo.bestScore=bestScore;
        tubo.distanza=altezzaCilindro;
        // Aggiungi il primo punto (start) al tubo
        int firstNode = migliorAttuale[0];
        tubo.aggiungiPunto(nodi[firstNode].baricentro);
        tubo.barPartenza=nodi[firstNode].baricentro;
        //Aggiungi il value del nodo di partenza
        float valuesPercorso=nodi[firstNode].baricentro.value;
        //Aggiungi prima distanza massimo-baricentro
        tubo.barmaxDist.push_back(nodi[firstNode].distance_c_max_raw);

        tubo.valuePartenza=valuesPercorso;
        tubo.barValues.push_back(tubo.valuePartenza);
        //Inizializza valori matriciali del percorso
        float sommaMatrixValues=matrix( (int)nodi[firstNode].baricentro.x,
                                        (int)nodi[firstNode].baricentro.y,
                                        (int)nodi[firstNode].baricentro.z);
        tubo.matrixValues.push_back(sommaMatrixValues);
        tubo.valueMaxLocali.push_back(matrix( (int)nodi[firstNode].maxLocale.x,
                                        (int)nodi[firstNode].maxLocale.y,
                                        (int)nodi[firstNode].maxLocale.z));
        //Aggiungi la direzione del tubo
        tubo.direction=Point3D(direzione_from);
        Point3D direzione_norm= Point3D(direzione_from);
        direzione_norm.normalize();
        //Aggiorno angoli con se stessi
        tubo.pcaAngles.push_back(1);
        tubo.pcaStartAngles.push_back(1);

        //ciclo sul percorso per plottarlo
        for (int i = 0; i < migliorAttuale.size() - 1; i++)
            {
                int start = migliorAttuale[i];
                int end = migliorAttuale[i + 1];

                //Aggiorno flag di esclusione sui due nodi per cui passa l'arco
                vertex[start].flagEscluso=true;
                vertex[end].flagEscluso=true;
                // Calcola la distanza cumulativa per il peso della linea
                float lineCost = vertex[from].distance;

                // Aggiungi la linea al vettore 'lines'
                lines.push_back(Line(nodi[start].baricentro, nodi[end].baricentro, vertex[migliorAttuale[i]].distEuclidea));

                //Salvataggio file
                //Aggiungi l'intensità
                tubo.aggiungiIntensita(vertex[migliorAttuale[i]].distEuclidea);
                //Aggiungi valore baricentro corrente
                tubo.barValues.push_back(nodi[end].baricentro.value);
                //Aggiungi distanza baricentro-massimo corrente
                tubo.barmaxDist.push_back(nodi[end].distance_c_max_raw);
                //Aggiorna somma per media
                valuesPercorso+=nodi[end].baricentro.value;
                //Aggiorna valori Matriciali
                tubo.matrixValues.push_back(matrix((int)nodi[end].baricentro.x,
                    (int)nodi[end].baricentro.y,
                    (int)nodi[end].baricentro.z));
                tubo.valueMaxLocali.push_back(matrix( (int)nodi[end].maxLocale.x,
                                        (int)nodi[end].maxLocale.y,
                                        (int)nodi[end].maxLocale.z));
                sommaMatrixValues+=matrix((int)nodi[end].baricentro.x,
                    (int)nodi[end].baricentro.y,
                    (int)nodi[end].baricentro.z);
                //Aggiungo angolo tra PCA dei due Baricentri correnti
                //Controllo le direzioni degli ultimi due baricentri
                Point3D pcaDirCurrent=Point3D(nodi[start].autovetMax[0],nodi[start].autovetMax[1],nodi[start].autovetMax[2]);
                Point3D pcaDirEnd=Point3D(nodi[end].autovetMax[0],nodi[end].autovetMax[1],nodi[end].autovetMax[2]);
              
                //Angolo tra pca di due nodi correnti
                tubo.pcaAngles.push_back(fabs(pcaDirCurrent*pcaDirEnd));
                //Angolo di PCA end con PCA iniziale
                tubo.pcaStartAngles.push_back(fabs(direzione_from*pcaDirEnd));
                // Aggiungi il punto 'end'
                tubo.aggiungiPunto(nodi[end].baricentro);
            }
    //Inserisco Media dei valori dei baricentri
    valuesPercorso/=migliorAttuale.size();
    tubo.mediaValueBaricentri=valuesPercorso;
    //Inserisco Media e mediana valori matriciali baricentri
    sommaMatrixValues/=migliorAttuale.size();
    tubo.mediaMatrixBar=sommaMatrixValues;
    tubo.medianMatrixBar= calculateMedian(tubo.matrixValues);

    tubiJson.push_back(tubo);
    std::cout << "[DEBUG] Aggiunto un tubo nel vettore tubiJson. Nome: "
          << tubo.nome << ", epoca=" << tubo.epoca
          << ", sizePoints=" << tubo.points.size() << std::endl;


    float raggio = diametroOOF / 2.0f;    

    //flag booleano, se true costruisce il cilindro ideale, altrimenti lo fa "a segmenti" su singoli archi dello shortestPath
    bool cilinderPCA=false;
    if(cilinderPCA)
    {
        for(int i = 0; i < vertex.size(); i++)
        {
            if(vertex[i].flagEscluso)
            {
                continue;  // Salta se già escluso
            }

            //baricentro corrente
            Point3D p = nodi[i].baricentro;
            // Calcola il vettore dal centro del cilindro al baricentro corrente
            Point3D vettoreDalCentro = p - nodi[from].baricentro;

            // Componente del punto lungo l'asse del cilindro
            float componenteAsse = vettoreDalCentro * direzione_from;

            // Se il punto è fuori dall'altezza del cilindro, continua
            if (componenteAsse < 0 || componenteAsse > altezzaCilindro)
            {
                continue;
            }

            // Proiezione del punto sull'asse del cilindro
            Point3D proiezione = Point3D(nodi[from].baricentro.x + direzione_from.x * componenteAsse,
            nodi[from].baricentro.y + direzione_from.y * componenteAsse,
            nodi[from].baricentro.z + direzione_from.z * componenteAsse);

            // Distanza dalla proiezione al punto, nel piano perpendicolare all'asse
            Point3D distanzaVettoriale = p - proiezione;
            float distanza = distanzaVettoriale.norm(distanzaVettoriale);  // Usa la funzione per calcolare la norma del vettore

            // Se la distanza è minore o uguale al raggio, escludi il vertice
            if (distanza <= raggio) {
                vertex[i].flagEscluso = true;
            }
        }
    }
    // else // Creo un cilindro per ciascun arco
    // {
    //     for (int i = 0; i < migliorAttuale.size() - 1; i++)
    //     {
    //         int start = migliorAttuale[i];
    //         int end = migliorAttuale[i + 1];

    //         // Calcola l'altezza del cilindro per l'arco corrente
    //         altezzaCilindro = distance(nodi[start].baricentro, nodi[end].baricentro);
    //         Point3D direzione_arco = nodi[end].baricentro - nodi[start].baricentro;
    //         direzione_arco.normalize();

    //         // Raggio del cilindro
    //         float raggio = diametro / 2.0f;

    //         // Escludi i nodi all'interno del cilindro per l'arco corrente
    //         for (int j = 0; j < vertex.size(); j++)
    //         {
    //             if (vertex[j].flagEscluso)
    //                 continue;

    //             Point3D p = nodi[j].baricentro;
    //             Point3D vettoreDalCentro = p - nodi[start].baricentro;

    //             // Componente del punto lungo la direzione dell'arco
    //             float componenteAsse = vettoreDalCentro * direzione_arco;

    //             // Controlla se il punto è all'interno dell'altezza del cilindro
    //             if (componenteAsse < 0 || componenteAsse > altezzaCilindro)
    //                 continue;

    //             // Proiezione del punto sull'asse dell'arco
    //             Point3D proiezione = Point3D(nodi[start].baricentro.x + direzione_arco.x * componenteAsse,
    //                                         nodi[start].baricentro.y + direzione_arco.y * componenteAsse,
    //                                         nodi[start].baricentro.z + direzione_arco.z * componenteAsse);

    //             // Calcola la dsalvatubiistanza dalla proiezione al punto
    //             Point3D distanzaVettoriale = p - proiezione;
    //             float distanza = distanzaVettoriale.norm(distanzaVettoriale);

    //             // Se la distanza è minore o uguale al raggio, escludi il nodo
    //             if (distanza <= raggio)
    //                 vertex[j].flagEscluso = true;
    //         }
    //     }
    // }
    else // Creo un cilindro per ciascun arco
    {
        for (int i = 0; i < migliorAttuale.size() - 1; i++)
        {
            int start = migliorAttuale[i];
            int end = migliorAttuale[i + 1];

            // Calcola l'altezza del cilindro per l'arco corrente
            altezzaCilindro = distance(nodi[start].baricentro, nodi[end].baricentro);
            Point3D direzione_arco = nodi[end].baricentro - nodi[start].baricentro;
            direzione_arco.normalize();

            // Raggio del cilindro
            float raggio = diametroOOF / 2.0f;

            // Inizializzazione dei cerchi di base e cima
            vector<Point3D> baseCircle;
            vector<Point3D> topCircle;
            float circlePrecision = CIRCLE_PRECISION;

            // Generazione del cerchio alla base
            for (int j = 0; j < circlePrecision; j++)
            {
                float pX = raggio * cos(j / circlePrecision * 2 * 3.1415);
                float pY = raggio * sin(j / circlePrecision * 2 * 3.1415);
                
                float dx = pX * direzione_arco.y;  // Perpendicolare alla direzione dell'arco
                float dy = -pX * direzione_arco.x;
                float dz = pY;  // Altezza verticale rispetto al raggio
                
                float x = nodi[start].baricentro.x + dx;
                float y = nodi[start].baricentro.y + dy;
                float z = nodi[start].baricentro.z + dz;
                baseCircle.push_back(Point3D(x, y, z));
            }

            // Generazione del cerchio alla cima traslando ogni punto del cerchio base
            for (int j = 0; j < circlePrecision; j++)
            {
                float dx = direzione_arco.x * altezzaCilindro;
                float dy = direzione_arco.y * altezzaCilindro;
                float dz = direzione_arco.z * altezzaCilindro;
                
                float x = baseCircle[j].x + dx;
                float y = baseCircle[j].y + dy;
                float z = baseCircle[j].z + dz;
                topCircle.push_back(Point3D(x, y, z));
            }

            // Collega i punti per disegnare il cilindro
            float color= bestScore + 0*nodi[from].baricentro.value;
            for (int j = 0; j < circlePrecision; j++)
            {
                int nextIndex = (j + 1) % (int)circlePrecision;
                sectionCircle.push_back(Line(baseCircle[j], baseCircle[nextIndex], color));   // Linea alla base
                sectionCircle.push_back(Line(topCircle[j], topCircle[nextIndex], color));     // Linea alla cima
                sectionCircle.push_back(Line(baseCircle[j], topCircle[j], color));            // Linea laterale
            }

            // Escludi i nodi all'interno del cilindro per l'arco corrente
            for (int j = 0; j < vertex.size(); j++)
            {
                if (vertex[j].flagEscluso)
                    continue;

                Point3D p = nodi[j].baricentro;
                Point3D vettoreDalCentro = p - nodi[start].baricentro;

                // Componente del punto lungo la direzione dell'arco
                float componenteAsse = vettoreDalCentro * direzione_arco;

                // Controlla se il punto è all'interno dell'altezza del cilindro
                if (componenteAsse < 0 || componenteAsse > altezzaCilindro)
                    continue;

                // Proiezione del punto sull'asse dell'arco
                Point3D proiezione = Point3D(
                    nodi[start].baricentro.x + direzione_arco.x * componenteAsse,
                    nodi[start].baricentro.y + direzione_arco.y * componenteAsse,
                    nodi[start].baricentro.z + direzione_arco.z * componenteAsse);

                // Calcola la distanza dalla proiezione al punto
                Point3D distanzaVettoriale = p - proiezione;
                float distanza = norm(distanzaVettoriale);

                // Se la distanza è minore o uguale al raggio, escludi il nodo
                if (distanza <= raggio)
                    vertex[j].flagEscluso = true;
            }
        }
    }


    /**Rendo disponibili ad altri percorsi il primo e l'ultimo baricentro, in questo modo i percorsi potranno
     * collegarsi nei loro estremi
     */
    vertex[from].flagEscluso=false;
    vertex[migliorAttuale[migliorAttuale.size() - 1]].flagEscluso=false;
}

}
//lines.clear();
if (0)
    for (int i = 0; i < nodi.size(); i++)
    {
        if (vertex[i].prev != -1)
        {
            Point3D start = Point3D(nodi[i].baricentro);
            int j = vertex[i].prev;
            Point3D end = Point3D(nodi[j].baricentro);
            lines.push_back(Line(start, end, vertex[i].distance));
            cout<<"i: " << i << "j: " << j << "distance: " << vertex[i].distance << endl;
        }
    }
 }

// Max
void Graph::shortestPathMax(int from, MinHeap &heap)
{
     // cout << "from " << from << endl;

     float val = 0;
     // cout << "calcolo val: " << endl;
     if (nodi[from].maxLocale.x < matrix.getX() && nodi[from].maxLocale.y < matrix.getY() && nodi[from].maxLocale.z < matrix.getZ())
     {
         val = matrix((int)nodi[from].maxLocale.x, (int)nodi[from].maxLocale.y, (int)nodi[from].maxLocale.z);
     }
     else
     {
         cout << "X: " << nodi[from].maxLocale.x << " XMatrix: " << matrix.getX() << endl;
         cout << "Errore: indice della matrice fuori dai limiti" << endl;
         val = 0.0;
     }
     // cout << "Val: " << val << endl;

     vector<float> migliorAttualeV;

     // inizializzo i nodi
     for (int i = 0; i < nodi.size(); i++)
     {
         Vertex &ver = vertex[i]; // Vertex è un vettore che contiene tutti i nodi del grafo
         ver.distance = heap.INF; // inizializzato con distanza infinita
         ver.visited = false;
         ver.prev = -1; // nessun predecessore

         if (ver.flagEscluso)
             continue;

         // calcolo distanza euclidea tra nodo corrente e nodo di partenza
         float dx = nodi[i].maxLocale.x - nodi[from].maxLocale.x;
         float dy = nodi[i].maxLocale.y - nodi[from].maxLocale.y;
         float dz = (nodi[i].maxLocale.z - nodi[from].maxLocale.z) * SCALA;
         ver.distEuclidea = sqrt(dx * dx + dy * dy + dz * dz);
     }
     vertex[from].distance = 0; // distanza nodo da se stesso

     heap.reset();
     heap.insert(from, 0);

     while (!heap.isEmpty())
     {
         int debug = 0;
         // Estraggo il nodo con valore minimo
         long idU = heap.extractMinID();

         Vertex &u = vertex[idU];
         if (u.visited && !u.flagEscluso)
         {
             continue;
         }
         u.visited = true;

         // direzione rappresentata dall'autovettore massimo delle PCA nelle tre coordinate xyz se non è presente un nodo precedente
         Point3D direzione = Point3D(nodi[from].autovetMax[0], nodi[from].autovetMax[1], nodi[from].autovetMax[2]);

         // se c'è il predecessore calcolo facendo la differenza tra coordinate dell'attuale e coordinate del predecessore, la differenza rappresenta
         // il vettore che punta dal predecessore al corrente
         if (u.prev != -1)
         {
             direzione = Point3D(nodi[idU].maxLocale - nodi[u.prev].maxLocale);
             direzione.normalize();
         }

         // itero sugli archi del nodo corrente, per ogni arco analizzo il nodo di destinazione
         for (const Edge &e : edges[idU])
         {

             int id = e.to;          // nodo destinazione
             Vertex &v = vertex[id]; // nodo destinazione nel grafo

             float val_id = matrix((int)nodi[id].maxLocale.x, (int)nodi[id].maxLocale.y, (int)nodi[id].maxLocale.z);

             float penalty = pow(1 - fabs(val - val_id), EXP_PENALTY);

             if (debug)
                 cout << "entrato " << "idU: " << idU << "id: " << id << "direction: " << e.direction << endl;

             if (v.flagEscluso)
             {
                 continue;
             }
             Point3D b = Point3D(nodi[id].maxLocale - nodi[idU].maxLocale); // vettore che punta dal nodo corrente idU al nodo destinazione id

             float d = norm(b); // distanza tra idU e id
             b.normalize();     // normalizzo b in un vettore unitario (lunghezza 1) per tenere conto solo della direzione

             float cosTheta = direzione * b; // prodotto scalare tra vettore direzione e arco per vedere l'angolo
             if (u.prev == -1)
             {
                 cosTheta = fabs(cosTheta); // se non c'è predecessore prendo valore assoluto per tenere in considerazione entrambe le direzioni
             }
             // cosTheta=(cosTheta-THR_ANG)/(1-THR_ANG);
             // float thr_ang = 0.5;

             // Se l'angolo non ha almeno coseno thr_ang lo scarto
             if (cosTheta < THR_ANG)
             {
                 continue;
             }
             float velocity = e.direction * pow(cosTheta, EXP_COS_THETA) + 0.5 * pow(nodi[id].maxLocale.value, EXP_VALUE_BAR_SP);

             // Normalizza la distanza nel calcolo del costo
             float d_norm = d / maxDistance;

             // Calcolo del costo
             float cost = u.distance + d_norm / ((velocity * penalty) + 0.001); // (velocity + 0.001);

             // verifico se il nodo è presente nell'heap, se presente si controlla se il nuovo costo è inferiore alla distanza già registrata per quel nodo
             if (v.distance > cost)
             {
                 if (heap.containsID(id))
                 {
                     if (debug)
                         cout << "Nodo " << id << " trovato nell'heap, aggiornamento distanza da " << v.distance << " a " << cost << endl;
                     v.distance = cost;
                     v.prev = idU;
                     heap.decreaseKey(id, cost);
                 }
                 else
                 {
                     if (debug)
                         cout << "Nodo " << id << " non trovato nell'heap, lo inserisco con distanza " << cost << endl;
                     v.distance = cost;
                     v.prev = idU;
                     heap.insert(id, cost);
                 }
             }
         }
         // FINE espansione archi
     }
     // fine SP

     if (1)
         // QUI FARE IL MERGE PER VEDERE DA ENTRAMBI I LATI
         for (int i = 1; i >= -1; i -= 2)
         {
             Point3D direzione_from = Point3D(nodi[from].autovetMax[0], nodi[from].autovetMax[1], nodi[from].autovetMax[2]);
             direzione_from.x *= i;
             direzione_from.y *= i;
             direzione_from.z *= i;
             vector<int> stackPunti;                              // memorizzo i punti attraversati nel cammino
             float bestScore = std::numeric_limits<float>::max(); // punteggio iniziale massimo
             vector<int> migliorAttuale;                          // memorizzo miglior cammino
             ricercaRicorsiva(from, stackPunti, bestScore, migliorAttuale, direzione_from, 0);
             // inizializzo l'altezza del cilindro
             float altezzaCilindro = 0.0f;
             if (migliorAttuale.size() > 1)
                 for (int i = 0; i < migliorAttuale.size() - 1; i++)
                 {
                     int start = migliorAttuale[i];
                     int end = migliorAttuale[i + 1];
                     // incremento l'altezza del cilindro
                     altezzaCilindro += distance(nodi[start].maxLocale, nodi[end].maxLocale);
                 }
             if (migliorAttuale.size() > 1 && altezzaCilindro > diametro * FATTORE_LUNGHEZZA_MINIMA_SP)
             {

                 // Inizializzo l'oggetto TuboJSON per il salvataggio dei dati
                 TuboJSON tubo = TuboJSON("Tubo " + to_string(++contaTubi));
                 tubo.epoca = epoca;
                 tubo.diametro = diametro;
                 tubo.bestScore = bestScore;
                 tubo.distanza = altezzaCilindro;
                 // Aggiungi il primo punto (start) al tubo
                 int firstNode = migliorAttuale[0];
                 tubo.aggiungiPunto(nodi[firstNode].maxLocale);
                 tubo.barPartenza = nodi[firstNode].maxLocale;
                 // Aggiungi il value del nodo di partenza
                 float valuesPercorso = nodi[firstNode].maxLocale.value;
                 tubo.valuePartenza = valuesPercorso;
                 tubo.barValues.push_back(tubo.valuePartenza);
                 // Inizializza valori matriciali del percorso
                 float sommaMatrixValues = matrix((int)nodi[firstNode].maxLocale.x,
                                                  (int)nodi[firstNode].maxLocale.y,
                                                  (int)nodi[firstNode].maxLocale.z);
                 tubo.matrixValues.push_back(sommaMatrixValues);
                 // Aggiungi la direzione del tubo
                 tubo.direction = Point3D(direzione_from);
                 Point3D direzione_norm = Point3D(direzione_from);
                 direzione_norm.normalize();
                 // Aggiorno angoli con se stessi
                 tubo.pcaAngles.push_back(1);
                 tubo.pcaStartAngles.push_back(1);

                 // ciclo sul percorso per plottarlo
                 for (int i = 0; i < migliorAttuale.size() - 1; i++)
                 {
                     int start = migliorAttuale[i];
                     int end = migliorAttuale[i + 1];

                     // Aggiorno flag di esclusione sui due nodi per cui passa l'arco
                     vertex[start].flagEscluso = true;
                     vertex[end].flagEscluso = true;
                     // Calcola la distanza cumulativa per il peso della linea
                     float lineCost = vertex[from].distance;

                     // Aggiungi la linea al vettore 'lines'
                     lines.push_back(Line(nodi[start].maxLocale, nodi[end].maxLocale, vertex[migliorAttuale[i]].distEuclidea));

                     // Salvataggio file
                     // Aggiungi l'intensità
                     tubo.aggiungiIntensita(vertex[migliorAttuale[i]].distEuclidea);
                     // Aggiungi valore baricentro corrente
                     tubo.barValues.push_back(nodi[end].maxLocale.value);
                     // Aggiorna somma per media
                     valuesPercorso += nodi[end].maxLocale.value;
                     // Aggiorna valori Matriciali
                     tubo.matrixValues.push_back(matrix((int)nodi[end].maxLocale.x,
                                                        (int)nodi[end].maxLocale.y,
                                                        (int)nodi[end].maxLocale.z));
                     sommaMatrixValues += matrix((int)nodi[end].maxLocale.x,
                                                 (int)nodi[end].maxLocale.y,
                                                 (int)nodi[end].maxLocale.z);
                     // Aggiungo angolo tra PCA dei due Baricentri correnti
                     // Controllo le direzioni degli ultimi due baricentri
                     Point3D pcaDirCurrent = Point3D(nodi[start].autovetMax[0], nodi[start].autovetMax[1], nodi[start].autovetMax[2]);
                     Point3D pcaDirEnd = Point3D(nodi[end].autovetMax[0], nodi[end].autovetMax[1], nodi[end].autovetMax[2]);

                     // Angolo tra pca di due nodi correnti
                     tubo.pcaAngles.push_back(fabs(pcaDirCurrent * pcaDirEnd));
                     // Angolo di PCA end con PCA iniziale
                     tubo.pcaStartAngles.push_back(fabs(direzione_from * pcaDirEnd));
                     // Aggiungi il punto 'end'
                     tubo.aggiungiPunto(nodi[end].maxLocale);
                 }
                 // Inserisco Media dei valori dei baricentri
                 valuesPercorso /= migliorAttuale.size();
                 tubo.mediaValueBaricentri = valuesPercorso;
                 // Inserisco Media e mediana valori matriciali baricentri
                 sommaMatrixValues /= migliorAttuale.size();
                 tubo.mediaMatrixBar = sommaMatrixValues;
                 tubo.medianMatrixBar = calculateMedian(tubo.matrixValues);

                 tubiJson.push_back(tubo);

                 float raggio = diametroOOF / 2.0f;

                 // flag booleano, se true costruisce il cilindro ideale, altrimenti lo fa "a segmenti" su singoli archi dello shortestPath
                 bool cilinderPCA = false;
                 if (cilinderPCA)
                 {
                     for (int i = 0; i < vertex.size(); i++)
                     {
                         if (vertex[i].flagEscluso)
                         {
                             continue; // Salta se già escluso
                         }

                         // baricentro corrente
                         Point3D p = nodi[i].maxLocale;
                         // Calcola il vettore dal centro del cilindro al baricentro corrente
                         Point3D vettoreDalCentro = p - nodi[from].maxLocale;

                         // Componente del punto lungo l'asse del cilindro
                         float componenteAsse = vettoreDalCentro * direzione_from;

                         // Se il punto è fuori dall'altezza del cilindro, continua
                         if (componenteAsse < 0 || componenteAsse > altezzaCilindro)
                         {
                             continue;
                         }

                         // Proiezione del punto sull'asse del cilindro
                         Point3D proiezione = Point3D(nodi[from].maxLocale.x + direzione_from.x * componenteAsse,
                                                      nodi[from].maxLocale.y + direzione_from.y * componenteAsse,
                                                      nodi[from].maxLocale.z + direzione_from.z * componenteAsse);

                         // Distanza dalla proiezione al punto, nel piano perpendicolare all'asse
                         Point3D distanzaVettoriale = p - proiezione;
                         float distanza = distanzaVettoriale.norm(distanzaVettoriale); // Usa la funzione per calcolare la norma del vettore

                         // Se la distanza è minore o uguale al raggio, escludi il vertice
                         if (distanza <= raggio)
                         {
                             vertex[i].flagEscluso = true;
                         }
                     }
                 }
                 // else // Creo un cilindro per ciascun arco
                 // {
                 //     for (int i = 0; i < migliorAttuale.size() - 1; i++)
                 //     {
                 //         int start = migliorAttuale[i];
                 //         int end = migliorAttuale[i + 1];

                 //         // Calcola l'altezza del cilindro per l'arco corrente
                 //         altezzaCilindro = distance(nodi[start].baricentro, nodi[end].baricentro);
                 //         Point3D direzione_arco = nodi[end].baricentro - nodi[start].baricentro;
                 //         direzione_arco.normalize();

                 //         // Raggio del cilindro
                 //         float raggio = diametro / 2.0f;

                 //         // Escludi i nodi all'interno del cilindro per l'arco corrente
                 //         for (int j = 0; j < vertex.size(); j++)
                 //         {
                 //             if (vertex[j].flagEscluso)
                 //                 continue;

                 //             Point3D p = nodi[j].baricentro;
                 //             Point3D vettoreDalCentro = p - nodi[start].baricentro;

                 //             // Componente del punto lungo la direzione dell'arco
                 //             float componenteAsse = vettoreDalCentro * direzione_arco;

                 //             // Controlla se il punto è all'interno dell'altezza del cilindro
                 //             if (componenteAsse < 0 || componenteAsse > altezzaCilindro)
                 //                 continue;

                 //             // Proiezione del punto sull'asse dell'arco
                 //             Point3D proiezione = Point3D(nodi[start].baricentro.x + direzione_arco.x * componenteAsse,
                 //                                         nodi[start].baricentro.y + direzione_arco.y * componenteAsse,
                 //                                         nodi[start].baricentro.z + direzione_arco.z * componenteAsse);

                 //             // Calcola la dsalvatubiistanza dalla proiezione al punto
                 //             Point3D distanzaVettoriale = p - proiezione;
                 //             float distanza = distanzaVettoriale.norm(distanzaVettoriale);

                 //             // Se la distanza è minore o uguale al raggio, escludi il nodo
                 //             if (distanza <= raggio)
                 //                 vertex[j].flagEscluso = true;
                 //         }
                 //     }
                 // }
                 else // Creo un cilindro per ciascun arco
                 {
                     for (int i = 0; i < migliorAttuale.size() - 1; i++)
                     {
                         int start = migliorAttuale[i];
                         int end = migliorAttuale[i + 1];

                         // Calcola l'altezza del cilindro per l'arco corrente
                         altezzaCilindro = distance(nodi[start].maxLocale, nodi[end].maxLocale);
                         Point3D direzione_arco = nodi[end].maxLocale - nodi[start].maxLocale;
                         direzione_arco.normalize();

                         // Raggio del cilindro
                         float raggio = diametroOOF / 2.0f;

                         // Inizializzazione dei cerchi di base e cima
                         vector<Point3D> baseCircle;
                         vector<Point3D> topCircle;
                         float circlePrecision = CIRCLE_PRECISION;

                         // Generazione del cerchio alla base
                         for (int j = 0; j < circlePrecision; j++)
                         {
                             float pX = raggio * cos(j / circlePrecision * 2 * 3.1415);
                             float pY = raggio * sin(j / circlePrecision * 2 * 3.1415);

                             float dx = pX * direzione_arco.y; // Perpendicolare alla direzione dell'arco
                             float dy = -pX * direzione_arco.x;
                             float dz = pY; // Altezza verticale rispetto al raggio

                             float x = nodi[start].maxLocale.x + dx;
                             float y = nodi[start].maxLocale.y + dy;
                             float z = nodi[start].maxLocale.z + dz;
                             baseCircle.push_back(Point3D(x, y, z));
                         }

                         // Generazione del cerchio alla cima traslando ogni punto del cerchio base
                         for (int j = 0; j < circlePrecision; j++)
                         {
                             float dx = direzione_arco.x * altezzaCilindro;
                             float dy = direzione_arco.y * altezzaCilindro;
                             float dz = direzione_arco.z * altezzaCilindro;

                             float x = baseCircle[j].x + dx;
                             float y = baseCircle[j].y + dy;
                             float z = baseCircle[j].z + dz;
                             topCircle.push_back(Point3D(x, y, z));
                         }

                         // Collega i punti per disegnare il cilindro
                         float color = bestScore + 0 * nodi[from].maxLocale.value;
                         for (int j = 0; j < circlePrecision; j++)
                         {
                             int nextIndex = (j + 1) % (int)circlePrecision;
                             sectionCircle.push_back(Line(baseCircle[j], baseCircle[nextIndex], color)); // Linea alla base
                             sectionCircle.push_back(Line(topCircle[j], topCircle[nextIndex], color));   // Linea alla cima
                             sectionCircle.push_back(Line(baseCircle[j], topCircle[j], color));          // Linea laterale
                         }

                         // Escludi i nodi all'interno del cilindro per l'arco corrente
                         for (int j = 0; j < vertex.size(); j++)
                         {
                             if (vertex[j].flagEscluso)
                                 continue;

                             Point3D p = nodi[j].maxLocale;
                             Point3D vettoreDalCentro = p - nodi[start].maxLocale;

                             // Componente del punto lungo la direzione dell'arco
                             float componenteAsse = vettoreDalCentro * direzione_arco;

                             // Controlla se il punto è all'interno dell'altezza del cilindro
                             if (componenteAsse < 0 || componenteAsse > altezzaCilindro)
                                 continue;

                             // Proiezione del punto sull'asse dell'arco
                             Point3D proiezione = Point3D(
                                 nodi[start].maxLocale.x + direzione_arco.x * componenteAsse,
                                 nodi[start].maxLocale.y + direzione_arco.y * componenteAsse,
                                 nodi[start].maxLocale.z + direzione_arco.z * componenteAsse);

                             // Calcola la distanza dalla proiezione al punto
                             Point3D distanzaVettoriale = p - proiezione;
                             float distanza = norm(distanzaVettoriale);

                             // Se la distanza è minore o uguale al raggio, escludi il nodo
                             if (distanza <= raggio)
                                 vertex[j].flagEscluso = true;
                         }
                     }
                 }

                 /**Rendo disponibili ad altri percorsi il primo e l'ultimo baricentro, in questo modo i percorsi potranno
                  * collegarsi nei loro estremi
                  */
                 vertex[from].flagEscluso = false;
                 vertex[migliorAttuale[migliorAttuale.size() - 1]].flagEscluso = false;
             }
         }
     // lines.clear();
     if (0)
         for (int i = 0; i < nodi.size(); i++)
         {
             if (vertex[i].prev != -1)
             {
                 Point3D start = Point3D(nodi[i].maxLocale);
                 int j = vertex[i].prev;
                 Point3D end = Point3D(nodi[j].maxLocale);
                 lines.push_back(Line(start, end, vertex[i].distance));
                 cout << "i: " << i << "j: " << j << "distance: " << vertex[i].distance << endl;
             }
         }
 }




/**
 * @brief Valuta se due numeri float sono approssimativamente uguali entro una certa tolleranza.
 *
 * Questa funzione confronta due numeri float, `a` e `b`, determinando se la loro differenza assoluta è
 * inferiore o uguale a una soglia di tolleranza `epsilon`. È utile per gestire imprecisioni nei calcoli
 * con numeri float, dove le discrepanze minori possono derivare da errori di arrotondamento.
 *
 * @param a Primo numero float da confrontare.
 * @param b Secondo numero float da confrontare.
 * @param epsilon Tolleranza per il confronto; valore predefinito è 1e-5.
 * @return true Se i numeri sono considerati uguali entro la tolleranza, false altrimenti.
 */
bool areApproximatelyEqual(float a, float b, float epsilon = 1e-5f)
{
    return fabs(a - b) <= epsilon;
}

/**
 * @brief Verifica se due punti 3D sono approssimativamente uguali entro una certa tolleranza per ogni coordinata.
 *
 * Questa funzione controlla se le coordinate x, y e z di due punti 3D, `p1` e `p2`, sono vicine entro
 * una soglia specificata `epsilon`. Utilizza la funzione `areApproximatelyEqual` per confrontare
 * le singole coordinate. È utile per determinare se due punti sono quasi identici, tenendo conto delle
 * piccole imprecisioni che possono derivare dai calcoli in virgola mobile.
 *
 * @param p1 Primo punto 3D per il confronto.
 * @param p2 Secondo punto 3D per il confronto.
 * @param epsilon Tolleranza per il confronto di ciascuna coordinata; valore predefinito è 1e-4.
 * @return true Se tutte le coordinate corrispondenti dei due punti sono considerate uguali entro la tolleranza, false altrimenti.
 */
bool arePointsApproximatelyEqual(const Point3D &p1, const Point3D &p2, float epsilon = 1e-4f)
{
    return areApproximatelyEqual(p1.x, p2.x, epsilon) &&
           areApproximatelyEqual(p1.y, p2.y, epsilon) &&
           areApproximatelyEqual(p1.z, p2.z, epsilon);
}

/**
 * @brief Trova l'indice di un nodo nel grafo basato sulle coordinate del baricentro, confrontando le coordinate con un punto specificato.
 *
 * Questa funzione scorre tutti i nodi nel grafo e utilizza `arePointsApproximatelyEqual` per determinare se il baricentro di un nodo
 * è approssimativamente uguale al punto fornito in input. Questo è utile per identificare nodi specifici in una struttura dati
 * quando solo le coordinate del punto sono note, considerando possibili piccole variazioni nelle coordinate dovute a imprecisioni di calcolo.
 *
 * @param point Il punto 3D il cui indice si vuole trovare nel vettore di nodi.
 * @return L'indice del nodo se trovato, altrimenti -1.
 */
int Graph::findNodeIndex(const Point3D &point)
{
    for (int i = 0; i < nodi.size(); ++i)
    {
        if (arePointsApproximatelyEqual(nodi[i].baricentro, point))
        {
            return i; // Nodo trovato
        }
    }
    return -1; // Nodo non trovato
}

/**
 * @brief Esegue una ricerca ricorsiva per trovare il miglior percorso nel grafo partendo da un nodo specifico.
 *
 * Questa funzione esplora ricorsivamente i cammini possibili nel grafo, seguendo gli archi connessi e mantenendo
 * traccia del percorso attuale attraverso `stackPunti`. Ogni nodo viene visitato una sola volta per percorso,
 * evitando cicli. La funzione valuta il percorso basandosi sulla distanza cumulativa e sulla conformità del percorso
 * rispetto a un certo 'diametro' e a una 'direzione' specificata, agendo come una sorta di filtro per percorsi non validi.
 *
 * @param current Nodo corrente da esplorare.
 * @param stackPunti Vettore che tiene traccia dei nodi visitati nel percorso attuale.
 * @param bestScore Riferimento al punteggio migliore trovato finora; utilizzato per confrontare i percorsi.
 * @param migliorAttuale Vettore che memorizza il miglior percorso trovato durante le chiamate ricorsive.
 * @param dir Direzione di movimento basata sul vettore PCA; aiuta a mantenere la direzionalità del percorso.
 * @param lev Livello di profondità della ricorsione, utilizzato per gestire la condizione di base e avanzamento della ricerca.
 *
 * La funzione si basa su una serie di calcoli geometrici e metriche di distanza per determinare la validità e l'efficacia
 * di ciascun percorso esplorato. Aggiorna dinamicamente il miglior percorso e il suo punteggio corrispondente in base
 * alla distanza euclidea e alla conformità del percorso con i parametri predefiniti.
 */
bool Graph::ricercaRicorsiva(int current, vector<int> &stackPunti, float &bestScore, vector<int> &migliorAttuale, Point3D dir, int lev)
{
    bool valido = false;
    //cout<<"Current "<<current<<endl;
    if (current != -1 && (lev==0 || !vertex[current].flagEscluso))
    {
        valido=true;
        //cout<<"Ok..."<<endl;
        Vertex &c = vertex[current];
        stackPunti.push_back(current);
        Point3D a(nodi[current].baricentro.x, nodi[current].baricentro.y, nodi[current].baricentro.z, c.distance);
        float fattoreLunghezzaMinima = FATTORE_LUNGHEZZA_MINIMA_SP;

        float dist_cumulativa = 0; // distanza cumulativa tra i punti

        // Se 'stackPunti' ha meno di 2 elementi, salta il ciclo
        if (stackPunti.size() >= 2)
        {
            //for (int i = 2; i < stackPunti.size(); i++)
            int i=stackPunti.size()-1;
            {
                //cout<<"i: "<<i<<endl;
                // distanza totale tra il primo punto fino al punto i
                for (int k = 1; k <= i; k++)
                {
                    Point3D diff = nodi[stackPunti[k]].baricentro - nodi[stackPunti[k-1]].baricentro;
                    dist_cumulativa += norm(diff);
                }
                //cout << "dist cumulativa: " << dist_cumulativa << endl;
                if (dist_cumulativa < diametro * fattoreLunghezzaMinima)
                {
                    //cout << "percorso troppo breve" << endl;
                    //continue;
                    valido=true;
                }
                else
                {
                    
                    // Point3D v = linearRegression(stackPunti, 0, i + 1);
                    // Point3D m(0, 0, 0);
                    // for (int j = 0; j <= i; j++)
                    // {
                    //     m.x += nodi[stackPunti[j]].baricentro.x;
                    //     m.y += nodi[stackPunti[j]].baricentro.y;
                    //     m.z += nodi[stackPunti[j]].baricentro.z;
                    // }
                    // m.x /= (i + 1);
                    // m.y /= (i + 1);
                    // m.z /= (i + 1);

                    // v.normalize();
                    // //double dist = a.pointToLineDistance(a, m, v);

                    // //Nuovo, seguo pca
                    int from=stackPunti[0];
                    Point3D direzionePCA = Point3D(nodi[from].autovetMax[0],nodi[from].autovetMax[1],nodi[from].autovetMax[2]);
                    // //scarto regressione e uso direzione pca, potrei fare angolo tra i due
                    // v=direzionePCA;
                    // double dist = a.pointToLineDistance(a, nodi[from].baricentro, v);
                    // cout<<"Dist: "<<dist<<endl;
                    // if (dist > (diametro* fattoreLunghezzaMinima) / 2)
                    // {
                    //     valido = false;
                    //     cout<<"Fuori diametro "<<endl;
                    //     //break;
                    // }
                    // Assume che `stackPunti` contenga gli indici dei punti nel percorso
                    //Verifica coerenza tra PCA e direzione
                    int prev= stackPunti[stackPunti.size()-2];
                    //differenza 
                    Point3D direzione = Point3D(nodi[current].baricentro-nodi[prev].baricentro);
                    direzione.normalize();
                    //Analisi coerenza tra movimento e direzione PCA
                    float prodScal= fabs(direzionePCA * direzione);
                    if(prodScal<PCA_ANG)
                    {
                        valido=false;
                    }
                    
                    //Controllo le direzioni degli ultimi due baricentri
                    Point3D pcaDirCurrent=Point3D(nodi[current].autovetMax[0],nodi[current].autovetMax[1],nodi[current].autovetMax[2]);
                    Point3D pcaDirPrev=Point3D(nodi[prev].autovetMax[0],nodi[prev].autovetMax[1],nodi[prev].autovetMax[2]);
                    float lastTwoAnglePCA = fabs(pcaDirCurrent*pcaDirPrev);
                    //angolo tra PCA corrente e arco
                    float cosAlpha=fabs(pcaDirCurrent*direzione);
                    if(cosAlpha<PCA_ANG)
                        valido=false;
                    //angolo tra arco corrente e primo arco
                    Point3D dirPrimoArco=nodi[current].baricentro-nodi[current+1].baricentro;
                    float cosBeta=fabs(direzione*dirPrimoArco);
                    //Se la somma del delta tra la soglia e last
                    float deltaAngoli=PCA_ANG - lastTwoAnglePCA + (PCA_ANG - cosBeta);
                    if (deltaAngoli > PCA_ANG/2)
                        valido=false;
                    if (lastTwoAnglePCA >= PCA_ANG && cosBeta >= PCA_ANG){
                        
                    }
                    else
                    {
                        // Calcola il baricentro iniziale (centro della base del cilindro)
                        Point3D centroBase = nodi[from].baricentro;
                        // Direzione del cilindro basata sull'autovettore maggiore
                        Point3D direzioneCilindro = Point3D(dir);
                        //direzioneCilindro.normalize();

                        // Raggio del cilindro
                        float raggio = diametro / 2.0f;

                        // Calcola l'altezza del cilindro (distanza cumulativa tra i punti nel percorso
                        // for (int i = 0; i < stackPunti.size() - 1; i++) {
                        //     int start = stackPunti[i];
                        //     int end = stackPunti[i + 1];
                        //     altezzaCilindro += distance(nodi[start].baricentro, nodi[end].baricentro);
                        // }

                        // Verifica ogni punto nel percorso rispetto al cilindro
                        bool tuttiPuntiValidi = true;

                        for (int i = 0; i < stackPunti.size(); i++) 
                        {
                            Point3D p = nodi[stackPunti[i]].baricentro;

                            // Vettore dal centro della base del cilindro al punto `p`
                            Point3D vettoreDalCentro = p - centroBase;

                            // Componente del vettore nella direzione del cilindro
                            float componenteLungoAsse = vettoreDalCentro * direzioneCilindro;

                            // Calcola la proiezione del punto `p` sull'asse del cilindro
                            Point3D proiezione = centroBase + (direzioneCilindro * componenteLungoAsse);

                            // Calcola la distanza tra il punto `p` e la proiezione sull'asse del cilindro
                            Point3D distanzaVettoriale = p - proiezione;
                            float distanzaDalCentro = norm(distanzaVettoriale);

                            // Verifica se il punto si trova entro il raggio del cilindro
                            if (distanzaDalCentro > raggio) {
                                tuttiPuntiValidi = false;
                                //cout << "Punto fuori dal raggio del cilindro: " << i << endl;
                            }
                        }

                        // Output del risultato
                        if (tuttiPuntiValidi) {
                            //cout << "Tutti i punti sono contenuti all'interno del cilindro." << endl;
                            //valido=true;
                        } else {
                            //cout << "Alcuni punti sono fuori dal cilindro." << endl;
                            valido=false;
                        }
                    }

                    
                }
            }
        }
        // AGGIUNGERE TEST DEI VALORI DEI NODI
        float val_id = matrix((int)nodi[current].baricentro.x, (int)nodi[current].baricentro.y, (int)nodi[current].baricentro.z);
        int id=stackPunti[0];
        float val_id0 = matrix((int)nodi[id].baricentro.x, (int)nodi[id].baricentro.y, (int)nodi[id].baricentro.z);
        if(val_id<val_id0*THRESHOLD_INTENSITA){
            valido=false;
        }

        if (valido)
        {
            
            bool nonvalutare=false;
            for (const Edge &e : edges[current])
            {
                int id = e.to;
                Vertex &v = vertex[id];

                if (v.prev == current)
                {
                    bool prosegui = true;
                    if (lev == 0)
                    {
                        Point3D b(nodi[id].baricentro.x, nodi[id].baricentro.y, nodi[id].baricentro.z, v.distance);
                        if ((b-a) * dir < 0)
                        {
                            //cout<<"Prosegui False!!"<<endl;
                            prosegui = false;
                        }
                    }
                    if (prosegui)
                    {
                        bool temp=ricercaRicorsiva(id, stackPunti, bestScore, migliorAttuale, dir, lev+1);
                        nonvalutare= nonvalutare || temp;
                    }
                }
            }
            if (1)
            {
                    // parametro da guardare
                //calcolo della posizione a cui tagliare in caso di arco molto debole
                int posizioneTaglio=stackPunti.size()-1;
                float accumuloScoreArchi=0.0f;
                float ctscoreArchi=0;
                for (int i = 0; i < stackPunti.size() - 1; i++) {
                        int start = stackPunti[i];
                        int end = stackPunti[i + 1];
                        for (const Edge &e : edges[start])
                        {
                            int id = e.to;
                            Vertex &v = vertex[id];
                            if(id==end){
                                Point3D diff = nodi[start].baricentro - nodi[end].baricentro;
                                accumuloScoreArchi+= (e.direction)*(norm(diff));
                                ctscoreArchi+= norm(diff);
                            }
                        }

                    }
                if(ctscoreArchi!=0)
                    accumuloScoreArchi /= ctscoreArchi;

                float score = 3-(dist_cumulativa / (diametro * fattoreLunghezzaMinima));
                if (dist_cumulativa > diametro * fattoreLunghezzaMinima)
                    //score = pow(c.distance, EXP_DISTANCE_NUMERATOR) / pow(dist_cumulativa, EXP_DIST_CUMULATIVA);
                    //score= -dist_cumulativa;
                    score= (1- accumuloScoreArchi)/pow(dist_cumulativa, EXP_DIST_CUMULATIVA);


                c.distEuclidea = score;
                //cout<<"Score: "<<score<<" Best Score: "<<bestScore<<endl;

                // for (int i = stackPunti.size()-2; i >=0; i--) {
                //         int start = stackPunti[i];
                //         int end = stackPunti[i + 1];
                //         for (const Edge &e : edges[start])
                //         {
                //             int id = e.to;
                //             Vertex &v = vertex[id];
                //             if(id==end && e.direction< accumuloScoreArchi* SOGLIA_ARCHI){
                //                 posizioneTaglio=i;
                //             }
                //         }
                //     }


                //Se lo score è inferiore a quello migliore aggiorno
                if (score < bestScore)
                {
                    //cout<<"Best score trovato..."<<endl;
                    bestScore = score;
                    migliorAttuale.resize(0);

                    for (int k=0; k<posizioneTaglio; k++)
                    {
                        migliorAttuale.push_back(stackPunti[k]);
                    }
                }

            }

        }

        stackPunti.pop_back();
    }
    return valido;
}

bool Graph::ricercaRicorsivaMax(int current, vector<int> &stackPunti, float &bestScore, vector<int> &migliorAttuale, Point3D dir, int lev)
{
    bool valido = false;
    // cout<<"Current "<<current<<endl;
    if (current != -1 && (lev == 0 || !vertex[current].flagEscluso))
    {
        valido = true;
        // cout<<"Ok..."<<endl;
        Vertex &c = vertex[current];
        stackPunti.push_back(current);
        Point3D a(nodi[current].maxLocale.x, nodi[current].maxLocale.y, nodi[current].maxLocale.z, c.distance);
        float fattoreLunghezzaMinima = FATTORE_LUNGHEZZA_MINIMA_SP;

        float dist_cumulativa = 0; // distanza cumulativa tra i punti

        // Se 'stackPunti' ha meno di 2 elementi, salta il ciclo
        if (stackPunti.size() >= 2)
        {
            // for (int i = 2; i < stackPunti.size(); i++)
            int i = stackPunti.size() - 1;
            {
                // cout<<"i: "<<i<<endl;
                //  distanza totale tra il primo punto fino al punto i
                for (int k = 1; k <= i; k++)
                {
                    Point3D diff = nodi[stackPunti[k]].maxLocale - nodi[stackPunti[k - 1]].maxLocale;
                    dist_cumulativa += norm(diff);
                }
                // cout << "dist cumulativa: " << dist_cumulativa << endl;
                if (dist_cumulativa < diametro * fattoreLunghezzaMinima)
                {
                    // cout << "percorso troppo breve" << endl;
                    // continue;
                    valido = true;
                }
                else
                {

                    // Point3D v = linearRegression(stackPunti, 0, i + 1);
                    // Point3D m(0, 0, 0);
                    // for (int j = 0; j <= i; j++)
                    // {
                    //     m.x += nodi[stackPunti[j]].baricentro.x;
                    //     m.y += nodi[stackPunti[j]].baricentro.y;
                    //     m.z += nodi[stackPunti[j]].baricentro.z;
                    // }
                    // m.x /= (i + 1);
                    // m.y /= (i + 1);
                    // m.z /= (i + 1);

                    // v.normalize();
                    // //double dist = a.pointToLineDistance(a, m, v);

                    // //Nuovo, seguo pca
                    int from = stackPunti[0];
                    Point3D direzionePCA = Point3D(nodi[from].autovetMax[0], nodi[from].autovetMax[1], nodi[from].autovetMax[2]);
                    // //scarto regressione e uso direzione pca, potrei fare angolo tra i due
                    // v=direzionePCA;
                    // double dist = a.pointToLineDistance(a, nodi[from].baricentro, v);
                    // cout<<"Dist: "<<dist<<endl;
                    // if (dist > (diametro* fattoreLunghezzaMinima) / 2)
                    // {
                    //     valido = false;
                    //     cout<<"Fuori diametro "<<endl;
                    //     //break;
                    // }
                    // Assume che `stackPunti` contenga gli indici dei punti nel percorso
                    // Verifica coerenza tra PCA e direzione
                    int prev = stackPunti[stackPunti.size() - 2];
                    // differenza
                    Point3D direzione = Point3D(nodi[current].maxLocale - nodi[prev].maxLocale);
                    direzione.normalize();
                    // Analisi coerenza tra movimento e direzione PCA
                    float prodScal = fabs(direzionePCA * direzione);
                    if (prodScal < PCA_ANG)
                    {
                        valido = false;
                    }

                    // Controllo le direzioni degli ultimi due baricentri
                    Point3D pcaDirCurrent = Point3D(nodi[current].autovetMax[0], nodi[current].autovetMax[1], nodi[current].autovetMax[2]);
                    Point3D pcaDirPrev = Point3D(nodi[prev].autovetMax[0], nodi[prev].autovetMax[1], nodi[prev].autovetMax[2]);
                    float lastTwoAnglePCA = fabs(pcaDirCurrent * pcaDirPrev);
                    // angolo tra PCA corrente e arco
                    float cosAlpha = fabs(pcaDirCurrent * direzione);
                    if (cosAlpha < PCA_ANG)
                        valido = false;
                    // angolo tra arco corrente e primo arco
                    Point3D dirPrimoArco = nodi[current].maxLocale - nodi[current + 1].maxLocale;
                    float cosBeta = fabs(direzione * dirPrimoArco);
                    // Se la somma del delta tra la soglia e last
                    float deltaAngoli = PCA_ANG - lastTwoAnglePCA + (PCA_ANG - cosBeta);
                    if (deltaAngoli > PCA_ANG / 2)
                        valido = false;
                    if (lastTwoAnglePCA >= PCA_ANG && cosBeta >= PCA_ANG)
                    {
                    }
                    else
                    {
                        // Calcola il baricentro iniziale (centro della base del cilindro)
                        Point3D centroBase = nodi[from].maxLocale;
                        // Direzione del cilindro basata sull'autovettore maggiore
                        Point3D direzioneCilindro = Point3D(dir);
                        // direzioneCilindro.normalize();

                        // Raggio del cilindro
                        float raggio = diametro / 2.0f;

                        // Calcola l'altezza del cilindro (distanza cumulativa tra i punti nel percorso
                        // for (int i = 0; i < stackPunti.size() - 1; i++) {
                        //     int start = stackPunti[i];
                        //     int end = stackPunti[i + 1];
                        //     altezzaCilindro += distance(nodi[start].baricentro, nodi[end].baricentro);
                        // }

                        // Verifica ogni punto nel percorso rispetto al cilindro
                        bool tuttiPuntiValidi = true;

                        for (int i = 0; i < stackPunti.size(); i++)
                        {
                            Point3D p = nodi[stackPunti[i]].maxLocale;

                            // Vettore dal centro della base del cilindro al punto `p`
                            Point3D vettoreDalCentro = p - centroBase;

                            // Componente del vettore nella direzione del cilindro
                            float componenteLungoAsse = vettoreDalCentro * direzioneCilindro;

                            // Calcola la proiezione del punto `p` sull'asse del cilindro
                            Point3D proiezione = centroBase + (direzioneCilindro * componenteLungoAsse);

                            // Calcola la distanza tra il punto `p` e la proiezione sull'asse del cilindro
                            Point3D distanzaVettoriale = p - proiezione;
                            float distanzaDalCentro = norm(distanzaVettoriale);

                            // Verifica se il punto si trova entro il raggio del cilindro
                            if (distanzaDalCentro > raggio)
                            {
                                tuttiPuntiValidi = false;
                                // cout << "Punto fuori dal raggio del cilindro: " << i << endl;
                            }
                        }

                        // Output del risultato
                        if (tuttiPuntiValidi)
                        {
                            // cout << "Tutti i punti sono contenuti all'interno del cilindro." << endl;
                            // valido=true;
                        }
                        else
                        {
                            // cout << "Alcuni punti sono fuori dal cilindro." << endl;
                            valido = false;
                        }
                    }
                }
            }
        }
        // AGGIUNGERE TEST DEI VALORI DEI NODI
        float val_id = matrix((int)nodi[current].maxLocale.x, (int)nodi[current].maxLocale.y, (int)nodi[current].maxLocale.z);
        int id = stackPunti[0];
        float val_id0 = matrix((int)nodi[id].maxLocale.x, (int)nodi[id].maxLocale.y, (int)nodi[id].maxLocale.z);
        if (val_id < val_id0 * THRESHOLD_INTENSITA)
        {
            valido = false;
        }

        if (valido)
        {

            bool nonvalutare = false;
            for (const Edge &e : edges[current])
            {
                int id = e.to;
                Vertex &v = vertex[id];

                if (v.prev == current)
                {
                    bool prosegui = true;
                    if (lev == 0)
                    {
                        Point3D b(nodi[id].maxLocale.x, nodi[id].maxLocale.y, nodi[id].maxLocale.z, v.distance);
                        if ((b - a) * dir < 0)
                        {
                            // cout<<"Prosegui False!!"<<endl;
                            prosegui = false;
                        }
                    }
                    if (prosegui)
                    {
                        bool temp = ricercaRicorsiva(id, stackPunti, bestScore, migliorAttuale, dir, lev + 1);
                        nonvalutare = nonvalutare || temp;
                    }
                }
            }
            if (1)
            {
                // parametro da guardare
                // calcolo della posizione a cui tagliare in caso di arco molto debole
                int posizioneTaglio = stackPunti.size() - 1;
                float accumuloScoreArchi = 0.0f;
                float ctscoreArchi = 0;
                for (int i = 0; i < stackPunti.size() - 1; i++)
                {
                    int start = stackPunti[i];
                    int end = stackPunti[i + 1];
                    for (const Edge &e : edges[start])
                    {
                        int id = e.to;
                        Vertex &v = vertex[id];
                        if (id == end)
                        {
                            Point3D diff = nodi[start].maxLocale - nodi[end].maxLocale;
                            accumuloScoreArchi += (e.direction) * (norm(diff));
                            ctscoreArchi += norm(diff);
                        }
                    }
                }
                if (ctscoreArchi != 0)
                    accumuloScoreArchi /= ctscoreArchi;

                float score = 3 - (dist_cumulativa / (diametro * fattoreLunghezzaMinima));
                if (dist_cumulativa > diametro * fattoreLunghezzaMinima)
                    // score = pow(c.distance, EXP_DISTANCE_NUMERATOR) / pow(dist_cumulativa, EXP_DIST_CUMULATIVA);
                    // score= -dist_cumulativa;
                    score = (1 - accumuloScoreArchi) / pow(dist_cumulativa, EXP_DIST_CUMULATIVA);

                c.distEuclidea = score;
                // cout<<"Score: "<<score<<" Best Score: "<<bestScore<<endl;

                // for (int i = stackPunti.size()-2; i >=0; i--) {
                //         int start = stackPunti[i];
                //         int end = stackPunti[i + 1];
                //         for (const Edge &e : edges[start])
                //         {
                //             int id = e.to;
                //             Vertex &v = vertex[id];
                //             if(id==end && e.direction< accumuloScoreArchi* SOGLIA_ARCHI){
                //                 posizioneTaglio=i;
                //             }
                //         }
                //     }

                // Se lo score è inferiore a quello migliore aggiorno
                if (score < bestScore)
                {
                    // cout<<"Best score trovato..."<<endl;
                    bestScore = score;
                    migliorAttuale.resize(0);

                    for (int k = 0; k < posizioneTaglio; k++)
                    {
                        migliorAttuale.push_back(stackPunti[k]);
                    }
                }
            }
        }

        stackPunti.pop_back();
    }
    return valido;
}


/**
 * @brief Avvia il calcolo dei percorsi minimi per i nodi con il punteggio più alto nel grafo.
 *
 * Questa funzione seleziona i nodi con il punteggio più alto basandosi sul valore di baricentro e avvia
 * il calcolo del percorso minimo per ciascuno di essi. Il percorso minimo è calcolato utilizzando un
 * algoritmo basato sulla priorità dei nodi memorizzati in una coda di priorità implementata come MinHeap.
 * La funzione si occupa di gestire i nodi esclusi e di resettare l'heap dopo ogni iterazione per assicurare
 * che il cammino minimo per il prossimo nodo sia calcolato da una situazione iniziale pulita.
 *
 * La procedura include:
 * - La creazione di una lista di nodi con relativi score se non sono esclusi.
 * - L'ordinamento dei nodi per score in modo decrescente.
 * - La limitazione della ricerca ai primi 200 nodi o meno, a seconda della disponibilità.
 * - L'inizializzazione e l'uso dell'heap per il controllo delle priorità nella ricerca del percorso minimo.
 * - La determinazione e l'applicazione del diametro di influenza basato sulla PCA per ogni nodo di partenza selezionato.
 *
 * Viene effettuato un controllo per evitare di processare nodi già esclusi da altri percorsi, migliorando l'efficienza
 * e evitando ridondanze nel calcolo dei percorsi minimi.
 */
void Graph::startShortestPath()
{
    //std::cout << "Shortest Path On Graph" << std::endl;

    MinHeap heap(nodi.size());

    // Creare un vettore di coppie (indice, score) per ogni nodo non escluso
    std::vector<std::pair<int, float>> nodiConScore;

    for (int i = 0; i < nodi.size(); i++)
    {
        {
            // Aggiungi il nodo e il suo score alla lista se non è escluso
            nodiConScore.push_back(std::make_pair(i, nodi[i].baricentro.value));
        }
    }

    maxDistance = getMaxDistance();

    // Ordina i nodi in base allo score in ordine decrescente
    std::sort(nodiConScore.begin(), nodiConScore.end(),
                [](const std::pair<int, float> &a, const std::pair<int, float> &b)
                {
                    return a.second > b.second; // Ordina dal maggiore al minore
                });

    int limit = std::min(static_cast<int>(round(NODIDAITERARE*maxList.size())), (int)nodiConScore.size()); // Max per tutti
    // int limit = (int)nodiConScore.size(); // Max per tutti

     for (int i = 0; i < limit; i++)
        {
            int nodoIndex = nodiConScore[i].first;
        }
    //Inizializzo variabili per salvataggio json
    contaTubi=0;
    tubiJson= vector<TuboJSON>();

    //contatore percorsi reali
    int ct=0;
    // for (int i = 0; i < limit && ct<NODIDAITERARE; i++)
    for (int i = 0; i < limit; i++)
    {
        epoca = i;

        int nodoIndex = nodiConScore[i].first; // Ottieni l'indice del nodo

        if (!vertex[nodoIndex].flagEscluso)
        {
            ct++;
            //cout << "Score Nodo: " << nodiConScore[i].second << endl;
            //cout << "Nodo di partenza: ";
            //cout << vertex[nodoIndex].value << " Ciclo: " << i << endl;
            //cout << "Diameter... " << endl;
            float diameter = getDiameterFromPCA(nodi[nodoIndex].baricentro,
                                                nodi[nodoIndex].autovetMax,
                                                nodi[nodoIndex].autovetMid,
                                                nodi[nodoIndex].autovetMin);
            // std::cout << "Calcolando il cammino minimo a partire dal nodo " << nodoIndex << " con score " << nodi[nodoIndex].score << std::endl;
            //cout << "Diameter end..." << endl;
            this->diametro = diameter;
            this->diametroOOF = diameter * DIAMETER_INCREASE_FACTOR_OOF;
            this->diametroRR = diameter * DIAMETER_INCREASE_FACTOR_RR;
            //cout << "Begin shortest path... " << endl;
            shortestPath(nodoIndex, heap); // Calcola il cammino minimo per il nodo
        }
        else
            //cout << "Nodo " << nodoIndex << " è presente in un altro percorso" << endl;
            cout<<"";

        // Reset dell'heap dopo ogni esecuzione di shortestPath
        heap.reset(); // Assicura che l'heap sia pulito prima di calcolare il cammino per il prossimo nodo
        }
//Salvataggio su ../outputJson del file contente l'output

std::cout << "[DEBUG] Tubi trovati in tubiJson: " << tubiJson.size() << std::endl;

//QUESTA FUNZIONE VA IN CONFLITTO CON LIBRERIA DINAMICA, SALVARE SOLO SE SICURI O BISOGNA RIRUNNARE IL VISUALIZER
bool wannaSave= WANNASAVE;
if(wannaSave)
    salvaTubiFusiInJson(tubiJson, m_jsonPath, offsetUp, offsetDown, offsetLeft, offsetRight, offsetZ);
}

void Graph::startShortestPathMax()
{
    // std::cout << "Shortest Path On Graph" << std::endl;

    MinHeap heap(nodi.size());

    // Creare un vettore di coppie (indice, score) per ogni nodo non escluso
    std::vector<std::pair<int, float>> nodiConScore;

    for (int i = 0; i < nodi.size(); i++)
    {
        {
            // Aggiungi il nodo e il suo score alla lista se non è escluso
            nodiConScore.push_back(std::make_pair(i, nodi[i].maxLocale.value));
        }
    }

    maxDistance = getMaxDistance();

    // Ordina i nodi in base allo score in ordine decrescente
    std::sort(nodiConScore.begin(), nodiConScore.end(),
              [](const std::pair<int, float> &a, const std::pair<int, float> &b)
              {
                  return a.second > b.second; // Ordina dal maggiore al minore
              });

    int limit = std::min(static_cast<int>(round(NODIDAITERARE * maxList.size())), (int)nodiConScore.size()); // Max per tutti
                                                                                                             // int limit = (int)nodiConScore.size(); // Max per tutti

    for (int i = 0; i < limit; i++)
    {
        int nodoIndex = nodiConScore[i].first;
    }
    // Inizializzo variabili per salvataggio json
    contaTubi = 0;
    tubiJson = vector<TuboJSON>();

    // contatore percorsi reali
    int ct = 0;
    // for (int i = 0; i < limit && ct<NODIDAITERARE; i++)
    for (int i = 0; i < limit; i++)
    {
        epoca = i;

        int nodoIndex = nodiConScore[i].first; // Ottieni l'indice del nodo

        if (!vertex[nodoIndex].flagEscluso)
        {
            ct++;
            // cout << "Score Nodo: " << nodiConScore[i].second << endl;
            // cout << "Nodo di partenza: ";
            // cout << vertex[nodoIndex].value << " Ciclo: " << i << endl;
            // cout << "Diameter... " << endl;
            float diameter = getDiameterFromPCA(nodi[nodoIndex].maxLocale,
                                                nodi[nodoIndex].autovetMax,
                                                nodi[nodoIndex].autovetMid,
                                                nodi[nodoIndex].autovetMin);
            // std::cout << "Calcolando il cammino minimo a partire dal nodo " << nodoIndex << " con score " << nodi[nodoIndex].score << std::endl;
            // cout << "Diameter end..." << endl;
            this->diametro = diameter;
            this->diametroOOF = diameter * DIAMETER_INCREASE_FACTOR_OOF;
            this->diametroRR = diameter * DIAMETER_INCREASE_FACTOR_RR;
            // cout << "Begin shortest path... " << endl;
            shortestPath(nodoIndex, heap); // Calcola il cammino minimo per il nodo
        }
        else
            // cout << "Nodo " << nodoIndex << " è presente in un altro percorso" << endl;
            cout << "";

        // Reset dell'heap dopo ogni esecuzione di shortestPath
        heap.reset(); // Assicura che l'heap sia pulito prima di calcolare il cammino per il prossimo nodo
    }
    // Salvataggio su ../outputJson del file contente l'output

    std::cout << "[DEBUG] Tubi trovati in tubiJson: " << tubiJson.size() << std::endl;

    // QUESTA FUNZIONE VA IN CONFLITTO CON LIBRERIA DINAMICA, SALVARE SOLO SE SICURI O BISOGNA RIRUNNARE IL VISUALIZER
    bool wannaSave = WANNASAVE;
    if (wannaSave)
        salvaTubiFusiInJson(tubiJson, m_jsonPath, offsetUp, offsetDown, offsetLeft, offsetRight, offsetZ);

    // cout << "Numero di tubi trovati: " << tubiJson.size() << endl;
    // cout << "WANNASAVE: " << WANNASAVE << endl;
    // cout << "Percorso di salvataggio: ../output Json" << endl;
}


/**
 * @brief confronto le coordinate intere di un baricentro con quelle date
 *  e ritorno true se ho almeno un baricentro che le rispetti
 *
 * @param x Coordinata X
 * @param y Coordinata Y
 * @param z Coordinata Z
 */
bool Graph::isBaricenter(int x, int y, int z){
    for (auto& nodo : nodi) {
        Point3D bar=nodo.baricentro;
        //approssimo coordinate baricentriche a interi
        int xBar = static_cast<int>(round(bar.getX()));
        int yBar = static_cast<int>(round(bar.getY()));
        int zBar = static_cast<int>(round(bar.getZ()));

        if (x == xBar && y== yBar && z == zBar)
            return true;
    }
    return false;

}
/**
 * @brief Controllo se il punto dato è un massimo locale
 *
 * @param x Coordinata X
 * @param y Coordinata Y
 * @param z Coordinata Z
 */
bool Graph::isMaxLocal(const Point3D &maxLocal)
{
    for (auto& max : maxList) {
        if (max.x==maxLocal.x && max.y==maxLocal.y && max.z==maxLocal.z)
            return true;
    }
    return false;

}


/**
 * @brief Calcola il diametro basato sull'analisi PCA di un insieme di punti attorno a un baricentro nel contesto di una matrice 3D.
 *
 * Questa funzione calcola una stima del diametro di un insieme di punti attorno a un dato baricentro. Utilizza gli autovettori derivati
 * da un'analisi PCA per definire un piano di interesse e misura il raggio di un insieme di punti che cadono entro una certa distanza
 * dal piano definito dagli autovettori "medio" e "minimo". Il baricentro viene arrotondato ai valori interi più vicini per allinearsi
 * con gli indici della matrice 3D. Viene utilizzata una soglia per determinare quali punti considerare basandosi sulla loro distanza
 * dal piano principale definito dagli autovettori.
 *
 * Operazioni principali:
 * - Calcolo delle coordinate arrotondate del baricentro.
 * - Definizione di un insieme (matrice) di punti intorno al baricentro.
 * - Calcolo del gradiente e dell'intensità per ogni punto rispetto ai piani definiti dagli autovettori.
 * - Calcolo della proiezione dei punti sui piani per determinare le loro coordinate trasformate.
 * - Aggregazione e normalizzazione dei contributi di ciascun punto per stime del raggio.
 * - Calcolo del diametro finale basato sulla media pesata del raggio, considerando sia la densità che la distanza dei punti dal piano.
 *
 * @param bar Punto di riferimento centrale per l'analisi.
 * @param autovetMax Vettore PCA con la massima varianza.
 * @param autovetMid Vettore PCA con varianza media.
 * @param autovetMin Vettore PCA con la minima varianza.
 * @return Il diametro stimato del cluster di punti.
 */
float Graph::getDiameterFromPCA(const Point3D& bar,
    const float autovetMax[3],
    const float autovetMid[3],
    const float autovetMin[3])
{

    // Arrotonda il baricentro ai valori interi più vicini
    int bxInt = static_cast<int>(round(bar.getX()));
    int byInt = static_cast<int>(round(bar.getY()));
    int bzInt = static_cast<int>(round(bar.getZ()));
    Point3D intBar=Point3D(bxInt, byInt, bzInt);

    //Raggio di intorno su cui analizzare i punti
    int s=20;
    // Dimensione della matrice: (2s+1) x (2s+1)
    int size = 2 * S + 1;

    vector<vector<RadMatrixElement>> radMatrix(size, vector<RadMatrixElement>(size));

    vector<float> intensity;
    vector<float> angoliGrad;
    vector<Point3D> projectedPoints;
    // Scorri tutti i punti tra (bx-s, by-s, bz-s) e (bx+s, by+s, bz+s) e salva in stackPunti
    // quelli con distanza dal piano dato dagli autovettori medio e minimo minore o uguale a 3
    int sogliaDistanzaPiano=SOGLIA_DISTANZA_PIANO;
    float altezzaCilindro=0.0f;
    //cout<<"Cicla sui vicini... "<<endl;
    for (int x = bxInt - s; x <= bxInt + s; x++)
    {
        for (int y = byInt - s; y <= byInt + s; y++)
        {
            for (int z = bzInt - s; z <= bzInt + s; z++)
            {
                // Controlla se (x, y, z) è un punto valido nella tua matrice 3D
                if ((x>=0 && x <= this->matrix.getX()-2) && (y>=0 && y <= this->matrix.getY()-2) && (z<= this->matrix.getZ()-2 && z >= 0))
                {
                    Point3D candidato=Point3D(x,y,z);
                    //candidato.print();
                    //controllo che la distanza dal piano sia <= soglia
                    float distanzaPiano= fabs((candidato-bar) * Point3D(autovetMax[0], autovetMax[1], autovetMax[2]));

                    //if(distanzaPiano<0) distanzaPiano=fabs(distanzaPiano);
                    //cout<<"distanza dal Piano: "<<distanzaPiano<<endl;
                    if(distanzaPiano <= SOGLIA_DISTANZA_PIANO)
                    {
                            // Calcola i gradienti utilizzando il punto proiettato
                            float gradX = matrix.getValData(x, y, z) -
                                        matrix.getValData(x + 1, y, z);
                             float gradY = matrix.getValData(x, y, z) -
                                        matrix.getValData(x, y+1, z);
                            float gradZ = matrix.getValData(x, y, z) -
                                        matrix.getValData(x, y, z+1);

                            Point3D grad= Point3D(gradX, gradY, gradZ); //Point3D inteso come vettore 3D, non punto

                            //peso del candidato
                            float intensityCandidato= grad.norm(grad);

                            grad.normalize();
                            //calcolo angolo
                            Point3D direction= candidato - bar;

                            Point3D directionNorm= Point3D(direction);
                            directionNorm.normalize();
                            float cosTheta = grad * directionNorm; // Questo sarà già normalizzato tra -1 e 1
                            if(cosTheta<0) cosTheta=0;


                            //Calcolo X e Y proiettate
                            float xProjected= direction * Point3D(autovetMid[0], autovetMid[1], autovetMid[2]);
                            float yProjected= direction * Point3D(autovetMin[0], autovetMin[1], autovetMin[2]);
                            float zProjected= direction * Point3D(autovetMax[0], autovetMax[1], autovetMax[2]);
                            int xInt=static_cast<int>(round(xProjected))+s;
                            int yInt=static_cast<int>(round(yProjected))+s;

                            // Spostamenti rispetto al centro (baricentro)
                            float dx = xProjected * autovetMid[0] + yProjected * autovetMin[0];
                            float dy = xProjected * autovetMid[1] + yProjected * autovetMin[1];
                            float dz =  xProjected * autovetMid[2] + yProjected * autovetMin[2];

                            // Calcola le coordinate 3D del punto centrato sul baricentro
                            float xP= bar.x + dx;
                            float yP= bar.y + dy;
                            float zP= bar.z + dz;

                            //Inizializzo la matrice
                            if(xInt>=0 && xInt<size && yInt>=0 && yInt<size)
                            {
                                radMatrix[xInt][yInt].conteggio++;
                                radMatrix[xInt][yInt].accumulo += intensityCandidato * pow(cosTheta,2);
                                radMatrix[xInt][yInt].distPointPlane=distanzaPiano;
                            }
                    }
                }
            }
        }
    }

    //cout<<"Inizializza la matrice..."<<endl;
    // Inizializza la matrice
    for (int i = 0; i < size; ++i) {
        for (int j = 0; j < size; ++j) {
            // Spostamenti rispetto al centro (baricentro)
            float dx = (i - s) * autovetMid[0] + (j - s) * autovetMin[0];
            float dy = (i - s) * autovetMid[1] + (j - s) * autovetMin[1];
            float dz = (i - s) * autovetMid[2] + (j - s) * autovetMin[2];

            // Calcola le coordinate 3D del punto centrato sul baricentro
            float x= bar.x + dx;
            float y= bar.y + dy;
            float z= bar.z + dz;

            Point3D point=Point3D(x, y, z); //coordinate float

            //aggiorno il valore
            if(radMatrix[i][j].conteggio>0)
                point.setValue(radMatrix[i][j].accumulo / radMatrix[i][j].conteggio);
            else{
                point.setValue(0.0f);
            }

            //inserisco le coordinate
            radMatrix[i][j].coord = point;
            //Pusho i punti della matrice per il plot
            radPoints.push_back(point);


        }
    }


    float raggio = 0.0f;
    float totalWeight = 0.0f; // Somma totale dei pesi per la media
    
    //scorro gli elementi della matrice
    for (int i = 0; i < size; ++i) {
        for (int j = 0; j < size; ++j) {
            // Verifica se l'elemento è valido (non out of matrix e conteggio > 0)
            if (radMatrix[i][j].conteggio > 0) {
                // Calcola la distanza tra le coordinate del punto sulla matrice e il baricentro
                float dist= distance(radMatrix[i][j].coord, bar);

                // Calcola il peso basato sul rapporto accumulo/conteggio
                float weight = radMatrix[i][j].accumulo / radMatrix[i][j].conteggio;
                weight=pow(weight,2); //riduce rumore
                // Calcola il peso modificato basato sulla distanza dal piano
                // Linear scaling: peso = 1 - (distanzaPiano / sogliaDistanzaPiano)
                float scaledWeight = 1.0 - (radMatrix[i][j].distPointPlane / SOGLIA_DISTANZA_PIANO);
                scaledWeight = max(scaledWeight, 0.0f); // Assicurati che il peso non sia negativo

                // Somma pesata per il calcolo del raggio
                raggio += dist * weight * scaledWeight;
                totalWeight += weight * scaledWeight;
            }
        }
    }


    // Calcola la media pesata finale se il totale dei pesi non è zero
    if (totalWeight > 0.0f) {
        raggio /= totalWeight;
    }
    //cout<<"Raggio Stimato: "<<raggio<<endl;

    //Plot della circonferenza "sezione" del tubo di raggio "raggio" e centrato sul baricentro
    float circlePrecision=20.0;
    for(int i=0; i<circlePrecision; i++)
    {
        float pX= raggio* cos(i/circlePrecision * 2 * 3.1415);
        float pY= raggio* sin(i/circlePrecision * 2 * 3.1415);
        float dx = pX * autovetMid[0] + pY * autovetMin[0];
        float dy = pX * autovetMid[1] + pY * autovetMin[1];
        float dz = pX * autovetMid[2] + pY * autovetMin[2];

        float x= bar.x + dx;
        float y= bar.y + dy;
        float z= bar.z + dz;
        Point3D p1(x,y,z);

        pX= raggio* cos((i+1)/circlePrecision * 2 * 3.1415);
        pY= raggio* sin((i+1)/circlePrecision * 2 * 3.1415);
        dx = pX * autovetMid[0] + pY * autovetMin[0];
        dy = pX * autovetMid[1] + pY * autovetMin[1];
        dz = pX * autovetMid[2] + pY * autovetMin[2];

        x= bar.x + dx;
        y= bar.y + dy;
        z= bar.z + dz;
        Point3D p2(x,y,z);
        Line linea = Line(p1, p2, 0.0f);
        sectionCircle.push_back(linea);
    }
    return 2*raggio; //diametro
}
/**
 * @brief Restituisce le linee geometriche degli archi trovati nel grafo
 *
 * @return vector<Line> contenente le linee geometriche che rappresentano gli archi del grafo
 */
vector<Line> &Graph::getLines()
{
    return lines;
}

