#include "ImageLoader.hh"
#include <vector>
#include <filesystem>
#include <fstream>
#include <cassert>
#include <cmath>
#include <algorithm>  // Per std::sort
#include <chrono>     // Per timing
#include <opencv2/opencv.hpp>
#include "Matrix3D.hh"
#include <regex>
namespace fs = filesystem;
using namespace std; 

#include <opencv2/opencv.hpp>
#include <iostream>
#include <gdal_priv.h>
#include <cpl_conv.h> 

void ImageLoader::resizeGeoreferencedTiff(const std::string &inputPath, const std::string &outputPath, double targetGSD) {
    GDALAllRegister();

    GDALDataset *srcDataset = static_cast<GDALDataset *>(GDALOpen(inputPath.c_str(), GA_ReadOnly));
    if (!srcDataset) {
        std::cerr << "Impossibile aprire il file: " << inputPath << std::endl;
        return;
    }

    double geoTransform[6];
    if (srcDataset->GetGeoTransform(geoTransform) != CE_None) {
        std::cerr << "Errore nel recuperare la trasformazione georeferenziata: " << inputPath << std::endl;
        GDALClose(srcDataset);
        return;
    }

    //std::cout << "GeoTransform recuperato:\n";
    // for (int i = 0; i < 6; i++) {
    //     std::cout << geoTransform[i] << " ";
    // }
    // std::cout << std::endl;

    double originalGSD = geoTransform[1]; // Dimensione del pixel
    double scaleFactor = originalGSD / targetGSD;
    //se devo fare upsample setto la scala a 1
    if(scaleFactor >1 ) scaleFactor=1;

    int newWidth = static_cast<int>(srcDataset->GetRasterXSize() * scaleFactor);
    int newHeight = static_cast<int>(srcDataset->GetRasterYSize() * scaleFactor);

    //std::cout << "Dimensioni originali: " << srcDataset->GetRasterXSize() << "x" << srcDataset->GetRasterYSize() << std::endl;
    //std::cout << "Nuove dimensioni: " << newWidth << "x" << newHeight << std::endl;

    geoTransform[1] = targetGSD;
    geoTransform[5] = -targetGSD;

    GDALDriver *driver = GetGDALDriverManager()->GetDriverByName("GTiff");
    if (!driver) {
        std::cerr << "Driver GTiff non disponibile!" << std::endl;
        GDALClose(srcDataset);
        return;
    }

    GDALDataset *outputDataset = driver->Create(
        outputPath.c_str(),
        newWidth,
        newHeight,
        srcDataset->GetRasterCount(),
        srcDataset->GetRasterBand(1)->GetRasterDataType(),
        nullptr
    );

    if (!outputDataset) {
        std::cerr << "Errore nella creazione del file di output: " << outputPath << std::endl;
        GDALClose(srcDataset);
        return;
    }

    // std::cout << "File di output creato con successo: " << outputPath << std::endl;

    outputDataset->SetGeoTransform(geoTransform);
    outputDataset->SetProjection(srcDataset->GetProjectionRef());

    for (int i = 1; i <= srcDataset->GetRasterCount(); i++) {
        GDALRasterBand *srcBand = srcDataset->GetRasterBand(i);
        GDALRasterBand *dstBand = outputDataset->GetRasterBand(i);

        void *data = CPLMalloc(newWidth * newHeight * GDALGetDataTypeSizeBytes(srcBand->GetRasterDataType()));
        if (srcBand->RasterIO(
                GF_Read,
                0,
                0,
                srcDataset->GetRasterXSize(),
                srcDataset->GetRasterYSize(),
                data,
                newWidth,
                newHeight,
                srcBand->GetRasterDataType(),
                0,
                0
            ) != CE_None) {
            std::cerr << "Errore nel ridimensionamento del raster." << std::endl;
            CPLFree(data);
            GDALClose(srcDataset);
            GDALClose(outputDataset);
            return;
        }

        if (dstBand->RasterIO(
                GF_Write,
                0,
                0,
                newWidth,
                newHeight,
                data,
                newWidth,
                newHeight,
                srcBand->GetRasterDataType(),
                0,
                0
            ) != CE_None) {
            std::cerr << "Errore nella scrittura del raster." << std::endl;
            CPLFree(data);
            GDALClose(srcDataset);
            GDALClose(outputDataset);
            return;
        }

        CPLFree(data);
    }

    // std::cout << "Ridimensionamento completato per: " << outputPath << std::endl;

    GDALClose(outputDataset);
    GDALClose(srcDataset);
}

void ImageLoader::processGeoreferencedImages(const std::string &inputDir, const std::string &outputDir, double targetGSD) {
    // Controlla se la directory di input esiste
    if (!fs::exists(inputDir)) {
        std::cerr << "Errore: La directory di input non esiste: " << inputDir << std::endl;
        return;
    }

//    // Rimuove la directory di output se esiste
//    if (fs::exists(outputDir)) {
//        std::cout << "Eliminazione della directory esistente: " << outputDir << std::endl;
//        fs::remove_all(outputDir); // Rimuove la directory e il suo contenuto
//    }

    // Ricrea la directory di output
    std::cout << "Creazione della directory: " << outputDir << std::endl;
    try {
        fs::create_directory(outputDir); // Ricrea la directory
    } catch (const std::filesystem::filesystem_error &e) {
        std::cerr << "Errore durante la creazione della directory: " << e.what() << std::endl;
        return;
    }

    // Processa i file nella directory di input
    try {
        for (const auto &entry : fs::directory_iterator(inputDir)) {
            if (entry.path().extension() == ".tiff") {
                std::string inputPath = entry.path().string();
                std::string outputPath = outputDir + "/resized_" + entry.path().filename().string();

                // std::cout << "Processing: " << inputPath << " -> " << outputPath << std::endl;

                // Chiama la funzione per ridimensionare il file
                try {
                    resizeGeoreferencedTiff(inputPath, outputPath, targetGSD);
                } catch (const std::exception &e) {
                    std::cerr << "Errore durante il ridimensionamento dell'immagine: " << inputPath << " - " << e.what() << std::endl;
                }
            } else {
                std::cout << "File ignorato (non TIFF): " << entry.path().string() << std::endl;
            }
        }
    } catch (const std::filesystem::filesystem_error &e) {
        std::cerr << "Errore durante l'iterazione della directory di input: " << e.what() << std::endl;
        return;
    }

    // Debug: Verifica della directory di output
    std::cout << "Verifica della directory di output: " << outputDir << std::endl;

    // Normalizza il percorso per evitare problemi
    std::string correctedPath = fs::absolute(outputDir).string();
    std::cout << "Percorso corretto: " << correctedPath << std::endl;

    if (!fs::exists(correctedPath)) {
        std::cerr << "Errore: La directory non esiste: " << correctedPath << std::endl;
        return;
    }

//    // Itera i file nella directory di output
//    try {
//        for (const auto &entry : fs::directory_iterator(correctedPath)) {
//            std::cout << "File trovato: " << entry.path().string() << std::endl;
//        }
//    } catch (const std::filesystem::filesystem_error &e) {
//        std::cerr << "Errore durante l'accesso alla directory: " << e.what() << std::endl;
//    }

}





ImageLoader::CroppedPixels ImageLoader::calculateCroppedPixels(const cv::Mat &image) {
    CroppedPixels result = {0, 0, 0, 0};

    // Analizza la parte superiore
    for (int i = 0; i < image.rows; i++) {
        bool rowHasContent = false;
        for (int j = 0; j < image.cols; j++) {
            if (image.at<uchar>(i, j) != 0) {  // Se troviamo un pixel non nero
                rowHasContent = true;
                break;
            }
        }
        if (rowHasContent) break;
        result.top++;
    }

    // Analizza la parte inferiore
    for (int i = image.rows - 1; i >= 0; i--) {
        bool rowHasContent = false;
        for (int j = 0; j < image.cols; j++) {
            if (image.at<uchar>(i, j) != 0) {
                rowHasContent = true;
                break;
            }
        }
        if (rowHasContent) break;
        result.bottom++;
    }

    // Analizza la parte sinistra
    for (int j = 0; j < image.cols; j++) {
        bool colHasContent = false;
        for (int i = 0; i < image.rows; i++) {
            if (image.at<uchar>(i, j) != 0) {
                colHasContent = true;
                break;
            }
        }
        if (colHasContent) break;
        result.left++;
    }

    // Analizza la parte destra
    for (int j = image.cols - 1; j >= 0; j--) {
        bool colHasContent = false;
        for (int i = 0; i < image.rows; i++) {
            if (image.at<uchar>(i, j) != 0) {
                colHasContent = true;
                break;
            }
        }
        if (colHasContent) break;
        result.right++;
    }

    return result;
}

double ImageLoader::gauss(double sigma, double x)
{
    if(sigma == 0)
        return x == 0 ? 1.0 : 0.0;
    double expVal = -1 * (x * x) / (2 * sigma * sigma);
    double divider = sqrt(2 * M_PI * sigma * sigma);
    return exp(expVal) / divider;
}

void ImageLoader::gausKernel(float *&kernel, int samples, double sigma)
{
    int steps = (samples - 1) / 2;
    double stepSize = (3 * sigma) / steps;
    int index = 0;

    for (int i = steps; i >= 1; i--)
    {
        kernel[index++] = gauss(sigma, i * stepSize * -1);
    }

    kernel[index++] = gauss(sigma, 0);

    for (int i = 1; i <= steps; i++)
    {
        kernel[index++] = gauss(sigma, i * stepSize);
    }

    assert(index == samples);
}

void ImageLoader::gaussSmoothenX(float *&kernel, Matrix3D<float> &mat, Matrix3D<float> &out, int z, int y, double sigma, int samples)
{
    int sampleSide = samples / 2;
    int ubound = mat.getX();
    for (int i = 0; i < ubound; i++)
    {
        double sample = 0;
        float sampleCtr = 0;
        for (long j = i - sampleSide; j <= i + sampleSide; j++)
        {
            if (j > 0 && j < ubound)
            {
                int sampleWeightIndex = sampleSide + (j - i);
                sample += kernel[sampleWeightIndex] * mat(j, y, z );
                sampleCtr += kernel[sampleWeightIndex];
            }
        }
        float smoothed = 0.0f;
        if (sampleCtr != 0)
            smoothed = sample / sampleCtr;

        out(i, y, z) = smoothed;
    }
}

void ImageLoader::gaussSmoothenY(float *&kernel, Matrix3D<float> &mat, Matrix3D<float> &out, int z, int x, double sigma, int samples)
{
    int sampleSide = samples / 2;
    int ubound = mat.getY();
    int index = 0;
    for (int i = 0; i < ubound; i++)
    {
        double sample = 0;
        float sampleCtr = 0;
        for (long j = i - sampleSide; j <= i + sampleSide; j++)
        {
            if (j > 0 && j < ubound)
            {
                int sampleWeightIndex = sampleSide + (j - i);
                sample += kernel[sampleWeightIndex] * mat(x, j, z);
                sampleCtr += kernel[sampleWeightIndex];
            }
        }
        float smoothed = 0.0f;
        if (sampleCtr != 0)
            smoothed = sample / sampleCtr;

        out(x, i, z) = smoothed;
    }
}

void ImageLoader::gaussSmoothenZ(float *&kernel, Matrix3D<float> &mat, Matrix3D<float> &out, int y, int x, double sigma, int samples)
{
    int sampleSide = samples / 2;
    int ubound = mat.getZ();
    for (int i = 0; i < ubound; i++)
    {
        double sample = 0;
        float sampleCtr = 0;
        for (int j = i - sampleSide; j <= i + sampleSide; j++)
        {
            if (j > 0 && j < ubound)
            {
                int sampleWeightIndex = sampleSide + (j - i);
                sample += kernel[sampleWeightIndex] * mat(x, y, j);
                sampleCtr += kernel[sampleWeightIndex];
            }
        }
        float smoothed = 0.0f;
        if (sampleCtr != 0)
            smoothed = sample / sampleCtr;
        out(x, y, i) = smoothed;
    }
}

bool ImageLoader::contains(string file, vector<fs::directory_entry> files)
{
    for (const auto &entry : files)
    {
        if (entry.path().filename().string() == file)
            return true;
    }
    return false;
}

bool sortByCm(const string &a, const string &b)
{
    // Extract the number before "cm.tiff" from the filenames
    size_t a_pos = a.rfind('_');
    size_t b_pos = b.rfind('_');
    string a_num_str = a.substr(a_pos + 1, a.find("cm.tiff") - a_pos - 1);
    string b_num_str = b.substr(b_pos + 1, b.find("cm.tiff") - b_pos - 1);

    // Check if the substrings consist only of digits
    if (a_num_str.find_first_not_of("0123456789") != string::npos ||
        b_num_str.find_first_not_of("0123456789") != string::npos)
    {
        // Handle invalid input (e.g., non-numeric substring)
        // You can choose how to handle this case (e.g., by returning false)
        return false;
    }

    int a_num = stoi(a_num_str);
//    cout<<a<<" a_num "<<a_num<<endl;
    int b_num = stoi(b_num_str);
//    cout<<b<<" b_num "<<b_num<<endl;
    return a_num < b_num; // Compare the extracted numbers
}

vector<fs::directory_entry>
// restituisce una lista di file da una directory e li ordina
ImageLoader::getFiles(string path)
{
    vector<fs::directory_entry> files;
    {
        for (const auto &entry : fs::directory_iterator(path))
            if (entry.is_regular_file() && entry.path().filename().string().substr(0, 2) != "._")
            {
                files.push_back(entry);
            }
    }

    sort(files.begin(), files.end(), [](const fs::directory_entry &entry1, const fs::directory_entry &entry2)
              { return sortByCm(entry1.path().filename().string(), entry2.path().filename().string()); });

    return files;
}

ImageLoader::ImageLoader(){
    string outputImages = "../Cartelle/Resampled_C";
    string inputImages = "../Cartelle/Tiff_Input";

    try
    {
        if (!fs::exists(inputImages))
            fs::create_directory(inputImages);

        if (!fs::exists(outputImages))
            fs::create_directory(outputImages);

//        if (!fs::exists(data))
//            fs::create_directory(data);
    }
    catch (const exception &e)
    {
        cerr <<"Error: "<< e.what() << endl;
    }
}

bool ImageLoader::compareFilenames(const fs::directory_entry &a, const fs::directory_entry &b) {
    std::regex rgx("\\d+");  // Regular expression to find digits
    std::smatch matchA, matchB;
    std::string filenameA = a.path().filename().string();
    std::string filenameB = b.path().filename().string();
    
    std::regex_search(filenameA, matchA, rgx);
    std::regex_search(filenameB, matchB, rgx);
    
    int numberA = matchA.empty() ? 0 : std::stoi(matchA.str());
    int numberB = matchB.empty() ? 0 : std::stoi(matchB.str());
    
    return numberA < numberB;
}

void ImageLoader::loadImages(std::string dirName) {
    std::string path = dirName;

    try {
        int counter = 0; 
        std::vector<fs::directory_entry> files = this->getFiles(path); 
        std::cout << "Loading Images..." << std::endl; 
        for (const auto &entry : files) {
            std::cout << counter << " " << entry.path() << std::endl; 
            cv::Mat image = cv::imread(entry.path(), cv::IMREAD_GRAYSCALE); 
            
            if (!image.empty()) {
                if (counter == 0) {
                    string name= entry.path().filename().string();
                    size_t a_pos = name.rfind('_');
                    string a_num_str = name.substr(a_pos + 1, name.find("cm.tiff") - a_pos - 1);
                    int a_num = stoi(a_num_str);

                    CroppedPixels cropped = calculateCroppedPixels(image);
                    // std::cout << "Pixel neri tagliati:\n"
                    //           << "In alto: " << cropped.top << " pixel\n"
                    //           << "In basso: " << cropped.bottom << " pixel\n"
                    //           << "A sinistra: " << cropped.left << " pixel\n"
                    //           << "A destra: " << cropped.right << " pixel\n";
                    top=cropped.top;
                    bottom=cropped.bottom;
                    left=cropped.left;
                    right=cropped.right;
                    z=a_num;       
                    matrix3d = Matrix3D<float>(files.size(), image); 
                } else {
                    matrix3d.createFromCV(image, counter); 
                }
            } else {
                std::cerr << "Unable to load the image" << std::endl;
            }
            counter++; 
        }
        // std::cout << "Images Loaded!" << std::endl;
    }
    catch (const std::filesystem::filesystem_error &e) {
        std::cerr << "Error accessing folder: " << e.what() << std::endl; 
    }
}

//void ImageLoader::loadBin(string fileName)
//{
//    matrix3d = Matrix3D<float>("../data", fileName);
//    matrix3d.printSize();
//}

//void ImageLoader::splitBin(string fileName, int splitSizeX, int splitSizeY)
//{
//    string dataDir = "../data/";
//
//    for (const auto &entry : fs::directory_iterator(dataDir))
//    {
//        string filePath = entry.path().string();
//        string fileBaseName = fileName + "_part_";
//        if (filePath.find(fileBaseName) != string::npos)
//        {
//            fs::remove(entry.path());
//        }
//    }
//
//    matrix3d = Matrix3D<float>(dataDir, fileName);
//    matrix3d.printSize();
//
//    int numSplitsX = (matrix3d.getX() + splitSizeX - 1) / splitSizeX;
//    int numSplitsY = (matrix3d.getY() + splitSizeY - 1) / splitSizeY;
//
//        for (int splitY = 0; splitY < numSplitsY; ++splitY)
//    {
//        for (int splitX = 0; splitX < numSplitsX; ++splitX)
//        {
//            // Crea il nome del file per la suddivisione corrente
//            string splitFileName = fileName + "_part_" + to_string(splitY) + "_" + to_string(splitX);
//
//            // Scrive le dimensioni della matrice per la suddivisione corrente
//            ofstream sizeFile(dataDir + "size_" + splitFileName + ".bin", ios::binary);
//            if (!sizeFile.is_open())
//            {
//                cerr << "Error opening size file for writing: " << splitFileName << endl;
//                continue;
//            }
//
//            int sizeZ = matrix3d.getZ();
//            int sizeY = min(splitSizeY, matrix3d.getY() - splitY * splitSizeY);
//            int sizeX = min(splitSizeX, matrix3d.getX() - splitX * splitSizeX);
//            //TODO: sizes 5 con offset
//            int sizes[3] = {sizeZ, sizeY, sizeX};
//            sizeFile.write(reinterpret_cast<char *>(sizes), sizeof(sizes));
//            sizeFile.close();
//
//            ofstream dataFile(dataDir + "data_" + splitFileName + ".bin", ios::binary);
//            if (!dataFile.is_open())
//            {
//                cerr<<"Errore opening data file for writing: " << splitFileName << endl;
//                continue;
//            }
//
//            // TODO: mettere x y z
//            for (int z = 0; z < matrix3d.getZ(); ++z)
//            {
//                for (int y = 0; y < splitSizeY && (splitY * splitSizeY + y) < matrix3d.getY(); ++y)
//                {
//                    dataFile.write(reinterpret_cast<char *>(matrix3d.ptr(z, (splitY * splitSizeY) + y) + (splitX * splitSizeX)), sizeof(float) * min(splitSizeX, matrix3d.getX() - (splitX * splitSizeX)));
//                }
//            }
//            dataFile.close();
//        }
//    }
//}

//void ImageLoader::save(string fileName)
//{
//    matrix3d.save("../data", fileName);
//}

void ImageLoader::applyGauss(int sigma)
{

    Matrix3D<float> out(matrix3d.getX(), matrix3d.getY(), matrix3d.getZ());
    //Matrix3D<float> out(matrix3d.getZ(), matrix3d.getY(), matrix3d.getX());
    int samples = ceil(sigma * 3); // obbligatorio dispari
    samples = (samples % 2 == 0) ? samples + 1 : samples;

    float *kernel = new float[samples];

    this->gausKernel(kernel, samples, sigma);

    cout << "Starting X gaussian" << endl;
    for (int i = 0; i < matrix3d.getZ(); i++)
    {
        for (int j = 0; j < matrix3d.getY(); j++)
        {
            gaussSmoothenX(kernel, matrix3d, out, i, j, sigma, samples);
        }
    }
    cout << "Ended X gaussian" << endl;

    cout << "Starting Y gaussian" << endl;
    for (int i = 0; i < matrix3d.getZ(); i++){
        for (int j = 0; j < matrix3d.getX(); j++)
        {
            gaussSmoothenY(kernel, out, matrix3d, i, j, sigma, samples);
        }
    }
    cout << "Ended Y gaussian" << endl;

    cout << "Starting Z gaussian" << endl;
    for (int i = 0; i < matrix3d.getY(); i++)
    {
        for (int j = 0; j < matrix3d.getX(); j++)
        {
            gaussSmoothenZ(kernel, matrix3d, out, i, j, sigma, samples);
        }
    }
    cout << "Ended Z gaussian" << endl;

    delete[] kernel;
    matrix3d = move(out);
}

Matrix3D<float> &ImageLoader::getMatrix()
{
    return matrix3d;
}


// NUOVO: Implementazione per caricare formato binario
void ImageLoader::loadBinaryTomography(std::string binaryFile) {
    std::cout << "Loading Binary Tomography File: " << binaryFile << std::endl;
    
    auto start_time = std::chrono::high_resolution_clock::now();
    
    try {
        std::ifstream file(binaryFile, std::ios::binary);
        if (!file.is_open()) {
            throw std::runtime_error("Impossibile aprire il file binario: " + binaryFile);
        }
        
        // Leggi header: width, height, depth, data_size
        uint32_t width, height, depth, data_size;
        file.read(reinterpret_cast<char*>(&width), sizeof(uint32_t));
        file.read(reinterpret_cast<char*>(&height), sizeof(uint32_t));
        file.read(reinterpret_cast<char*>(&depth), sizeof(uint32_t));
        file.read(reinterpret_cast<char*>(&data_size), sizeof(uint32_t));
        
        // header info
        
        // Verifica che data_size sia 4 (float32)
        if (data_size != 4) {
            throw std::runtime_error("Tipo dati non supportato. Atteso: 4 bytes (float32), trovato: " + std::to_string(data_size));
        }
        
        // Calcola dimensione totale dei dati
        size_t total_elements = width * height * depth;
        size_t data_bytes = total_elements * data_size;
        
        // loading info
        
        // Crea matrice 3D
        matrix3d = Matrix3D<float>(width, height, depth);
        
        // Leggi tutti i dati in una volta
        std::vector<float> data_buffer(total_elements);
        file.read(reinterpret_cast<char*>(data_buffer.data()), data_bytes);
        
        if (file.gcount() != static_cast<std::streamsize>(data_bytes)) {
            throw std::runtime_error("Errore nella lettura dei dati. Letti " + std::to_string(file.gcount()) + 
                                    " bytes, attesi " + std::to_string(data_bytes));
        }
        
        // Copia i dati nella matrice 3D
        auto copy_start = std::chrono::high_resolution_clock::now();
        
        for (size_t z = 0; z < depth; z++) {
            for (size_t y = 0; y < height; y++) {
                for (size_t x = 0; x < width; x++) {
                    size_t index = z * (width * height) + y * width + x;
                    matrix3d.set(x, y, z, data_buffer[index]);
                }
            }
        }
        
        auto copy_end = std::chrono::high_resolution_clock::now();
        auto copy_duration = std::chrono::duration_cast<std::chrono::milliseconds>(copy_end - copy_start);
        
        file.close();
        
        auto end_time = std::chrono::high_resolution_clock::now();
        auto total_duration = std::chrono::duration_cast<std::chrono::milliseconds>(end_time - start_time);
        
        // minimal logging only
        
    } catch (const std::exception& e) {
        std::cerr << "❌ Errore nel caricamento file binario: " << e.what() << std::endl;
        throw;
    }
}