#include "Point3D.hh"
#include <iostream>
#include <vector>
#include <stdexcept>
#include <opencv2/opencv.hpp>
#include <fstream>
#include <array>
#include "Matrix3D.hh"
#include "PCAResult.hh"

using namespace std;

/**
 * @brief Costruttore di default di una matrice 10x10x10
 */
template <typename T>
//costruttore di default matrice 10x10
Matrix3D<T>::Matrix3D() : startRow(0), startColumn(0) 
{
    this->x = this->y = this->z = 10;

    this->data = new T **[this->x]; //allocazione memoria
    for (int i = 0; i < this->x; i++)
    {
        this->data[i] = new T *[this->y];
        for (int j = 0; j < this->y; j++)
        {
            this->data[i][j] = new T [this->z];
        }
    }
}
/**
 * @brief creare una matrice 3D a partire da una matrice 2D di OpenCV (cv::Mat) 
 * e da un valore di dimensione dim
 * 
 * @param dim Valore di dimensione
 * @param mat Matrice 2D di OpenCV
 */
template <typename T>

Matrix3D<T>::Matrix3D(size_t dim, const cv::Mat &mat) : startRow(0), startColumn(0) 
{
    this->z = dim; //imposto il numero di piani della matrice

    array<int, 4> len = getRowsCols(mat);
    int r1 = len[0];
    int r2 = len[1];
    int c1 = len[2];
    int c2 = len[3];

    cout << "r1: " << r1 << ", r2: " << r2 << ", c1: " << c1 << ", c2: " << c2 << endl;

    this->x = c2 - c1;
    this->y = r2 - r1;
    

    cout << "x: " << this->getX() << ", y: " << this->getY() << ", z: " << this->getZ() << endl;

    this->data = new T **[this->x]; //allocazione memoria
    for (int i = 0; i < this->x; i++)
    {
        this->data[i] = new T *[this->y];
        for (int j = 0; j < this->y; j++)
        {
            this->data[i][j] = new T [this->z];
        }
        
    }

    this->startRow = r1;
    this->startColumn = c1;
    createFromCV(mat, 0);
    cout<<"x2: " << this->getX() << " y2: " << this->getY() << " z2: " << this->getZ() << endl;
     
}

template <typename T> //da fare
Matrix3D<T>::Matrix3D(string binary_path, string binary_File) : startRow(0), startColumn(0)  //carica una matrice 3D da file binari
{
    // apertura del file delle dimensioni
    ifstream fileSize(binary_path + "/size_" + binary_File + ".bin", std::ios::binary);
    if (!fileSize.is_open())
    {
        cerr << "Error opening file for reading." << endl;
        return;
    }

    cout << "Reading size from file..." << endl;

    // lettura delle dimensioni della matrice
    int data[3];

    fileSize.read(reinterpret_cast<char *>(data), sizeof(data));

    this->x = data[0];
    this->y = data[1];
    this->z = data[2];

    cout << "X: " << this->x << " Y: " << this->y << " Z: " << this->z << endl;

    fileSize.close();

    this->data = new T**[this->x];
    for (int i = 0; i < this->x; i++)
    {
        this->data[i] = new T*[this->y];
        for (int j = 0; j < this->y; j++)
        {
            this->data[i][j] = new T[this->z];
        }
    }

    // apertura del file dei dati
    ifstream file(binary_path + "/data_" + binary_File + ".bin", ios::binary);
    if (!file.is_open())
    {
        cerr << "Error opening file for reading." << endl;
        return;
    }

    cout << "Reading from file..." << endl;

    // i dati vengono letti dal e memorizzati nella matrice 3D allocata in precedenza
    for (int i = 0; i < this->x; i++)
    {
        for (int j = 0; j < this->y; j++)
        {
            file.read(reinterpret_cast<char*>(this->data[i][j]), sizeof(T) * this->z);
        }
    }

    cout << "File loaded!!!" << endl;

    file.close();

    bool useUpper = 1; // zona sopra dovè c'è tubo;
    if (1)
    {
        int sizeX = 0;
        int sizeY = 3000;
        int offSetX = 0;
        int offSetY = 4700;

        if (useUpper)
        {
            sizeX = this->getX(); // 500;//this->getX();
            sizeY = this->getY(); // 1000;//this->getY();
            offSetX = 0;          // 2000;
            offSetY = 0;          // 1000;
        }

        T***data1 = new T**[this->x];
        for (int i = 0; i < this->x; i++)
        {
            data1[i] = new T*[sizeY];
            for (int j = 0; j < sizeY; j++)
            {
                data1[i][j] = new T[sizeX];
                for (int k = 0; k < sizeX; k++)
                {
                    data1[i][j][k] = this->data[i][j + offSetY][k + offSetX];
                }
            }
        }

        for (int i = 0; i < this->x; i++)
        {
            for (int j = 0; j < this->y; j++)
            {
                delete[] this->data[i][j];
            }
            delete[] this->data[i];
        }

        this->x = sizeX;
        this->y = sizeY;

        this->data = data1;
    }
}
/**
 * @brief Alloca dinamicamente una matrice 3D delle dimensioni specificate
 * 
 * @param x Dimensione X
 * @param y Dimensione Y
 * @param z Dimensione Z
 * 
 */
template <typename T>
inline Matrix3D<T>::Matrix3D(size_t x, size_t y, size_t z) : x(x), y(y), z(z), startRow(0), startColumn(0)
{
    this->data = new T **[this->x];
    for (int i = 0; i < this->x; i++)
    {
        this->data[i] = new T *[this->y];
        for (int j = 0; j < this->y; j++)
        {
            this->data[i][j] = new T[this->z];
        }
    }
}

/**
 * @brief Costruttore di copia
 * 
 * @param other Matrice 3D da copiare
 */
template <typename T>
Matrix3D<T>::Matrix3D(const Matrix3D &other) : x(other.x), y(other.y), z(other.z), startRow(other.startRow), startColumn(other.startColumn)
{
    cout << "copyng matrix constructor" << endl;
    if (this->data)
    {
        for (size_t i = 0; i < this->x; ++i)
        {
            for (size_t j = 0; j < this->y; ++j)
            {
                delete[] this->data[i][j];
            }
            delete[] this->data[i];
        }
        delete[] this->data;        
    }

    this->data = new T **[this->x];
    for (int i = 0; i < this->x; i++)
    {
        this->data[i] = new T *[this->y];
        for (int j = 0; j < this->y; j++)
        {
            this->data[i][j] = new T[this->z];
            copy(other.data[i][j], other.data[i][j] + this->z, this->data[i][j]);
        }
    }
    
}
/**
 * @brief Costruttore di spostamento
 */
template <typename T>
// costruttore di spostamento
Matrix3D<T>::Matrix3D(Matrix3D &&other) noexcept : data(std::move(other.data)), z(other.z), y(other.y), x(other.x), startRow(other.startRow), startColumn(other.startColumn)
{
    other.data = nullptr;
    other.y = other.x = other.z = other.startRow = other.startColumn = 0;
}

/**
 * @brief Operatore di assegnazione per copiare una matrice 3D.
 * 
 * Questo operatore permette di copiare i dati da una matrice 3D esistente (`other`)
 * all'oggetto corrente, eseguendo una copia profonda degli elementi della matrice. 
 * Prima di copiare i dati, la funzione dealloca la memoria eventualmente occupata 
 * dalla matrice corrente per evitare perdite di memoria.
 * 
 * Viene allocata nuova memoria per contenere gli elementi della matrice, e 
 * ogni elemento viene copiato dalla matrice `other` alla matrice corrente.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param other Riferimento costante a una matrice 3D da cui copiare i dati.
 * 
 * @return Matrix3D<T>& Riferimento all'oggetto corrente dopo l'assegnazione.
 */
template <typename T>
//operatore di assegnamento
Matrix3D<T> &Matrix3D<T>::operator=(const Matrix3D &other)
{
    cout<<"Coping Matrix operator"<<endl;
    if (this != &other)
    {
        if (this->data)
        {
            for (size_t i = 0; i < this->x; ++i)            
            {
                for (size_t j = 0; j < this->y; ++j)
                {
                    delete[] this->data[i][j];
                }
                delete[] this->data[i];
            }
            delete[] this->data;
        }

        this->x = other.x;
        this->y = other.y;
        this->z = other.z;
        this->startRow = other.startRow;
        this->startColumn = other.startColumn;
        
        this->data = new T **[this->x];
        for (int i = 0; i < this->x; i++)
        {
            data[i] = new T *[this->y];
            for (int j = 0; j < this->y; j++)
            {
                data[i][j] = new T[this->z];
                copy(other.data[i][j], other.data[i][j] + this->z, this->data[i][j]);
            }
        }   
    }
    return *this;
}
/**
 * @brief Operatore di assegnazione per spostamento della matrice 3D.
 * 
 * Questo operatore consente di trasferire le risorse da una matrice 3D (oggetto `other`) 
 * a un'altra (l'oggetto corrente) in modo efficiente, senza effettuare copie profonde. 
 * I dati e le dimensioni della matrice vengono trasferiti dall'oggetto `other` 
 * a quello corrente, mentre `other` viene resettato a uno stato vuoto.
 * 
 * Prima del trasferimento, vengono deallocati i dati esistenti nella matrice corrente 
 * (se presenti) per evitare perdite di memoria. La funzione è dichiarata `noexcept` 
 * per garantire che non sollevi eccezioni.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param other Riferimento rvalue a una matrice 3D da cui spostare i dati.
 * 
 * @return Matrix3D<T>& Riferimento all'oggetto corrente dopo l'assegnazione per spostamento.
 */
template <typename T>
Matrix3D<T> &Matrix3D<T>::operator=(Matrix3D<T> &&other) noexcept
{
    if (this != &other)
    {
        if (this->data)
        {
            for (size_t i = 0; i < this->x; ++i)
            {
                for (size_t j = 0; j < this->y; ++j)
                {
                    delete[] this->data[i][j];
                }
                delete[] this->data[i];
            }
            delete[] this->data;
        }

        this->data = other.data;
        this->x = other.x;
        this->y = other.y;
        this->z = other.z;
        this->startRow = other.startRow;
        this->startColumn = other.startColumn;

        other.data = nullptr;
        other.x = 0;
        other.y = 0;
        other.z = 0;
        other.startRow = 0;
        other.startColumn = 0;
    }
    return *this;
}

/**
 * @brief Distruttore di classe
 */
template <typename T>
// distruttore
Matrix3D<T>::~Matrix3D()
{
    for (size_t x = 0; x < this->x; ++x)
    {
        for (size_t y = 0; y < this->y; ++y)
        {
            delete[] data[x][y];
        }
        delete[] data[x];
    }
    delete[] data;
}
/**
 * @brief Ritorna la dimensione X della matrice
 * 
 * @return dimensione x della matrice
 */
template <typename T>
int Matrix3D<T>::getX() const
{
    return this->x;
}
/**
 * @brief Ritorna la dimensione Y della matrice
 * 
 * @return dimensione y della matrice
 */
template <typename T>
int Matrix3D<T>::getY() const
{
    return this->y;
}
/**
 * @brief Ritorna la dimensione Z della matrice
 * 
 * @return dimensione z della matrice
 */
template <typename T>
int Matrix3D<T>::getZ() const
{
    return this->z;
}
/**
 * @brief Operatore di accesso per gli elementi della matrice 3D.
 * 
 * Questo operatore consente di accedere a un elemento specifico della matrice 3D
 * utilizzando le coordinate (x, y, z). Se le coordinate sono fuori dai limiti
 * della matrice, viene lanciata un'eccezione `out_of_range`.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param x Coordinata x (prima dimensione) dell'elemento da accedere.
 * @param y Coordinata y (seconda dimensione) dell'elemento da accedere.
 * @param z Coordinata z (terza dimensione) dell'elemento da accedere.
 * 
 * @return T& Riferimento all'elemento della matrice 3D nelle coordinate specificate.
 * 
 * @throws std::out_of_range Se le coordinate (x, y, z) sono fuori dai limiti della matrice.
 */
template <typename T>
// operatore () per l'accesso agli elementi della matrice 3D
T& Matrix3D<T>::operator()(size_t x, size_t y, size_t z) {
    if (x >= this->x || y >= this->y || z >= this->z)
    {
        throw out_of_range("Matrix index out of range");
    }
    return data[x][y][z];
    
}
/**
 * @brief Operatore di accesso per gli elementi della matrice 3D.
 * 
 * Questo operatore consente di accedere a un elemento specifico della matrice 3D
 * utilizzando le coordinate (x, y, z). Se le coordinate sono fuori dai limiti
 * della matrice, viene lanciata un'eccezione `out_of_range`.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param x Coordinata x (prima dimensione) dell'elemento da accedere.
 * @param y Coordinata y (seconda dimensione) dell'elemento da accedere.
 * @param z Coordinata z (terza dimensione) dell'elemento da accedere.
 * 
 * @return T Elemento della matrice 3D nelle coordinate specificate.
 * 
 * @throws std::out_of_range Se le coordinate (x, y, z) sono fuori dai limiti della matrice.
 */
template <typename T>
// operatore () per l'accesso agli elementi della matrice 3D
T Matrix3D<T>::operator()(size_t x, size_t y, size_t z) const
{
    cout << "Accessing matrix at x=" << x << ", y=" << y << ", z=" << z << endl;

    if (x >= this->x || y >= this->y || z >= this->z)
    {
        cerr << "Index out of range: x=" << x << ", y=" << y << ", z=" << z << endl;
        throw out_of_range("Matrix index out of range");
    }
    return data[x][y][z];
    
}
/**
 * @brief Operatore di accesso per gli elementi della matrice 3D.
 * 
 * Questo operatore consente di assegnare un valore nella matrice 3D
 * utilizzando le coordinate (x, y, z). Se le coordinate sono fuori dai limiti
 * della matrice, viene lanciata un'eccezione `out_of_range`.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param x Coordinata x (prima dimensione) dell'elemento da accedere.
 * @param y Coordinata y (seconda dimensione) dell'elemento da accedere.
 * @param z Coordinata z (terza dimensione) dell'elemento da accedere.
 * @param v Valore
 * 
 * @throws std::out_of_range Se le coordinate (x, y, z) sono fuori dai limiti della matrice.
 */
template <typename T>
// per impostare un valore in una posizione specificata nella matrice 3D
void Matrix3D<T>::set(size_t x, size_t y, size_t z, T v)
{
    if (x >= this->x || y >= this->y || z >= this->z || x < 0 || y < 0 || z < 0)
    {
        throw out_of_range("Matrix index out of range");
    }
    data[x][y][z] = v;
}
/**
 * @brief Carica una singola immagine in una slice della matrice 3D.
 * 
 * Questa funzione carica un'immagine OpenCV (`cv::Mat`) in una specifica slice (strato) 
 * della matrice 3D, indicata dal parametro `pos`. L'immagine è composta da valori di intensità 
 * dei pixel che vengono convertiti in valori compresi tra 0 e 1 e memorizzati nella slice 
 * corrispondente della matrice.
 * 
 * Se la profondità indicata da `pos` è maggiore della dimensione della matrice 3D, 
 * viene lanciata un'eccezione `std::out_of_range`.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param image Immagine OpenCV (`cv::Mat`) che deve essere caricata nella matrice 3D.
 * @param pos Posizione della slice (strato) nella matrice 3D in cui caricare l'immagine.
 * 
 * @throws std::out_of_range Se il valore di `pos` è maggiore della profondità della matrice.
 * 
 */
template <typename T>
// per caricare una singola immagine in una delle slices della matrice 3D
void Matrix3D<T>::createFromCV(const cv::Mat &image, int pos)
{
    if (pos > this->z)
    {
        throw std::out_of_range("Matrix depth index out of range");
    }
    
    for (int row = 0; row < this->y; ++row)
    {
        for (int col = 0; col < this->x; ++col)
        {
            uchar pixelValue = image.at<uchar>(row + startRow, col + startColumn);
            float t = static_cast<float>(pixelValue);
            //cout << "Aggiungo in posizione: " << col << " " << row << " " << pos << " Value-->" << t << endl;
            data[col][row][pos] = t / 255.0;
        }
    }
    
}
/**
 * @brief Salva la matrice 3D su disco in due file binari.
 * 
 * Questa funzione salva la matrice 3D in due file binari: uno per le dimensioni 
 * della matrice e uno per i dati effettivi. Il file delle dimensioni contiene le 
 * informazioni sulle dimensioni (x, y, z) della matrice, mentre il file dei dati 
 * memorizza tutti gli elementi della matrice.
 * 
 * Il nome dei file è generato concatenando il percorso (`path`), il nome del file 
 * (`fileName`), e una specifica indicazione per le dimensioni ("size_") o i dati 
 * ("data_").
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param path Percorso in cui salvare i file binari.
 * @param fileName Nome base del file da usare per i file binari (verrà aggiunto 
 *                 "_size" o "_data" al nome).
 * 
 * @return void
 * 
 * @throws std::ios_base::failure Se non è possibile aprire i file binari per la scrittura.
 */
template <typename T>
// per salvare la matrice 3D su disco in due file binari: uno per le dimensioni della matrice e uno per i dati effettivi
void Matrix3D<T>::save(string path, string fileName)
{
    ofstream fileSize(path + "/size_" + fileName + ".bin", ios::binary);
    if (!fileSize.is_open())
    {
        cerr << "Error opening file for writing." << endl;
        return;
    }

    cout << "Writing size to file..." << endl;

    int data[3];

    data[0] = this->x;
    data[1] = this->y;
    data[2] = this->z;
    
    fileSize.write(reinterpret_cast<const char *>(data), sizeof(data));

    cout << "Size written to file Size" << endl;

    fileSize.close();

    ofstream file(path + "/data_" + fileName + ".bin", ios::binary);

    if (!file.is_open())
    {
        cerr << "Error opening file for writing." << endl;
        return;
    }

    cout << "Writing to file..." << endl;

    for (int i = 0; i < this->x; i++)
    {
        for (int j = 0; j < this->y; j++)
        {
            file.write(reinterpret_cast<const char *>(this->data[i][j]), sizeof(T) * this->z);
        }
    }

    cout << "File Saved!!!" << endl;

    file.close();
}
/**
 * @brief Stampa il contenuto della matrice 3D su console.
 * 
 * Questa funzione stampa tutti gli elementi della matrice 3D, visualizzando 
 * ciascun piano (slice) della matrice. Ogni piano viene stampato in righe 
 * e colonne corrispondenti agli assi x e y, e separato da una linea di 
 * demarcazione per facilitare la lettura.
 * 
 * Il formato della stampa è: 
 * - Ogni elemento della matrice è separato da una virgola.
 * - I piani della matrice (lungo l'asse z) sono separati da una linea 
 *   orizzontale "____________________________".
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * 
 * @return void
 */
template <typename T>
void Matrix3D<T>::print(){
    // for (int i = 0; i < this->x; i++)
    // {
    //     for (int j = 0; j < this->y; j++)
    //     {
    //         for (int k = 0; k < this->z; k++)
    //         {
    //             cout << data[i][j][k] << ", ";
    //         }
    //         cout << endl;
    //     }
    //     cout << "____________________________" << endl;
    // }
    for (int i = 0; i < this->z; i++)
    {
        for (int j = 0; j < this->y; j++)
        {
            for (int k = 0; k < this->x; k++)
            {
                cout << data[k][j][i] << ", ";
            }
            cout << endl;
        }
        cout << "____________________________" << endl;
    }
}
/**
 * @brief Stampa le dimensioni della matrice 3D su console.
 * 
 * Questa funzione stampa le dimensioni della matrice 3D, indicando il numero 
 * di elementi lungo i tre assi: x (larghezza), y (altezza), e z (profondità).
 * 
 * Il formato della stampa è: "X: [valore] Y: [valore] Z: [valore]".
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * 
 * @return void
 */
template <typename T>
inline void Matrix3D<T>::printSize()
{
    cout << "X: " << this->x << " Y: " << this->y << " Z: " << this->z << endl;
}

/**
 * @brief Verifica se una riga di una matrice OpenCV è vuota.
 * 
 * Questa funzione controlla se tutti i pixel di una determinata riga 
 * di un'immagine OpenCV (rappresentata da `cv::Mat`) sono a zero.
 * Se tutti i valori dei pixel nella riga specificata sono zero, la riga 
 * è considerata vuota, e la funzione restituisce true. Altrimenti, 
 * restituisce false.
 * 
 * @tparam T Tipo degli elementi della matrice 3D.
 * @param mat Riferimento a una matrice OpenCV (`cv::Mat`) contenente l'immagine.
 * @param row Numero della riga da controllare nella matrice `mat`.
 * 
 * @return bool Restituisce true se la riga è vuota (tutti i pixel a zero), 
 *         altrimenti false.
 */
template <typename T>
inline bool Matrix3D<T>::isEmptyRow(const cv::Mat &mat, int row)
{
    for (int j = 0; j < mat.cols; j++)
    {
        if (static_cast<float>(mat.at<uchar>(row, j)) != 0.0f)
        {
            return false;
        }
    }
    return true;
}
/**
 * @brief Verifica se una colonna di una matrice OpenCV è vuota.
 * 
 * Questa funzione controlla se tutti i pixel di una determinata colonna 
 * di un'immagine OpenCV (rappresentata da `cv::Mat`) sono a zero.
 * Se tutti i valori dei pixel nella colonna specificata sono zero, la colonna 
 * è considerata vuota, e la funzione restituisce true. Altrimenti, 
 * restituisce false.
 * 
 * @tparam T Tipo degli elementi della matrice 3D.
 * @param mat Riferimento a una matrice OpenCV (`cv::Mat`) contenente l'immagine.
 * @param col Numero della colonna da controllare nella matrice `mat`.
 * 
 * @return bool Restituisce true se la colonna è vuota (tutti i pixel a zero), 
 *         altrimenti false.
 */
template <typename T>
inline bool Matrix3D<T>::isEmptyCol(const cv::Mat &mat, int col)
{
    for (int i = 0; i < mat.rows; i++)
    {
        if (static_cast<float>(mat.at<uchar>(i, col)) != 0.0f)
        {
            return false;
        }
    }
    return true;
}

/**
 * @brief Restituisce le coordinate delle prime e ultime righe e colonne non vuote di una matrice OpenCV.
 * 
 * Questa funzione analizza una matrice OpenCV (`cv::Mat`) e restituisce le coordinate 
 * della prima e dell'ultima riga e colonna non vuote (contenenti valori diversi da zero).
 * La funzione cerca la prima riga/colonna non vuota e l'ultima, e restituisce queste coordinate 
 * in un array di 4 elementi.
 * 
 * Le coordinate sono restituite nell'ordine: {r1, r2, c1, c2}, dove:
 * - `r1` è l'indice della prima riga non vuota.
 * - `r2` è l'indice della prima riga vuota dopo l'ultima riga non vuota.
 * - `c1` è l'indice della prima colonna non vuota.
 * - `c2` è l'indice della prima colonna vuota dopo l'ultima colonna non vuota.
 * 
 * @tparam T Tipo degli elementi della matrice 3D.
 * @param mat Riferimento a una matrice OpenCV (`cv::Mat`) da analizzare.
 * 
 * @return std::array<int, 4> Array contenente le coordinate {r1, r2, c1, c2}.
 */
template <typename T>
inline std::array<int, 4> Matrix3D<T>::getRowsCols(const cv::Mat &mat)
{
    int r1 = -1, r2 = mat.rows;
    int c1 = -1, c2 = mat.cols;

    // Cerca la prima e l'ultima riga non vuota
    for (int i = 0; i < mat.rows; i++)
    {
        if (r1 == -1 && !isEmptyRow(mat, i))
        {
            r1 = i;
        }
        else if (r1 != -1 && isEmptyRow(mat, i))
        {
            r2 = i;
            break;
        }
    }

    // Cerca la prima e l'ultima colonna non vuota
    for (int j = 0; j < mat.cols; j++)
    {
        if (c1 == -1 && !isEmptyCol(mat, j))
        {
            c1 = j;
        }
        else if (c1 != -1 && isEmptyCol(mat, j))
        {
            c2 = j;
            break;
        }
    }

    // Aggiungi un controllo di sicurezza per c2 e r2
    if (c2 > mat.cols) {
        c2 = mat.cols;
    }
    if (r2 > mat.rows) {
        r2 = mat.rows;
    }

    //return {0, 3, 0, 3};
    return {r1, r2, c1, c2};
}

/**
 * @brief Restituisce un puntatore all'elemento specificato della matrice 3D.
 * 
 * Questa funzione restituisce un puntatore all'elemento della matrice 3D che si trova 
 * alla profondità `z` e alla riga `y` della matrice. Il puntatore punta al primo elemento 
 * della colonna 0 nella slice specificata. Può essere utilizzato per accedere direttamente 
 * agli elementi della matrice per operazioni più efficienti.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param z Coordinata lungo l'asse z (profondità) della matrice.
 * @param y Coordinata lungo l'asse y (altezza) della matrice.
 * 
 * @return T* Puntatore all'elemento nella matrice 3D situato alla profondità `z` e riga `y`.
 */
template <typename T>
inline T *Matrix3D<T>::ptr(int z, int y) const{
    //cout << "Prendo da qua dalla texture: " << this->data[0][y][z] << endl;
    return &this->data[0][y][z];
}

/**
 * @brief Restituisce il valore di un elemento specifico della matrice 3D.
 * 
 * Questa funzione restituisce il valore dell'elemento della matrice 3D situato 
 * alle coordinate specificate (x, y, z). Viene utilizzata per accedere in modo 
 * diretto al valore memorizzato in una posizione specifica della matrice.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param x Coordinata lungo l'asse x (larghezza) della matrice.
 * @param y Coordinata lungo l'asse y (altezza) della matrice.
 * @param z Coordinata lungo l'asse z (profondità) della matrice.
 * 
 * @return T Valore dell'elemento della matrice 3D nelle coordinate (x, y, z).
 */
template <typename T>
inline T Matrix3D<T>::getValData(int x, int y, int z) const{
    return this->data[x][y][z];
}

/**
 * @brief Inverte i valori binari in tutta la matrice 3D.
 * 
 * Questa funzione itera su tutti gli elementi della matrice 3D e inverte i valori binari,
 * cioè sostituisce ogni valore 0 con 1 e ogni valore 1 con 0. La funzione è utile per 
 * operazioni su matrici contenenti solo valori binari (0 o 1).
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D (assunto che sia compatibile con valori binari).
 * 
 * @return void
 */
template <typename T>
//inversione dei valori iterando su tutta la matrice 3D
void Matrix3D<T>::invertBinaryValues() 
{
    for (size_t x = 0; x < this->x; ++x)
    {
        for (size_t y = 0; y < this->y; ++y)
        {
            for (size_t z = 0; z < this->z; ++z)
            {
                this->data[x][y][z] = 1 - this->data[x][y][z]; 
            }  
        } 
    }
}

/**
 * @brief Restituisce un puntatore al primo elemento di uno specifico strato della matrice 3D.
 * 
 * Questa funzione restituisce un puntatore al primo elemento di uno specifico strato (slice) 
 * della matrice 3D. La matrice viene trattata come un array piatto (flatten), e lo strato 
 * selezionato viene calcolato in base all'indice `z`.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param z Indice dello strato (slice) della matrice di cui si vuole ottenere il puntatore.
 * 
 * @return T* Puntatore al primo elemento dello strato specificato.
 */
template <typename T>
// CREAZIONE DEL PUNTATORE PIATTO E FUNZIONE CHE MI RESTITUISCE IL PUNTATORE DEL PRIMO STRATO DELLA MATRICE
T* Matrix3D<T>::getSlicePtr(int z) const{
    return &this->data_flatten[z * (this->x * this->y)];
}
template <typename T>
void Matrix3D<T>::flatten() {
    this->totalElements = x * y * z;
    this->data_flatten = new T[totalElements];
    size_t index = 0;
    for (size_t k = 0; k < z; ++k) {
        for (size_t j = 0; j < y; ++j) {
            for (size_t i = 0; i < x; ++i) {
                this->data_flatten[index++] = this->data[i][j][k];
            }
        }
    }
    cout << "Flatten creato"  << endl;
}

/**
 * @brief Restituisce il valore di un elemento della matrice 3D utilizzando indici piatti.
 * 
 * Questa funzione accede a un elemento specifico della matrice 3D, utilizzando le coordinate 
 * (x_index, y_index, z_index). Gli indici vengono convertiti in un singolo indice piatto 
 * per accedere direttamente all'array piatto (flattened) che rappresenta la matrice.
 * 
 * Se uno degli indici è fuori dai limiti della matrice, viene lanciata un'eccezione 
 * `std::out_of_range`.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param x_index Coordinata lungo l'asse x (larghezza) della matrice.
 * @param y_index Coordinata lungo l'asse y (altezza) della matrice.
 * @param z_index Coordinata lungo l'asse z (profondità) della matrice.
 * 
 * @return T Valore dell'elemento della matrice 3D alle coordinate (x_index, y_index, z_index).
 * 
 * @throws std::out_of_range Se uno degli indici (x_index, y_index, z_index) è fuori dai limiti della matrice.
 */
template <typename T>
T Matrix3D<T>::index(size_t x_index, size_t y_index, size_t z_index){
    if (x_index >= this->x || y_index >= this->y || z_index >= this->z)
    {
        throw out_of_range("Matrix index out of range");
    }
    size_t flattened_index = (z_index * y + y_index) * x + x_index;
    
    return this->data_flatten[flattened_index];
}

/**
 * @brief Imposta i codici per la matrice 3D.
 * 
 * Questa funzione permette di assegnare un vettore tridimensionale di interi 
 * alla variabile `codici` della matrice 3D. Il vettore `codici` rappresenta 
 * una struttura dati separata che può contenere informazioni aggiuntive 
 * associate alla matrice 3D.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param codici Vettore tridimensionale di interi da assegnare alla variabile `codici`.
 * 
 * @return void
 */
template <typename T>
// CREAZIONE DEL PUNTATORE PIATTO E FUNZIONE CHE MI RESTITUISCE IL PUNTATORE DEL PRIMO STRATO DELLA MATRICE CON I CODICI
void Matrix3D<T>::setCodici(vector<vector<vector<int>>> codici){
    this->codici=codici;
}

/**
 * @brief Trasforma il vettore tridimensionale `codici` in un array piatto.
 * 
 * Questa funzione converte il vettore tridimensionale `codici` in un array lineare 
 * (flatten), memorizzando tutti gli elementi del vettore tridimensionale in un singolo 
 * array di interi. Ogni elemento del vettore `codici` viene copiato nell'array piatto 
 * `condiciFlatten` in ordine sequenziale.
 * 
 * Il numero totale di elementi (`totalElements`) viene calcolato come il prodotto delle 
 * dimensioni x, y, e z della matrice. Viene poi allocata memoria per l'array `condiciFlatten`.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * 
 * @return void
 */
template <typename T>
void Matrix3D<T>::codFlatten() {
    this->totalElements = x * y * z;
    this->condiciFlatten = new int[totalElements];
    size_t index = 0;
    for (size_t k = 0; k < z; ++k) {
        for (size_t j = 0; j < y; ++j) {
            for (size_t i = 0; i < x; ++i) {
                this->condiciFlatten[index++] = this->codici[i][j][k];
            }
        }
    }

}

/**
 * @brief Restituisce un puntatore al vettore piatto `codici` per un determinato piano `z` della matrice.
 * 
 * Questa funzione restituisce un puntatore all'inizio del vettore piatto `condiciFlatten` corrispondente 
 * al piano `z` specificato. Se l'indice `z` è fuori dai limiti, la funzione lo corregge automaticamente:
 * - Se `z` è inferiore a 0, viene impostato a 0.
 * - Se `z` è maggiore o uguale alla profondità della matrice, viene impostato all'ultimo piano disponibile.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param z Indice del piano `z` per il quale si vuole ottenere il puntatore.
 * 
 * @return int* Puntatore all'inizio del piano `z` nel vettore piatto `condiciFlatten`.
 */
template <typename T>
int* Matrix3D<T>::getCodiciFlatten(int z) const{
    if (z < 0) {
        z = 0;
    }
    if(z >= this->z)
        z = this->z - 1;

    // Calcola l'offset per il piano z
    return &this->condiciFlatten[z * (this->x * y)];
}


/**
 * @brief Imposta il vettore tridimensionale `flag` per la matrice 3D.
 * 
 * Questa funzione assegna un vettore tridimensionale di interi alla variabile 
 * `flag` della matrice 3D. Il vettore `flag` viene utilizzato per memorizzare 
 * informazioni specifiche associate a ciascun elemento della matrice, come ad esempio 
 * flag di stato o marcatori.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param flag Vettore tridimensionale di interi da assegnare alla variabile `flag`.
 * 
 * @return void
 */
template <typename T>
void Matrix3D<T>::setFlag(vector<vector<vector<int>>> flag){
    this->flag=flag;
}
/**
 * @brief Trasforma il vettore tridimensionale `flag` in un array piatto.
 * 
 * Questa funzione converte il vettore tridimensionale `flag` in un array lineare 
 * (flatten), memorizzando tutti gli elementi del vettore tridimensionale in un singolo 
 * array di interi. Ogni elemento del vettore `flag` viene copiato nell'array piatto 
 * `flagFlat` in ordine sequenziale.
 * 
 * Il numero totale di elementi (`totalElements`) viene calcolato come il prodotto delle 
 * dimensioni x, y, e z della matrice. La memoria per l'array `flagFlat` viene allocata 
 * e riempita con i dati presenti nel vettore `flag`.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * 
 * @return void
 */
template <typename T>
void Matrix3D<T>::flagFlatten() {
    this->totalElements = x * y * z;
    this->flagFlat = new int[totalElements];
    size_t index = 0;
    for (size_t k = 0; k < z; ++k) {
        for (size_t j = 0; j < y; ++j) {
            for (size_t i = 0; i < x; ++i) {
                this->flagFlat[index++] = this->flag[i][j][k];
            }
        }
    }

}
/**
 * @brief Restituisce un puntatore al vettore piatto `flag` per un determinato piano `z` della matrice.
 * 
 * Questa funzione restituisce un puntatore all'inizio del vettore piatto `flagFlat` corrispondente 
 * al piano `z` specificato. Se l'indice `z` è fuori dai limiti, la funzione lo corregge automaticamente:
 * - Se `z` è inferiore a 0, viene impostato a 0.
 * - Se `z` è maggiore o uguale alla profondità della matrice, viene impostato all'ultimo piano disponibile.
 * 
 * @tparam T Tipo degli elementi contenuti nella matrice 3D.
 * @param z Indice del piano `z` per il quale si vuole ottenere il puntatore.
 * 
 * @return int* Puntatore all'inizio del piano `z` nel vettore piatto `flagFlat`.
 */
template <typename T>
int* Matrix3D<T>::getflagFlatten(int z) const{
    if (z < 0) {
        z = 0;
    }
    if(z >= this->z)
        z = this->z - 1;

    // Calcola l'offset per il piano z
    return &this->flagFlat[z * (this->x * y)];
}

template class Matrix3D<int>; 
template class Matrix3D<float>; 
