#include "Matrix3DAnalysis.hh"
#include "Matrix3D.hh"
#include "Point3D.hh"
#include "libraries/dsyevj3.hh"
#include "PCAResult.hh"
#include "pixelParams.hh"
#include <algorithm>
#ifdef USE_STD
#include "configurations/configSTD.hh"
#endif
#ifdef USE_ALT1
#include "configurations/configALT1.hh"
#endif



template <typename T>
vector<vector<vector<int>>> Matrix3DAnalysis<T>::getCodici(){
    return this->codici;
}
template <typename T>
vector<vector<vector<int>>> Matrix3DAnalysis<T>::getFlag(){
    return this->flag;
}


template <typename T>
void Matrix3DAnalysis<T>::assignRec(int x, int y, int z)
{
    int c = codici[x][y][z];
    int sizeX = codici.size();
    int sizeY = codici[0].size();
    int sizeZ = codici[0][0].size();
    float vIniziale = matrix(x, y, z); //valore iniziale da confrontare con gli altri 

    for (int i = x - 1; i <= x + 1; i++){
        for (int j = y - 1; j <= y + 1; j++){
            for (int k = z - 1; k <= z + 1; k++){
                if (i >= 0 && j >= 0 && k >= 0 && i < sizeX && j < sizeY && k < sizeZ){ // controllo sui limiti della matrice 
                    if(matrix(i, j, k) < vIniziale && codici[i][j][k] == -1){ // se il valore corrente < valore iniziale e non è stato ancora assegnato 
                        float bestV = 0;
                        int bestI = 0;
                        int bestJ = 0;
                        int bestK = 0;
                        for (int i1 = i - 1; i1 <= i + 1; i1++){
                            for (int j1 = j - 1; j1 <= j + 1; j1++){
                                for (int k1 = k - 1; k1 <= k + 1; k1++){
                                    if (i1 >= 0 && j1 >= 0 && k1 >= 0 && i1 < sizeX && j1 < sizeY && k1 < sizeZ){ // controllo sui limiti della matrice con il nuovo punto 
                                        float v = matrix(i1, j1, k1);
                                        if (bestV == 0 || bestV < v){ 
                                            bestV = v;
                                            bestI = i1;
                                            bestJ = j1;
                                            bestK = k1;
                                        }
                                    }
                                }
                            }
                        }
                    if (bestV > 0 && bestI == x && bestJ == y && bestK == z){
                        codici[i][j][k] = c;
                        assignRec(i, j, k);
                    }
                }
                }
            }
        }
    }
}

/**
 * @brief Trova i massimi locali nella matrice 3D e li memorizza.
 * 
 * Questa funzione analizza la matrice 3D per identificare i punti che rappresentano dei massimi locali,
 * cioè i punti il cui valore è maggiore rispetto a quelli circostanti. I massimi locali trovati vengono
 * memorizzati in una struttura dati interna per un'analisi successiva.
 * 
 * La funzione confronta ogni elemento della matrice con i suoi vicini all'interno di un intorno specificato,
 * e se il valore corrente è il massimo rispetto a quelli vicini, viene considerato un massimo locale.
 * 
 */
template <typename T>
void Matrix3DAnalysis<T>::findLocalMaxima() {

    bool unMax = false;
    
    //creazione delle matrici codici e flag con le dimensioni della matrice originale 
    int sizez = matrix.getZ(), sizey = matrix.getY(), sizex = matrix.getX(); 
    codici.resize(sizex);
    flag.resize(sizex);
    for (int i = 0; i < sizex; i++) {
        codici[i].resize(sizey);
        flag[i].resize(sizey);
        for (int j = 0; j < sizey; j++) {
            codici[i][j].resize(sizez);
            flag[i][j].resize(sizez);
            for (int k = 0; k < sizez; k++){
                codici[i][j][k] = -1; // ogni elemento in codici impostato a -1 
                flag[i][j][k] = 0; // ogni elemento in flag impostato a 0 
            }
        }
    }

    int codice = 0;
    localMaxList.resize(sizex);

    // ricerca dei massimi locali 
    for (int i = 0; i < sizex; i++) {
        for (int j = 0; j < sizey; j++) {
            for (int k = 0; k < sizez; k++) {
                float max = matrix(i, j, k);
                bool isMax = true;

                if (max < 0.10) //0.22 0.15
                    continue;
                int viciniSx=- VICINI_MAX; //META' FATTORE MOLTIPLICATIVO DA AGGIUNGERE
                int viciniDx= VICINI_MAX;
                // in questo ciclo si itera su tutti i punti nell'intorno 3x3x3 dove gli indici l,m,n vanno da -1 a 1 quindi il ciclo esplora i vicini a distanza di un passo 
                for (int l = viciniSx; l <= viciniDx && isMax; l++) {
                    for (int m = viciniSx; m <= viciniDx && isMax; m++) {
                        for (int n = viciniSx; n <= viciniDx && isMax; n++) {
                            if (l != 0 || m != 0 || n != 0) { // escludo il punto stesso in base al quale sto facendo i controlli 
                                int newI = i + l, newJ = j + m, newK = k + n;
                                if (newI >= 0 && newI < sizex && newJ >= 0 && newJ < sizey && newK >= 0 && newK < sizez) { // controllo sui limiti della matrice 
                                    if (matrix(newI, newJ, newK) >= max) { // controllo se il punto nella nuova posizione è >= rispetto al punto protagonista 
                                        isMax = false; // se falso i cicli si interrompono grazie alla condizione && isMax 
                                    }
                                }
                            }
                        }
                    }
                }

                // assegnazione del codice al max locale 
                if (isMax) {
                    codici[i][j][k] = codice++;
                    localMaxList[i].push_back(Point3D(i, j, k, matrix(i, j, k))); // il massimo viene salvato come Point3D con le coordinate e il suo valore 
                    localMaxSize++;
                    //unMax = true;
                }
                if(unMax)
                    break;
            }
            if(unMax)
                break;
        }
        if(unMax)
            break;
    }

    cout << "Total max: " << localMaxSize << endl;

    // itero su localMacList per far partire assignRec con le coordinate dei maxLocali 
    for (int k = 0; k < sizex; k++)
        for (int i = 0; i < localMaxList[k].size(); i++) {
            int x = (int)localMaxList[k][i].getX(), y = (int)localMaxList[k][i].getY(), z = (int)localMaxList[k][i].getZ();
            int c = codici[x][y][z];
            if (c != -1)
                assignRec(x, y, z);
        }
}

/**
 * @brief Esegue un'analisi PCA (Principal Component Analysis) sulla matrice 3D per analizzare i pattern locali.
 * 
 * Questa funzione applica l'analisi delle componenti principali (PCA) su regioni selezionate della matrice 3D
 * e memorizza i risultati, come autovalori e autovettori, per estrarre caratteristiche geometriche e statistiche
 * significative dai punti dati nella matrice.
 * 
 * I risultati della PCA, come i vettori direzionali e i punteggi, vengono memorizzati nel vettore `pcaResults`
 * e utilizzati per ulteriori analisi all'interno della classe Matrix3DAnalysis.
 * 
 */
template <typename T>
void Matrix3DAnalysis<T>::pcaAnalysis() 
{
    float m_per_pixel_x=M_PER_PIXEL_X;
    float m_per_pixel_z=M_PER_PIXEL_Z;
    int sizez = matrix.getZ(), sizey = matrix.getY(), sizex = matrix.getX();
    int bestx = -1, besty = -1, bestz = -1;
    float bestv = 0;
    double ell_d[3], ell_a[3][3], ell_v[3][3], c[3], t[3];

    pcaResults.clear();
    vector<vector<vector<float>>> grads;
    grads.resize( 2*S_PCA+1);
    for(int i=0; i < 2*S_PCA+1; i++){
        grads[i].resize( 2*S_PCA+1);
        for(int j=0; j < 2*S_PCA+1; j++){
            grads[i][j].resize( 2*S_PCA+1);
        }
    }
    
    for (int k = 0; k < sizex; k++){
        for (int i = 0; i < localMaxList[k].size(); i++) {
            // itero dentro la lista dei max locali e recupero le loro coordinate 
            int x = (int)localMaxList[k][i].getX(), y = (int)localMaxList[k][i].getY(), z = (int)localMaxList[k][i].getZ();
            int co = codici[x][y][z]; // assegno il codice del max locale corrente
            if (co != -1) { // se c'è il codice del max locale procedo nell'if 
                int s = S_PCA;
                //flag[x][y][z]=1;
                //if(x==229 && y==812 && z==7) //Coordinate pixel --> x: 229 y: 812 z: 7 
                {
                    //flag[x][y][z]=1;

                    vector<float> allVal;
                    float maxGrad=0;
                    for (int x1 = x - s; x1 <= x + s; x1++) {
                        for (int y1 = y - s; y1 <= y + s; y1++) {
                            for (int z1 = z - s; z1 <= z + s; z1++) {
                                if (x1 >= 0 && y1 >= 0 && z1 >= 0 && x1 < sizex-1 && y1 < sizey-1 && z1 < sizez-1) { // controllo sui limiti della matrice 
                                    float g=pow(matrix(x1+1, y1, z1)- matrix(x1, y1, z1), 2)+
                                            pow(matrix(x1, y1+1, z1)- matrix(x1, y1, z1), 2)+
                                            pow(matrix(x1, y1, z1+1)- matrix(x1, y1, z1), 2);
                                    g=pow(g, 0.5);
                                    if(g > maxGrad) maxGrad=g;
                                    g=0;
                                    grads[x1-x+s][y1-y+s][z1-z+s]=g;
                                }
                                else{
                                    grads[x1-x+s][y1-y+s][z1-z+s]=0;
                                }
                            }
                        }
                    }
                    for (int x1 = 0; x1 <= 2*s; x1++) {
                        for (int y1 =0; y1 <= 2* s; y1++) {
                            for (int z1 = 0; z1 <= 2*s; z1++) {
                                if(maxGrad!=0)
                                    grads[x1][y1][z1]/=maxGrad;
                            }
                        }
                    }
                    float maxVal = matrix(x, y, z)*(1-grads[s][s][s]);

                    // itero su tutti i valori all'interno della sfera di raggio 30 attorno al massimo  
                    for (int x1 = x - s; x1 <= x + s; x1++) {
                        for (int y1 = y - s; y1 <= y + s; y1++) {
                            for (int z1 = z - s; z1 <= z + s; z1++) {
                                if (x1 >= 0 && y1 >= 0 && z1 >= 0 && x1 < sizex && y1 < sizey && z1 < sizez) { // controllo sui limiti della matrice 
                                    // raccolta dei valori all'interno della sfera <= del max locale 
                                    float v = matrix(x1, y1, z1)* (1-grads[x1-x+s][y1-y+s][z1-z+s]);
                                    if(v<=maxVal){
                                        allVal.push_back(v);
                                    }
                                }
                            }
                        }
                    }

                    // ordine crescente e applico un taglio per prendere solo i valori sopra la percentuale 
                    sort(allVal.begin(), allVal.end());
                    float med = allVal[(int)(allVal.size()*0.90)]; // 90% percentile

                    // applico il flag 1 ai valori sopra la soglia med, altrimenti 0 
                    for (int x1 = x - s; x1 <= x + s; x1++) {
                        for (int y1 = y - s; y1 <= y + s; y1++) {
                            for (int z1 = z - s; z1 <= z + s; z1++) {
                                if (x1 >= 0 && y1 >= 0 && z1 >= 0 && x1 < sizex && y1 < sizey && z1 < sizez) {
                                    float v = matrix(x1, y1, z1)* (1-grads[x1-x+s][y1-y+s][z1-z+s]);
                                    if(v>=med){
                                        flag[x1][y1][z1]=1;
                                    }
                                    else 
                                        flag[x1][y1][z1]=0;
                                }
                            }
                        }
                    }
                    // for (int x1 = x - s; x1 <= x + s; x1++) {
                    //     for (int y1 = y - s; y1 <= y + s; y1++) {
                    //         for (int z1 = z - s; z1 <= z + s; z1++) {
                    //             if (x1 >= 0 && y1 >= 0 && z1 >= 0 && x1 < sizex && y1 < sizey && z1 < sizez) {
                    //                 int v = 0;
                    //                 if(z1==z)
                    //                     v=1;
                    //                 flag[x1][y1][z1]=v;
                    //             }
                    //         }
                    //     }
                    // }

                }
                /////////// da rivedere questa parte /////////////////
                float scoreMax = 0;
                for(int cont=0;cont<localMaxSize;++cont){
                    // trovo lo scoreMax scorrendo la lista dei codici 
                    if(cont!=co && scoreMax<statCoppie[co][cont].score){
                        scoreMax=statCoppie[co][cont].score;
                    }
                }
                if(scoreMax==0)
                    scoreMax = 1;

                float mass = 0;
                c[0] = c[1] = c[2] = 0;
                
                float scale = SCALA_INV;
                ////////////////////////////////////////////////////////

                //if (matrix(x, y, z) > 0.3) {
                // calcolo del baricentro e della massa ponderata 
                    for (int x1 = x - s; x1 <= x + s; x1++) {
                        for (int y1 = y - s; y1 <= y + s; y1++) {
                            for (int z1 = z - s; z1 <= z + s; z1++) {
                                if (x1 >= 0 && y1 >= 0 && z1 >= 0 && x1 < sizex && y1 < sizey && z1 < sizez) {
                                    //float v = matrix(x1, y1, z1)*flag[x1][y1][z1];
                                    float v = flag[x1][y1][z1];
                                    int co1 = codici[x1][y1][z1];
                                    // ---
                                    v = pow(v,2);
                                    // --
                                    scale = 0;
                                    if (co == codici[x1][y1][z1])
                                        scale = SCALA_INV;
                                    else if (codici[x1][y1][z1] != -1)
                                        //scale = statCoppie[co][co1].score/scoreMax;//0.5;
                                        scale = SCALA_INV;
                                    if (v > 0) {
                                        // CALCOLO LE COORDINATE DEL BARICENTRO E DO CONTRIBUTO O MENO ALLA MASSA 
                                        v *= scale;
                                        c[0] += v * (x1-x);
                                        c[1] += v * (y1-y);
                                        c[2] += v * (z1-z)*m_per_pixel_z/m_per_pixel_x;
                                        mass += v;
                                    }
                                }
                            }
                        }
                    }
                    
                    // CALCOLO DEL BARICENTRO PONDERATO 
                    if (mass > 0) {
                        c[0] = c[0] / mass;
                        c[1] = c[1] / mass;
                        c[2] = c[2] / mass;
                        // MATRICE DI COVARIANZA
                        int dimCov= 3; //2*VICINI_MAX+1 sarebbe ideale ma libreria fa Jacobi in matrici 3x3
                        for (int idx1 = 1; idx1 <= dimCov; idx1++)
                            for (int idx2 = 1; idx2 <= dimCov; idx2++)
                                ell_a[idx1 - 1][idx2 - 1] = 0;
                        for (int x1 = x - s; x1 <= x + s; x1++) {
                            for (int y1 = y - s; y1 <= y + s; y1++) {
                                for (int z1 = z - s; z1 <= z + s; z1++) {
                                    if (x1 >= 0 && y1 >= 0 && z1 >= 0 && x1 < sizex && y1 < sizey && z1 < sizez) {
                                        scale = 0;
                                        int co1 = codici[x1][y1][z1];
                                        if (co == codici[x1][y1][z1])
                                            scale = SCALA_INV;
                                        else if (codici[x1][y1][z1] != -1)
                                            //scale = statCoppie[co][co1].score/scoreMax;//0.5;
                                            scale = SCALA_INV;
                                        //float v = matrix(x1, y1, z1)*flag[x1][y1][z1];
                                        float v = flag[x1][y1][z1];
                                        // ---
                                        v = pow(v,2);
                                        // --
                                        v *= scale;

                                        // SPOSTAMENTO DEL PUNTO RISPETTO AL MAX 
                                        t[0] = x1-x;
                                        t[1] = y1-y;
                                        t[2] = (z1-z)*m_per_pixel_z/m_per_pixel_x;
                                        for (int idx1 = 1; idx1 <= dimCov; idx1++)
                                            for (int idx2 = 1; idx2 <= dimCov; idx2++) {
                                                double temp = 0;
                                                temp = v * (t[idx1 - 1] - c[idx1 - 1]) * (t[idx2 - 1] - c[idx2 - 1]);
                                                ell_a[idx1 - 1][idx2 - 1] += temp / mass;
                                            }
                                    }
                                }
                            }
                        }
                        
                        // ell_d = autovalori; ell_v = autovettori; ell_a = matrice di covarianza  
                        int retc = dsyevj3(ell_a, ell_v, ell_d);
                        int best[3];
                        if (ell_d[0] >= ell_d[1] && ell_d[1] >= ell_d[2]) {
                            best[0] = 0;
                            best[1] = 1;
                            best[2] = 2;
                        }
                        if (ell_d[0] >= ell_d[2] && ell_d[2] > ell_d[1]) {
                            best[0] = 0;
                            best[1] = 2;
                            best[2] = 1;
                        }
                        if (ell_d[1] > ell_d[0] && ell_d[0] >= ell_d[2]) {
                            best[0] = 1;
                            best[1] = 0;
                            best[2] = 2;
                        }
                        if (ell_d[1] >= ell_d[2] && ell_d[2] > ell_d[0]) {
                            best[0] = 1;
                            best[1] = 2;
                            best[2] = 0;
                        }
                        if (ell_d[2] > ell_d[1] && ell_d[1] > ell_d[0]) {
                            best[0] = 2;
                            best[1] = 1;
                            best[2] = 0;
                        }
                        if (ell_d[2] > ell_d[0] && ell_d[0] >= ell_d[1]) {
                            best[0] = 2;
                            best[1] = 0;
                            best[2] = 1;
                        }
                        
                        

                        float D1 = ell_d[best[0]] / ell_d[best[2]];
                        float D1_2 = 1-(ell_d[best[1]] - ell_d[best[2]]) / (ell_d[best[0]] - ell_d[best[2]]);
                        // if (D1_2 < 0.1) {
                        //     D1_2 = 0.1;
                        // }
                        float comb = D1 / D1_2 * matrix(x, y, z);
                        if (D1 < THRESHOLD_COMB_D1) comb = 0;
                        if (D1_2 > THRESHOLD_COMB_D2) comb = 0;

                        PCAResult pcaResult;
                        
                        float soglia_D1 = THRESHOLD_D1;
                        if(D1> soglia_D1)
                            D1 = soglia_D1;
                        D1 = D1/soglia_D1;
                        // autovettore grande
                        pcaResult.autovetMax[0] = ell_v[best[0]][0];
                        pcaResult.autovetMax[1] = -ell_v[best[0]][1];
                        pcaResult.autovetMax[2] = ell_v[best[0]][2];
                        pcaResult.autovalore[0] = ell_d[best[0]];
                        
                        // autovettore medio
                        pcaResult.autovetMid[0] = ell_v[best[1]][0];
                        pcaResult.autovetMid[1] = -ell_v[best[1]][1];
                        pcaResult.autovetMid[2] = ell_v[best[1]][2];
                        pcaResult.autovalore[1] = ell_d[best[1]];

                        // autovettore piccolo
                        pcaResult.autovetMin[0] = ell_v[best[2]][0];
                        pcaResult.autovetMin[1] = -ell_v[best[2]][1];
                        pcaResult.autovetMin[2] = ell_v[best[2]][2];
                        pcaResult.autovalore[2] = ell_d[best[2]];

                        pcaResult.D1 = D1; 
                        pcaResult.D1_2 = D1_2;
                        pcaResult.score = D1*D1_2 * matrix(x, y, z);
                        
                        pcaResult.baricentro = Point3D(c[0]+x,c[1]+y,(c[2]/m_per_pixel_z*m_per_pixel_x)+z, pcaResult.score);
                        pcaResult.baricentro.setScore(D1);
                        pcaResult.maxLocale = Point3D(x, y, z, localMaxList[k][i].getValue());
                        pcaResult.maxLocale.setScore(0.0);
                        
                        Point3D direzione= Point3D(pcaResult.autovetMax[0],pcaResult.autovetMax[1],pcaResult.autovetMax[2]);
                        //direzione.normalize();
                        Point3D diff= pcaResult.maxLocale - pcaResult.baricentro;
                        float lambda= (pcaResult.maxLocale - pcaResult.baricentro)*direzione;
                        Point3D projection= Point3D(pcaResult.baricentro.x +direzione.x*lambda, pcaResult.baricentro.y +direzione.y*lambda,pcaResult.baricentro.z + direzione.z*lambda);
                        Point3D delta= projection - pcaResult.maxLocale; 
                        //pcaResult.distance_c_max= fabs((pcaResult.maxLocale - pcaResult.baricentro)*direzione);
                        pcaResult.distance_c_max= fabs(delta.norm(delta));
                        pcaResult.distance_c_max_raw= fabs(delta.norm(delta));

                        float soglia_distance = SOGLIA_DISTANZA_BARMAX; // soglia 10 da definire
                        if(pcaResult.distance_c_max > soglia_distance){ 
                            pcaResult.distance_c_max = soglia_distance;
                        }
                        pcaResult.distance_c_max = 1-(pcaResult.distance_c_max/SOGLIA_DISTANZA_BARMAX);
                        pcaResult.score = D1*D1_2* matrix(x, y, z) *pow(pcaResult.distance_c_max, EXP_BARMAX_DISTANCE);
                        pcaResult.baricentro.setValue(pcaResult.score);
                        // aggiungo la PCA al vettore PCA 
                        pcaResults.push_back(pcaResult);
                        //pcaResult.distance_c_max = pcaResult.baricentro.norm(pcaResult.maxLocale-pcaResult.baricentro);
                        // pcaResult.distance_c_max = pcaResult.baricentro.pointToLineDistance(pcaResult.maxLocale, pcaResult.baricentro, Point3D(pcaResult.autovetMax[0],pcaResult.autovetMax[1],pcaResult.autovetMax[2]));
                        // float soglia_distance = SOGLIA_DISTANZA_BARMAX; // soglia 10 da definire
                        // if(pcaResult.distance_c_max > soglia_distance){ 
                        //     pcaResult.distance_c_max = soglia_distance;
                        // }
                        // pcaResult.distance_c_max = 1-(pcaResult.distance_c_max/SOGLIA_DISTANZA_BARMAX);
                        // pcaResult.score = D1*D1_2* matrix(x, y, z) *pcaResult.distance_c_max;
                        // pcaResult.baricentro.setValue(pcaResult.score);
                        // // aggiungo la PCA al vettore PCA 
                        // pcaResults.push_back(pcaResult);

                        // {
                        //    cout << "ell-d: " << ell_d[best[0]] << " " << ell_d[best[1]] << " " << ell_d[best[2]]  << endl;
                        //    cout << "ell-v1: " << pcaResult.autovetMax[0] << " " << pcaResult.autovetMax[1] << " " << pcaResult.autovetMax[2]  << endl;
                        //    cout << "ell-v2: " << pcaResult.autovetMid[0] << " " << pcaResult.autovetMid[1] << " " << pcaResult.autovetMid[2]  << endl;
                        //    cout << "ell-v3: " << pcaResult.autovetMin[0] << " " << pcaResult.autovetMin[1] << " " << pcaResult.autovetMin[2]  << endl;
                        // }


                        if (comb > bestv && D1_2 > 0) {
                            bestv = comb;
                            bestx = x;
                            besty = y;
                            bestz = z;
                        }
                    } else {
                        c[0] = x;
                        c[1] = y;
                        c[2] = z;
                        ell_d[0] = ell_d[1] = ell_d[2] = 0.1;
                    }
                //}
            }
        }
    }
}
/**
 * @brief Analisi sulla Matrice che in ordine:
 * -Ricerca i massimi locali
 * -Ne calcola le statistiche tra essi
 * -Esegue l'analisi della PCA
 * -Una volta ottenuti i risultati della PCA, calcola le statistiche sui baricentri
 */
template <typename T>
void Matrix3DAnalysis<T>::maxAnalysis() {
    findLocalMaxima();
    statistiche();
    pcaAnalysis();
    statisticheBaricentri(pcaResults);
}
/**
 * @brief Calcola e restituisce un vettore tridimensionale basato sulle coordinate della matrice 3D.
 * 
 * Questa funzione genera un oggetto `Point3D` che rappresenta un vettore tridimensionale,
 * calcolato utilizzando le coordinate fornite (i, j, k) della matrice. Questo vettore può essere
 * utilizzato per rappresentare direzioni o posizioni nello spazio tridimensionale.
 * 
 * @tparam T Tipo di dati memorizzati nella matrice 3D (ad esempio, float, double).
 * @param i Coordinata i (x) nella matrice 3D.
 * @param j Coordinata j (y) nella matrice 3D.
 * @param k Coordinata k (z) nella matrice 3D.
 * 
 * @return Point3D Il vettore tridimensionale corrispondente alle coordinate (i, j, k).
 */
template <typename T>
Point3D Matrix3DAnalysis<T>::vettore(int i, int j, int k){
    ///////////// RIPETIZIONE RICERCA MAX LOCALI //////////////////
    int sizeZ = matrix.getZ(), sizeY = matrix.getY(), sizeX = matrix.getX();
    float bestV = 0;
    int bestI = 0;
    int bestJ = 0;
    int bestK = 0;
    for (int i1 = i - 1; i1 <= i + 1; i1++){
        for (int j1 = j - 1; j1 <= j + 1; j1++){
            for (int k1 = k - 1; k1 <= k + 1; k1++){
                if (i1 >= 0 && j1 >= 0 && k1 >= 0 && i1 < sizeX && j1 < sizeY && k1 < sizeZ){
                    float v = matrix(i1, j1, k1);
                    //cout << "   i: " << i1 << " j: " << j1 << " k: " << k1 << " matrix= " << matrix(i1,j1,k1) << endl;
                    if (bestV == 0 || bestV < v){
                        bestV = v;
                        bestI = i1;
                        bestJ = j1;
                        bestK = k1;
                    }
                }
            }
        }
    }
    //////////////////////////////////////////////////////////////////

    // CONFRONTO TRA PUNTO PIU ALTO NELL'INTORNO CON IL PUNTO CENTRALE 
    float dx = bestI-i;
    float dy = bestJ-j;
    float dz = bestK-k;
    
    // NORMALIZZAZIONE DEL VETTORE 
    float norm = sqrt(dx * dx + dy * dy + dz * dz); 
    //cout << "dx: " << dx << " dy: " << dy << " dz: " << dz << " norm: " << norm << endl; 
    Point3D point;
    if(norm > 0)
        point = Point3D(dx/norm,dy/norm, dz/norm);  
    else  point = Point3D(0, 0, 0);

    return point;
}

/**
 * @brief Calcola statistiche sulle relazioni tra i punti 3D della matrice.
 * 
 * Questa funzione calcola le statistiche di coppia tra punti 3D in una matrice triangolare utilizzando la lista delle posizioni
 * fornite e le memorizza in una matrice di statistiche. Ogni coppia di punti può essere collegata da linee basate 
 * su valori soglia. Logicamente i punti forniti sono vertici di un grafo e le linee i loro archi
 * 
 * @tparam T Tipo di dati usato nella matrice 3D.
 * @param isBaricentri Se true, calcola le statistiche utilizzando i baricentri. Se false, usa le posizioni locali massime.
 * @param posList Lista delle posizioni (locali massime o baricentri) per le quali calcolare le statistiche.
 * @param statCoppie Matrice di statistiche di coppia tra i punti.
 * @param lineeStat Vettore di linee statistiche risultanti dalle coppie con soglie di score e conteggio sufficienti.
 */
template <typename T>
void Matrix3DAnalysis<T>::calcolaStatistiche(bool isBaricentri, const std::vector<Point3D>& posList, std::vector<std::vector<Stats>>& statCoppie, std::vector<Line>& lineeStat) {
    int sizez = matrix.getZ(), sizey = matrix.getY(), sizex = matrix.getX();
    int maxSize = posList.size();
    // Reset delle statistiche
    statCoppie.resize(maxSize);
    for(int i = 0; i < maxSize; i++) {
        statCoppie[i].resize(maxSize);
        for(int j = 0; j < maxSize; j++) {
            statCoppie[i][j].conteggio = 0;
            statCoppie[i][j].angolo = 0;
            statCoppie[i][j].valMax = 0;
            statCoppie[i][j].valMedio = 0;
        }
    }

    // Calcolo delle statistiche
    // raccolgo valore massimo locale e punto corrente
    for (int i = 0; i < sizex; i++) {        
        for (int j = 0; j < sizey; j++) {
            for (int k = 0; k < sizez; k++) {
                int codice = codici[i][j][k];
                if(codice != -1) {
                    int x = posList[codice].x;
                    int y = posList[codice].y;
                    int z = posList[codice].z;
                    float valore_max = matrix(x, y, z); // valori massimi locali
                    float valore = matrix(i, j, k);     // valore punto corrente

                    // confronto con i vicini
                    for (int l = -1; l <= 1; l++) {
                        for (int m = -1; m <= 1; m++) {
                            for (int n = -1; n <= 1; n++) {
                                if (abs(l) + abs(m) + abs(n) == 1) {
                                    int newI = i + l, newJ = j + m, newK = k + n; // vicini nell'intorno di 1
                                    if (newI >= 0 && newI < sizex && newJ >= 0 && newJ < sizey && newK >= 0 && newK < sizez) {
                                        float valore_new = matrix(newI, newJ, newK);
                                        int codice_new = codici[newI][newJ][newK];
                                        if (codice_new != -1 && codice != codice_new) { // controllo se il vicino appartiene ad un altro massimo tramite il codice 
                                            Point3D p = vettore(i, j, k); // vettore direzione associato al massimo
                                            Point3D p_new = vettore(newI, newJ, newK); // vettore direzione associato al punto vicino al massimo
                                            // prodotto scalare tra i due vettori direzione, se allineati valore vicino a 1, perpendicolare 0, opposti negativo
                                            p.normalize();
                                            p_new.normalize();
                                            float prod_scal = p * p_new;
                                            float w = (valore_new + valore) / 2; // valore medio tra punto corrent e punto vicino
                                            statCoppie[codice][codice_new].angolo += prod_scal * w;
                                            statCoppie[codice][codice_new].pesoAngolo += w;
                                            statCoppie[codice][codice_new].conteggio++;
                                            float r = valore_new / valore_max;
                                            if(r > statCoppie[codice][codice_new].valMax)
                                                statCoppie[codice][codice_new].valMax = r;
                                            statCoppie[codice][codice_new].valMedio += r;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    // Fase finale di aggregazione e calcolo degli score
    for(int i = 0; i < maxSize; i++) {
        float maxConteggio = 0;
        // conteggio indica quante volte i due massimi locali sono stati trovati vicini
        for(int j = 0; j < maxSize; j++) {
            if(maxConteggio < statCoppie[i][j].conteggio)
                maxConteggio = statCoppie[i][j].conteggio;
        }
        for(int j = i + 1; j < maxSize; j++) {
            if(statCoppie[i][j].conteggio > 0) {
                float a = statCoppie[i][j].valMax;
                float b = statCoppie[j][i].valMax;
                float am = statCoppie[i][j].valMedio / statCoppie[i][j].conteggio;
                float bm = statCoppie[j][i].valMedio / statCoppie[j][i].conteggio;
                if(b < a)
                    a = b;
                if(bm < am)
                    am = bm;
                statCoppie[i][j].valMax = a;
                statCoppie[j][i].valMax = a;
                statCoppie[i][j].valMedio = am;
                statCoppie[j][i].valMedio = am;
                float ang = statCoppie[i][j].angolo / statCoppie[i][j].pesoAngolo;
                if(ang < 0)
                    ang = 0;
                float score = am; // pow(ang, 1);
                statCoppie[j][i].score = score;
            }
        }
    }

    // Filtraggio delle linee
    float sogliaCont = 0.1;
    float sogliaScore = 0.05;
    for(int i = 0; i < maxSize; i++) {
        float maxConteggio = 0;
        float maxScore = 0;
        for(int j = 0; j < maxSize; j++) {
            if(maxConteggio < statCoppie[i][j].conteggio)
                maxConteggio = statCoppie[i][j].conteggio;
            if(maxScore < statCoppie[i][j].score)
                maxScore = statCoppie[i][j].score;
        }
        for(int j = 0; j < maxSize; j++) {
            if(statCoppie[i][j].conteggio / maxConteggio >= sogliaCont && statCoppie[i][j].score / maxScore >= sogliaScore) {
                /**
                 * Se isBaricentri è true, sono nel caso in cui l'analisi è effettuata sui baricentri, ho dunque a disposizione
                 * un vettore di PCAResults su cui poter effettuare delle analisi.
                 */
                if(isBaricentri){
                    float angolo1;
                    float angolo2;
                    float angolo;

                    Point3D diff = pcaResults[i].baricentro-pcaResults[j].baricentro;
                    diff.normalize();
                    Point3D pointAvutovet = Point3D(pcaResults[i].autovetMax[0],pcaResults[i].autovetMax[1],pcaResults[i].autovetMax[2]);
                    angolo1 = diff*pointAvutovet;
                    Point3D pointAvutovet2 = Point3D(pcaResults[j].autovetMax[0],pcaResults[j].autovetMax[1],pcaResults[j].autovetMax[2]);
                    angolo2 = diff*pointAvutovet2;

                    angolo = fabs(angolo1)*fabs(angolo2);
                    //cout << "angolo " << angolo << endl;
                    float D1_i = pcaResults[i].D1 * 5;
                    if(D1_i > 1)  // molto piu tolleranti (5 volte) rispetto all'allungamento del primo autovalore
                        D1_i = 1;
                    float scoreNodo_i = D1_i*pcaResults[i].D1_2 *matrix((int)pcaResults[i].maxLocale.x,(int)pcaResults[i].maxLocale.y,(int)pcaResults[i].maxLocale.z);
                    
                    float D1_j = pcaResults[j].D1 * 5;
                    if(D1_j > 1)  // molto piu tolleranti (5 volte) rispetto all'allungamento del primo autovalore
                        D1_j = 1;
                    
                    // D1 media dell'estensione del primo
                    // D1_2 quanto è 1D/2D 
                    float D1=(D1_j+D1_i)/2;  
                    float D1_2=(pcaResults[j].D1_2+pcaResults[i].D1_2)/2;
                    float d = D1*D1_2; 
                    float m = (matrix((int)pcaResults[j].maxLocale.x, (int)pcaResults[j].maxLocale.y, (int)pcaResults[j].maxLocale.z)+matrix((int)pcaResults[i].maxLocale.x,(int)pcaResults[i].maxLocale.y,(int)pcaResults[i].maxLocale.z))/2;

                    // .score -> quanto le aree peggiorano quando si toccano rispetto ai massimi locali (vecchio delta)
                    // angolo -> lavora sulla vera direction in base agli autovetti max e al vettore che unisce baricentri (vecchio direction)
                    // m -> media dei valori dell'intensità della coppia
                    // d -> forma della PCA
                    statCoppie[i][j].score = pow(statCoppie[i][j].score, EXP_SCORE) * pow(angolo, EXP_ANGOLO) * pow(d, EXP_D); // * pow(m,1)
                    Line t(posList[i], posList[j], statCoppie[i][j].score);//*angolo
                    lineeStat.push_back(t);
                    if (statCoppie[i][j].score>0 && i == 1113 || j == 1113)
                    {
                        cout << "i: " << i << " j: " << j << " score: " << statCoppie[i][j].score << endl; 
                    }
                }
                else{
                    Line t(posList[i], posList[j], statCoppie[i][j].score);
                    lineeStat.push_back(t);
                }
            }

            else 
            {
                if (isBaricentri) 
                {
                    //Il valore .score è altrimenti già inizializzato dai calcoli precedenti
                    statCoppie[i][j].score = 0;
                }
            }
        }
    }
}



/**
 * @brief Calcola le statistiche per i massimi locali della matrice.
 * 
 * Questa funzione popola la lista delle posizioni locali massime e chiama la funzione 
 * `calcolaStatistiche` per calcolare le statistiche relative a questi massimi locali.
 * Logicamente crea un grafo che ha per vertici i massimi locali.
 */
template <typename T>
void Matrix3DAnalysis<T>::statistiche() {
    int sizex = matrix.getX();  // Ottieni la dimensione lungo l'asse X
    posMaxLocal.resize(localMaxSize);  // Ridimensiona il vettore dei massimi locali

    // Popola posMaxLocal con i massimi locali della matrice
    for (int i = 0; i < sizex; i++) {
        for (int l = 0; l < localMaxList[i].size(); l++) {
            float y = localMaxList[i][l].getY();
            float z = localMaxList[i][l].getZ();
            int codice = codici[i][y][z];
            posMaxLocal[codice].x = i;
            posMaxLocal[codice].y = y;
            posMaxLocal[codice].z = z;
        }
    }

    // Calcola le statistiche per i massimi locali
    calcolaStatistiche(false, posMaxLocal, statCoppie, lineeStat);
}


/**
 * @brief Calcola le statistiche utilizzando i baricentri.
 * 
 * Questa funzione popola la lista delle posizioni dei baricentri dai risultati della PCA 
 * e chiama la funzione `calcolaStatistiche` per calcolare le statistiche relative ai baricentri.
 * Logicamente crea un grafo che ha per vertici i baricentri
 * 
 * @param pcaResults Risultati dell'analisi PCA, contenenti i baricentri da analizzare.
 */
template <typename T>
void Matrix3DAnalysis<T>::statisticheBaricentri(const std::vector<PCAResult>& pcaResults) {
    int barycenterSize = pcaResults.size();  // Ottieni il numero di baricentri
    posBarycenter.resize(barycenterSize);  // Ridimensiona il vettore dei baricentri

    // Popola posBarycenter con i baricentri dai risultati della PCA
    for (int i = 0; i < barycenterSize; i++) {
        posBarycenter[i] = pcaResults[i].baricentro;
    }

    // Calcola le statistiche per i baricentri
    calcolaStatistiche(true, posBarycenter, statCoppieBar, lineeStatBarycenter);
}

/**
 * @brief Restituisce la lista di massimi locali contenuta nell'oggetto Matrix3DAnalysis
 */
template <typename T>
const vector<vector<Point3D>>& Matrix3DAnalysis<T>::getLocalMaxList() const {
    return localMaxList;
}
/**
 * @brief Restituisce il vettore contenente i risultati della PCA
 * 
 * Questa funzione restituisce il vettore contente un riferimento
 * costante al vettore dei risultati dell'analisi della componente principale (PCA)
 * dell'oggetto Matrix3DAnalysis.
 * 
 * @returns vector<PCAResult>& riferimento al vettore contenente i risultati della pca.
 */
template <typename T>
const vector<PCAResult>& Matrix3DAnalysis<T>::getPCAResults() const {
    return pcaResults;
}

/**
 * @brief Ritorna gli archi dei massimi
 * 
 * Questa funzione restituisce un riferimento costante a un vettore di oggetti Line,
 * che rappresentano gli archi che collegano i massimi all'interno dell'analisi 3D della matrice.
 * 
 * @tparam T Il tipo di dati utilizzato nella matrice 3D.
 * @return const vector<Line>& Un riferimento costante al vettore degli archi tra i massimi.
 */
template <typename T>
const vector<Line>& Matrix3DAnalysis<T>::getStats() const {
    return lineeStat;
}

/**
 * @brief Ritorna gli archi dei baricentri
 * 
 * Questa funzione restituisce un riferimento costante a un vettore di oggetti Line,
 * che rappresentano gli archi che collegano i baricentri all'interno dell'analisi 3D della matrice.
 * 
 * @tparam T Il tipo di dati utilizzato nella matrice 3D.
 * @return const vector<Line>& Un riferimento costante al vettore degli archi tra i baricentri.
 */
template <typename T>
const vector<Line>& Matrix3DAnalysis<T>::getStatsBarycenter() const {
    return lineeStatBarycenter;
}
/**
 * @brief Ritorna le statistiche delle coppie di massimi
 * 
 * Questa funzione restituisce un riferimento costante a un vettore di vettori di oggetti Stats,
 * che rappresentano le statistiche delle coppie di massimi all'interno dell'analisi 3D della matrice.
 * 
 * @tparam T Il tipo di dati utilizzato nella matrice 3D.
 * @return const vector<vector<Stats>>& Un riferimento costante al vettore di vettori delle statistiche delle coppie di elementi.
 */
template <typename T>
const vector<vector<Stats>>& Matrix3DAnalysis<T>::getStatsCoppie() const {
    return statCoppie;
}
/**
 * @brief Ritorna le statistiche delle coppie di baricentri
 * 
 * Questa funzione restituisce un riferimento costante a un vettore di vettori di oggetti Stats,
 * che rappresentano le statistiche delle coppie di baricentri all'interno dell'analisi 3D della matrice.
 * 
 * @tparam T Il tipo di dati utilizzato nella matrice 3D.
 * @return const vector<vector<Stats>>& Un riferimento costante al vettore di vettori delle statistiche delle coppie di elementi.
 */
template <typename T>
const vector<vector<Stats>>& Matrix3DAnalysis<T>::getStatsCoppieBar() const{
    return statCoppieBar;
}

template class Matrix3DAnalysis<float>;
