#include "MinHeap.hh"
#include <algorithm>
#include <unordered_map>

using namespace std;
/**
 * @brief Restituisce l'indice del genitore di un nodo
 *
 * @param n L'indice del nodo di cui trovare il genitore.
 * @return long L'indice del genitore, o -1 se il nodo è la radice.
 */
long MinHeap::parent(long n)
{
    if (n == 0)
        return -1;
    return (n - 1) / 2;
}

/**
 * @brief Restituisce l'indice del figlio sinistro di un nodo
 *
 * @param n L'indice del nodo di cui trovare il figlio sinistro.
 * @return long L'indice del figlio sinistro, o -1 se non esiste.
 */
long MinHeap::leftChild(long n)
{
    if (2 * n + 1 >= heap_size)
        return -1;
    return 2 * n + 1;
}
/**
 * @brief Restituisce l'indice del figlio destro di un nodo
 *
 * @param n L'indice del nodo di cui trovare il figlio destro.
 * @return long L'indice del figlio destro, o -1 se non esiste.
 */
long MinHeap::rightChild(long n)
{
    if (2 * n + 2 >= heap_size)
        return -1;
    return 2 * n + 2;
}

/**
 * @brief Verifica se un nodo è una foglia
 *
 * @param n L'indice del nodo da verificare.
 * @return long 1 se il nodo è una foglia, 0 altrimenti.
 */
long MinHeap::is_leaf(long n)
{
    return (leftChild(n == -1 && rightChild(n == -1))); // ridondante
}

/**
 * @brief Costruttore di default della classe MinHeap
 */
MinHeap::MinHeap() {}

/**
 * @brief Costruttore della classe MinHeap con capacità specificata
 *
 * @param capacity La capacità massima dello heap.
 */
MinHeap::MinHeap(long capacity) : MAX_SIZE(capacity), heap_size(0)
{
    heap = new float[capacity];
    id2Heap = new long[capacity];
    heap2Id = new long[capacity];
    reset();
}

/**
 * @brief Inserisce un nuovo elemento nell'heap
 *
 * @param id L'identificatore dell'elemento.
 * @param key Il valore dell'elemento.
 */
void MinHeap::insert(long id, float key)
{
    if (heap_size >= MAX_SIZE)
    {
        std::cout << "Heap pieno" << std::endl;
        return;
    }

    // Aggiungi il nuovo elemento alla fine dell'heap
    long i = heap_size;
    heap_size++;

    // Mappa l'ID alla posizione corrente
    id2Heap[id] = i;
    heap2Id[i] = id;
    heap[i] = key;

    // Risali l'heap per mantenere la proprietà del min-heap
    while (i != 0)
    {
        if (heap[parent(i)] <= heap[i])
        {
            return; // La proprietà del min-heap è già mantenuta
        }

        // Scambia l'elemento con il genitore
        swap(i, parent(i));
        i = parent(i);
    }
}

/**
 * @brief Scambia due elementi nello heap
 *
 * @param i L'indice del primo elemento.
 * @param j L'indice del secondo elemento.
 */
void MinHeap::swap(long i, long j)
{
    // std::cout << "Prima dello swap: i = " << i << ", j = " << j << std::endl;
    // std::cout << "heap2Id[i] = " << heap2Id[i] << ", heap2Id[j] = " << heap2Id[j] << std::endl;

    // Scambia i valori nel heap
    std::swap(heap[i], heap[j]);

    // Scambia gli ID associati alle posizioni nell'heap
    long id1 = heap2Id[i];
    long id2 = heap2Id[j];

    // std::cout << "Swapping id1 = " << id1 << ", id2 = " << id2 << std::endl;

    if (id1 != -1 && id2 != -1)
    {
        std::swap(heap2Id[i], heap2Id[j]);

        // Aggiorna id2Heap per riflettere il nuovo mapping
        id2Heap[id1] = j;
        id2Heap[id2] = i;
    }

    // std::cout << "Dopo lo swap: heap2Id[i] = " << heap2Id[i] << ", heap2Id[j] = " << heap2Id[j] << std::endl;
    // std::cout << "Dopo lo swap: heap2Id[i] = " << heap2Id[i] << ", heap2Id[j] = " << heap2Id[j] << std::endl;
}

/**
 * @brief Resetta lo heap
 */
void MinHeap::reset()
{
    heap_size = 0;
    for (long i = 0; i < MAX_SIZE; i++)
    {
        heap[i] = INF;
        id2Heap[i] = -1;
        heap2Id[i] = -1;
    }
}

/**
 * @brief Verifica se un identificatore è presente nello heap
 *
 * @param id L'identificatore da verificare.
 * @return bool True se l'identificatore è presente, false altrimenti.
 */
bool MinHeap::containsID(long id) const
{
    // cout<<"id2Heap: "<< id2Heap[id]<<endl;
    return -1 != id2Heap[id];
}

/**
 * @brief Stampa lo heap
 */
void MinHeap::print()
{
    // Stampa le chiavi (valori) dell'heap
    std::cout << "Heap (valori ordinati in base alla proprietà del MinHeap):" << std::endl;
    for (long i = 0; i < heap_size; i++)
    {
        std::cout << "Posizione " << i << ": Valore = " << heap[i] << std::endl;
    }
    std::cout << std::endl;

    // Stampa la mappatura id -> posizione nell'heap
    std::cout << "Mappa id -> posizione nell'heap (id2Heap):" << std::endl;
    for (long i = 0; i < MAX_SIZE; i++)
    {
        if (id2Heap[i] != -1)
            std::cout << "ID " << i << ": Posizione nell'heap = " << id2Heap[i] << std::endl;
    }
    std::cout << std::endl;

    // Stampa la mappatura posizione nell'heap -> id
    std::cout << "Mappa posizione nell'heap -> id (heap2Id):" << std::endl;
    for (long i = 0; i < heap_size; i++)
    {
        std::cout << "Posizione " << i << ": ID = " << heap2Id[i] << std::endl;
    }
    std::cout << std::endl;
}

/**
 * @brief Ripristina la proprietà del min-heap spostando l'elemento all'indice specificato verso il basso.
 *
 * Questa funzione viene chiamata quando un elemento è stato spostato nella radice (o in un'altra posizione)
 * e potrebbe violare la proprietà del min-heap. La funzione controlla i figli del nodo e lo scambia con
 * il figlio più piccolo se necessario, continuando il processo verso il basso fino a ripristinare la
 * struttura dell'heap.
 *
 * @param index L'indice del nodo da cui iniziare il processo di heapify verso il basso.
 */
void MinHeap::heapifyDown(long index)
{
    long left = leftChild(index);
    long right = rightChild(index);
    long smallest = index;

    // Trova il figlio più piccolo
    if (left != -1 && heap[left] < heap[smallest])
    {
        smallest = left;
    }

    if (right != -1 && heap[right] < heap[smallest])
    {
        smallest = right;
    }

    // Se il figlio più piccolo è minore del nodo corrente, scambia
    if (smallest != index)
    {
        swap(index, smallest);
        heapifyDown(smallest); // Continua il processo di heapify verso il basso
    }
}

/**
 * @brief Diminuisce la chiave di un nodo nello heap
 *
 * @param indice_nodo L'indice del nodo di cui diminuire la chiave.
 * @param key Il nuovo valore della chiave.
 */
void MinHeap::decreaseKey(long indice_nodo, float key)
{
    // Controlla se l'ID è valido
    if (id2Heap[indice_nodo] < 0 || id2Heap[indice_nodo] >= heap_size)
    {
        printf("MinHeap::decreaseKey: Nodo non esistente\n");
        return;
    }

    // Controlla se la nuova chiave è effettivamente più piccola
    if (heap[id2Heap[indice_nodo]] < key)
    {
        printf("MinHeap::decreaseKey: la chiave non è più piccola!\n");
        return;
    }

    // Aggiorna la chiave
    heap[id2Heap[indice_nodo]] = key;

    long i = id2Heap[indice_nodo];

    // Risalita (bubble-up)
    while (i != 0)
    {
        if (heap[parent(i)] <= heap[i])
        {
            return;
        }

        swap(i, parent(i)); // Usa la nuova funzione di swap
        i = parent(i);
    }
}

/**
 * @brief Estrae l'elemento minimo dall'heap
 *
 * @return long L'identificatore dell'elemento minimo, o -1 se l'heap è vuoto.
 */
long MinHeap::extractMinID()
{
    if (heap_size <= 0)
    {
        std::cout << "Errore: heap vuoto" << std::endl;
        return -1;
    }

    // Prendi il minimo (che è nella radice)
    long minId = heap2Id[0];

    // Sposta l'ultimo elemento nella radice
    swap(0, heap_size - 1);

    // Riduci la dimensione dell'heap
    heap_size--;

    // Invalida gli array per l'ID estratto
    id2Heap[minId] = -1;     // L'ID estratto non esiste più
    heap2Id[heap_size] = -1; // L'ultimo elemento è stato spostato in radice, quindi lo invalidiamo

    // Chiama heapifyDown per ripristinare l'ordine
    heapifyDown(0);

    return minId;
}

/**
 * @brief Verifica se l'heap è vuoto
 *
 * @return bool True se l'heap è vuoto, false altrimenti.
 */
bool MinHeap::isEmpty()
{
    return heap_size == 0;
}

/**
 * @brief Distruttore della classe MinHeap
 */
MinHeap::~MinHeap()
{
    delete[] heap;
    delete[] id2Heap;
    delete[] heap2Id;
}
