#ifndef POINT3D_H
#define POINT3D_H 

#include <math.h>
#include <iostream> 
#include <vector>
#include <stdexcept>
#include <fstream>
#include <array>

using namespace std; 

class Point3D
{
public:
    float x, y, z;
    float value; 
    float score; 
    float dir[3];
    Point3D(); 
    Point3D(float x, float y, float z); 
    Point3D(float x, float y, float z, float value); 
    Point3D(float x, float y, float z, float value, float distance, float score, float dirx, float diry, float dirz); 
    Point3D(const Point3D &other); 
    Point3D &operator=(const Point3D &other);
    bool operator==(const Point3D &other) const;  
    Point3D operator-(const Point3D &other) const; 
    Point3D operator+(const Point3D &other) const; 
    void operator/=(float norm); 
    float operator*(const Point3D &other) const; 
    Point3D operator*(float scalar) const;

    float getX() const;
    float getY() const;
    float getZ() const;
    float getValue() const;
    float getScore() const;
    float* getDir();
    const float* getDire() const;
    void normalize(); 
    Point3D cross(const Point3D& P, const Point3D& v);

    //float prodScalare(const Point3D &V1, const Point3D &V2);
    double pointToLineDistance(const Point3D& P, const Point3D& A, const Point3D& v);
    float norm(const Point3D &P); 
    float norm();
    float distance(const Point3D &P1, const Point3D &P2);
        // Funzione inline per calcolare il baricentro
    static inline Point3D calculateCentroid(const std::vector<Point3D>& points);

    // Funzione inline per calcolare la matrice di covarianza
    static inline std::array<std::array<float, 3>, 3> calculateCovarianceMatrix(const std::vector<Point3D>& points, const Point3D& centroid);

    // Funzione inline per calcolare l'autovettore principale
    static inline std::array<float, 3> powerIteration(const std::array<std::array<float, 3>, 3>& matrix, int iterations = 1000, float tolerance = 1e-6f);

    // Funzione inline per calcolare il segmento proiettato sulla retta di regressione
    static inline std::pair<Point3D, Point3D> calculateRegressionSegment(const std::vector<Point3D>& points);
    Point3D projectOntoPlane(const Point3D& P, const float autovetMid[3], const float autovetMin[3], const Point3D& Q);
    float distancePointToPlane(const float autovet1[3], const float autovet2[3], const Point3D& p, const Point3D& referencePoint);
    Point3D projection(const Point3D& p, const Point3D& d);

    void setScore(float s); 
    void setValue(float v); 
    void setDir(float x, float y, float z); 

    void print() const; 
};
/**
 * @brief Costruttore di default di Point 3D senza parametri
 *  Questo costruttore inizializza le coordinate geometriche del punto a -1, inoltre
 *  il campo 'value' è inizializzato a -1 come valore predefinito.
 * 
 * 
 * @return void
 */
inline Point3D::Point3D() : x(-1), y(-1), z(-1), value(-1.0) {}
/**
 * @brief Costruttore della classe Point3D che inizializza le coordinate tridimensionali.
 * 
 * Questo costruttore inizializza un oggetto `Point3D` con i valori delle coordinate 
 * x, y e z fornite dall'utente. Inoltre, il campo `value` viene inizializzato a -1 
 * come valore predefinito.
 * 
 * @param x Coordinata x del punto nello spazio tridimensionale.
 * @param y Coordinata y del punto nello spazio tridimensionale.
 * @param z Coordinata z del punto nello spazio tridimensionale.
 * 
 */
inline Point3D::Point3D(float x, float y, float z) : x(x), y(y), z(z), value(-1) {}
/**
 * @brief Costruttore della classe Point3D che inizializza le coordinate tridimensionali.
 * 
 * Questo costruttore inizializza un oggetto `Point3D` con i valori delle coordinate 
 * x, y e z fornite dall'utente. Inoltre, il campo `value` viene inizializzato con il valore
 * fornito dall'utente.
 * 
 * @param x Coordinata x del punto nello spazio tridimensionale.
 * @param y Coordinata y del punto nello spazio tridimensionale.
 * @param z Coordinata z del punto nello spazio tridimensionale.
 * @param v Valore di intensità del punto
 * 
 */
inline Point3D::Point3D(float x, float y, float z, float v) : x(x), y(y), z(z), value(v) {}
/**
 * @brief Costruttore della classe Point3D che inizializza le coordinate tridimensionali e altri attributi.
 * 
 * Questo costruttore inizializza un oggetto `Point3D` con i valori delle coordinate 
 * x, y e z, insieme ad altri attributi come `value`, `distance`, `score` e la direzione 
 * specificata dai componenti `dirx`, `diry` e `dirz`. Il vettore direzionale `dir` viene 
 * popolato con i valori forniti per rappresentare la direzione del punto nello spazio tridimensionale.
 * 
 * @param x Coordinata x del punto nello spazio tridimensionale.
 * @param y Coordinata y del punto nello spazio tridimensionale.
 * @param z Coordinata z del punto nello spazio tridimensionale.
 * @param value Valore associato al punto (può rappresentare una caratteristica specifica).
 * @param distance Distanza del punto (può rappresentare una distanza da un'origine o da un punto di riferimento).
 * @param score Punteggio associato al punto (può rappresentare una misura della qualità del punto).
 * @param dirx Componente x del vettore direzionale.
 * @param diry Componente y del vettore direzionale.
 * @param dirz Componente z del vettore direzionale.
 * 
 * @return void
 */
inline Point3D::Point3D(float x, float y, float z, float value, float distance, float score, float dirx, float diry, float dirz)
    : x(x), y(y), z(z), value(value), score(score)
{
    dir[0] = dirx;
    dir[1] = diry;
    dir[2] = dirz;
}
/**
 * @brief Costruttore di copia della classe Point3D.
 * 
 * Questo costruttore crea un nuovo oggetto `Point3D` copiando i valori delle coordinate 
 * x, y e z, e il valore `value` da un altro oggetto `Point3D`. Viene utilizzato per creare 
 * una copia di un punto già esistente.
 * 
 * @param other Riferimento all'oggetto `Point3D` da cui copiare i valori.
 * 
 * @return void
 */
inline Point3D::Point3D(const Point3D &other) : x(other.x), y(other.y), z(other.z), value(other.value) {}

/**
 * @brief Operatore di assegnazione per la classe Point3D.
 * 
 * Questo operatore permette di assegnare i valori di un altro oggetto `Point3D` 
 * all'oggetto corrente. Se l'oggetto assegnato è diverso dall'oggetto corrente, 
 * i valori delle coordinate x, y, z, `value`, `score`, e il vettore direzionale `dir` 
 * vengono copiati dall'oggetto `other` all'oggetto corrente.
 * 
 * @param other Riferimento costante all'oggetto `Point3D` da cui copiare i valori.
 * 
 * @return Point3D& Riferimento all'oggetto corrente dopo l'assegnazione.
 */
inline Point3D &Point3D::operator=(const Point3D &other)
{
    if (this != &other)
    {
        x = other.x; 
        y = other.y; 
        z = other.z; 
        value = other.value;
        score = other.score;
        dir[0] = other.dir[0];
        dir[1] = other.dir[1];
        dir[2] = other.dir[2];
    }
    return *this; 
}

/**
 * @brief Operatore di sottrazione per la classe Point3D.
 * 
 * Questo operatore sottrae le coordinate dell'oggetto `other` dalle coordinate 
 * dell'oggetto corrente. Restituisce un nuovo oggetto `Point3D` che rappresenta 
 * il risultato della sottrazione vettoriale delle coordinate x, y, e z dei due punti.
 * 
 * @param other Riferimento costante all'oggetto `Point3D` da sottrarre.
 * 
 * @return Point3D Nuovo oggetto `Point3D` contenente la differenza delle coordinate.
 */
inline Point3D Point3D::operator-(const Point3D &other) const
{
    return Point3D(x - other.x, y - other.y, z - other.z); 
}
/**
 * @brief Operatore di somma per la classe Point3D.
 * 
 * Questo operatore aggiunge le coordinate dell'oggetto `other` dalle coordinate 
 * dell'oggetto corrente. Restituisce un nuovo oggetto `Point3D` che rappresenta 
 * il risultato della somma vettoriale delle coordinate x, y, e z dei due punti.
 * 
 * @param other Riferimento costante all'oggetto `Point3D` da sottrarre.
 * 
 * @return Point3D Nuovo oggetto `Point3D` contenente la somma delle coordinate.
 */
inline Point3D Point3D::operator+(const Point3D &other) const
{
    return Point3D(x + other.x, y + other.y, z + other.z); 
}

/**
 * @brief Operatore di uguaglianza per la classe Point3D.
 * 
 * Questo operatore confronta le coordinate dell'oggetto corrente con quelle dell'oggetto `other`.
 * Restituisce true se le coordinate x, y e z di entrambi i punti sono uguali, altrimenti restituisce false.
 * 
 * @param other Riferimento costante all'oggetto `Point3D` da confrontare.
 * 
 * @return bool Restituisce true se le coordinate x, y e z sono uguali, altrimenti false.
 */
inline bool Point3D::operator==(const Point3D &other) const
{
    return (this->x == other.x && this->y == other.y && this->z == other.z); 
}

/**
 * @brief Operatore di divisione scalare per la classe Point3D.
 * 
 * Questo operatore divide le coordinate x, y e z dell'oggetto corrente per un valore scalare `norm`. 
 * Se il valore di `norm` è zero, la funzione non esegue la divisione per evitare errori di divisione 
 * per zero. L'operatore modifica le coordinate del punto corrente.
 * 
 * @param norm Valore scalare per la divisione delle coordinate.
 * 
 * @return void
 */
inline void Point3D::operator/=(float norm)
{
    if (norm == 0)
        return;
    x /= norm;
    y /= norm; 
    z /= norm; 
}

/**
 * @brief Operatore prodotto scalare per la classe Point3D.
 * 
 * Questo operatore calcola il prodotto scalare tra le coordinate dell'oggetto corrente 
 * e quelle dell'oggetto `other`. Il prodotto scalare è la somma dei prodotti delle 
 * coordinate corrispondenti (x, y, z) dei due punti.
 * 
 * @param other Riferimento costante all'oggetto `Point3D` con cui eseguire il prodotto scalare.
 * 
 * @return float Il risultato del prodotto scalare tra i due punti.
 */
inline float Point3D::operator*(const Point3D &other) const
{
    return x * other.x + y * other.y + z * other.z; 
}

// Moltiplicazione di un vettore per uno scalare
inline Point3D Point3D::operator*(float scalar) const {
    return Point3D(x * scalar, y * scalar, z * scalar);
}

//Proiezione di un punto P sulla retta individuata dal punto chiamante e dal vettore direzione normalizzato d
inline Point3D Point3D::projection(const Point3D& p, const Point3D& d) {
    Point3D vettoreDalCentro= p - *this; // Vettore tra il punto e la direzione
    float componenteAsse = vettoreDalCentro * d; // Prodotto scalare
    return Point3D(*this + (d*componenteAsse)); // Somma il vettore proiettato al punto di origine
}

/**
 * @brief Restituisce la coordinata x del punto.
 * 
 * Questa funzione restituisce il valore della coordinata x dell'oggetto `Point3D`. 
 * È un metodo di sola lettura che non modifica lo stato dell'oggetto.
 * 
 * @return float Valore della coordinata x.
 */
inline float Point3D::getX() const 
{
    return x; 
}

/**
 * @brief Restituisce la coordinata y del punto.
 * 
 * Questa funzione restituisce il valore della coordinata y dell'oggetto `Point3D`. 
 * È un metodo di sola lettura che non modifica lo stato dell'oggetto.
 * 
 * @return float Valore della coordinata y.
 */
inline float Point3D::getY() const 
{
    return y; 
}

/**
 * @brief Restituisce la coordinata z del punto.
 * 
 * Questa funzione restituisce il valore della coordinata z dell'oggetto `Point3D`. 
 * È un metodo di sola lettura che non modifica lo stato dell'oggetto.
 * 
 * @return float Valore della coordinata z.
 */
inline float Point3D::getZ() const
{
    return z;
}

/**
 * @brief Restituisce il valore di intensità value.
 * 
 * Questa funzione restituisce il valore di intensità value dell'oggetto `Point3D`. 
 * È un metodo di sola lettura che non modifica lo stato dell'oggetto.
 * 
 * @return float Valore del campo 'value'.
 */
inline float Point3D::getValue() const
{
    return value;
}

/**
 * @brief Assegna il valore di intensità value.
 * 
 * Questa funzione assegna al valore di intensità value dell'oggetto `Point3D` il
 * valore v fornito dall'utente. 
 * 
 * @param v Valore da assegnare al campo 'value'.
 */
inline void Point3D::setValue(float v)
{
    value = v;
}
/**
 * @brief Restituisce il valore score.
 * 
 * Questa funzione restituisce il valore scoree dell'oggetto `Point3D`. 
 * È un metodo di sola lettura che non modifica lo stato dell'oggetto.
 * 
 * @return float Valore del campo 'score'.
 */
inline float Point3D::getScore() const
{
    return score;
}

/**
 * @brief Assegna il valore al campo 'score'.
 * 
 * Questa funzione assegna al valore score dell'oggetto `Point3D` il
 * valore s fornito dall'utente. 
 * 
 * @param s Valore da assegnare al campo 'score'.
 */
inline void Point3D::setScore(float s)
{
    score = s;
}

/**
 * @brief Imposta il vettore direzionale del punto.
 * 
 * Questa funzione aggiorna i valori del vettore direzionale `dir` dell'oggetto `Point3D` 
 * con i nuovi valori forniti per le componenti x, y e z. Il vettore direzionale viene 
 * memorizzato in un array di 3 elementi.
 * 
 * @param x Componente x del vettore direzionale.
 * @param y Componente y del vettore direzionale.
 * @param z Componente z del vettore direzionale.
 * 
 * @return void
 */
inline void Point3D::setDir(float x, float y, float z)
{
    dir[0] = x;
    dir[1] = y;
    dir[2] = z;
}

/**
 * @brief Restituisce un puntatore al vettore tridimensionale di direzione `dir`.
 * 
 * Questa funzione restituisce un puntatore al vettore `dir`, che rappresenta la direzione 
 * dell'oggetto `Point3D` nello spazio tridimensionale. Le tre componenti della direzione 
 * (x, y, z) sono accessibili tramite questo puntatore.
 * 
 * @return float* Puntatore al vettore contenente le tre dimensioni della direzione.
 */

inline float *Point3D::getDir()
{
    return dir;
}
/**
 * @brief Restituisce un puntatore costante al vettore tridimensionale di direzione `dir`.
 * 
 * Questa funzione restituisce un puntatore costante al vettore `dir`, che rappresenta la direzione 
 * dell'oggetto `Point3D` nello spazio tridimensionale. Le tre componenti della direzione 
 * (x, y, z) sono accessibili tramite questo puntatore. Il puntatore è costante, quindi 
 * non permette la modifica dei valori.
 * 
 * @return const float* Puntatore costante al vettore contenente le tre dimensioni della direzione.
 */
inline const float* Point3D::getDire() const
{
    return dir;
}
/**
 * @brief Stampa i valori delle coordinate e degli attributi del punto.
 * 
 * Questa funzione stampa i valori delle coordinate x, y e z, insieme al valore 
 * `value` e al punteggio `score` dell'oggetto `Point3D`. I valori di `value` e `score` 
 * vengono stampati come interi. L'output è formattato in modo leggibile per fornire 
 * informazioni sulle proprietà del punto.
 * 
 * @return void
 */
inline void Point3D::print() const
{
    cout << "X: " << this->x << " Y: " << this->y << " Z: " << this->z << " Value: (int)" << (int)value << " Score: " << (int)score << endl;
}
/**
 * @brief Normalizza il vettore rappresentato dal punto.
 * 
 * Questa funzione normalizza il vettore tridimensionale rappresentato dalle coordinate 
 * x, y e z dell'oggetto `Point3D`. La normalizzazione consiste nel dividere ciascuna 
 * coordinata per la lunghezza del vettore (norma), in modo che il vettore risultante 
 * abbia lunghezza unitaria. Se la norma del vettore è zero o negativa, la funzione 
 * non esegue alcuna operazione per evitare divisioni per zero.
 * 
 * @return void
 */
inline void Point3D::normalize()
{
    float norm = sqrt(x * x + y * y + z * z);
    if (norm <= 0.0f)
        return;
    x /= norm;
    y /= norm;
    z /= norm;
}

/**
 * @brief Calcola la norma (lunghezza) del vettore tridimensionale di un punto.
 * 
 * Questa funzione calcola la norma (lunghezza euclidea) del vettore tridimensionale 
 * rappresentato dall'oggetto `Point3D` passato come parametro `P`. La norma viene 
 * calcolata come la radice quadrata della somma dei quadrati delle coordinate x, y e z.
 * 
 * @param P Riferimento costante a un oggetto `Point3D` di cui si desidera calcolare la norma.
 * 
 * @return float La norma (lunghezza) del vettore tridimensionale di `P`.
 */
inline float Point3D::norm(const Point3D &P)
{
    float norm = sqrt(P.x * P.x + P.y * P.y + P.z * P.z);
    return norm;
}
inline float Point3D::norm()
{
    float norm = sqrt(x * x + y * y + z * z);
    return norm;
}
/**
 * @brief Calcola il prodotto vettoriale (cross product) tra due vettori tridimensionali.
 * 
 * Questa funzione calcola il prodotto vettoriale (cross product) tra i vettori tridimensionali 
 * rappresentati dagli oggetti `Point3D` `P` e `v`. Il risultato del prodotto vettoriale 
 * è un nuovo vettore tridimensionale, perpendicolare ai due vettori originali.
 * 
 * @param P Riferimento costante al primo vettore `Point3D`.
 * @param v Riferimento costante al secondo vettore `Point3D`.
 * 
 * @return Point3D Nuovo oggetto `Point3D` che rappresenta il risultato del prodotto vettoriale.
 */
inline Point3D Point3D::cross(const Point3D& P, const Point3D& v){
    return Point3D( P.x * v.y - P.y * v.x,
                    P.z * v.x - P.x * v.z,
                    P.y * v.z - P.z * v.y
                    );
}
/**
 * @brief Calcola la distanza tra un punto e una linea nello spazio tridimensionale.
 * 
 * Questa funzione calcola la distanza tra il punto `P` e una linea definita dal punto `A` 
 * e dal vettore direzionale `v`. Utilizza il prodotto vettoriale tra il vettore AP (distanza tra `A` e `P`)
 * e il vettore `v`, seguito dalla divisione per la norma di `v`, per ottenere la distanza minima 
 * tra il punto e la linea.
 * 
 * @param P Riferimento costante al punto di cui si desidera calcolare la distanza dalla linea.
 * @param A Riferimento costante al punto su cui passa la linea.
 * @param v Riferimento costante al vettore direzionale della linea.
 * 
 * @return double La distanza minima tra il punto `P` e la linea definita da `A` e `v`.
 */
inline double Point3D::pointToLineDistance(const Point3D &P, const Point3D &A, const Point3D &v)
{
    Point3D AP = Point3D(P.z - A.z, P.y - A.y, P.x - A.x);
    Point3D crossProduct = cross(AP, v);
    float n = norm(v);
    return norm(crossProduct) / n;
}

inline float distance(const Point3D &P1, const Point3D &P2)
{
    return sqrt((P1.x - P2.x) * (P1.x - P2.x) +
                (P1.y - P2.y) * (P1.y - P2.y) +
                (P1.z - P2.z) * (P1.z - P2.z));
}

/**
 * @brief Calcola la distanza di un punto da un piano nello spazio tridimensionale.
 * 
 * Questa funzione determina la distanza minima tra un punto `p` e un piano definito da due autovettori `autovet1` e `autovet2`.
 * Il piano viene descritto dal vettore normale ottenuto tramite il prodotto vettoriale tra gli autovettori, 
 * e un punto di riferimento `referencePoint` appartenente al piano.
 * 
 * @param autovet1 Primo autovettore (array di 3 float) che definisce il piano.
 * @param autovet2 Secondo autovettore (array di 3 float) che definisce il piano.
 * @param p Punto `Point3D` di cui calcolare la distanza dal piano.
 * @param referencePoint Punto `Point3D` di riferimento che appartiene al piano.
 * 
 * @return float Distanza minima tra il punto `p` e il piano.
 */
inline float distancePointToPlane(const float autovet1[3], const float autovet2[3], const Point3D& p, const Point3D& referencePoint) {
    // Crea i Point3D per i due autovettori
    Point3D v1(autovet1[0], autovet1[1], autovet1[2]);
    Point3D v2(autovet2[0], autovet2[1], autovet2[2]);

    // Calcola il vettore normale al piano usando il prodotto vettoriale tra i due autovettori
    Point3D normal = normal.cross(v1, v2);

    // Normalizza il vettore normale
    normal.normalize();

    // Calcola il parametro d dell'equazione del piano ax + by + cz + d = 0
    float d = -(normal.x * referencePoint.x + normal.y * referencePoint.y + normal.z * referencePoint.z);
    float denom= sqrt(normal.x * normal.x + normal.y * normal.y + normal.z * normal.z);
    if (denom==0) return 0; //se ho zero al denominatore ritorno distanza nulla
    // Calcola la distanza dal punto p al piano
    float distance = (normal.x * p.x + normal.y * p.y + normal.z * p.z + d) / denom;

    return fabs(distance);
}

// Calcola il baricentro
inline Point3D Point3D::calculateCentroid(const std::vector<Point3D>& points) {
    Point3D centroid = {0.0f, 0.0f, 0.0f};
    for (const auto& point : points) {
        centroid.x += point.x;
        centroid.y += point.y;
        centroid.z += point.z;
    }
    centroid.x /= points.size();
    centroid.y /= points.size();
    centroid.z /= points.size();
    return centroid;
}

// Calcola la matrice di covarianza
inline std::array<std::array<float, 3>, 3> Point3D::calculateCovarianceMatrix(const std::vector<Point3D>& points, const Point3D& centroid) {
    std::array<std::array<float, 3>, 3> covariance = {{{0.0f, 0.0f, 0.0f}, {0.0f, 0.0f, 0.0f}, {0.0f, 0.0f, 0.0f}}};

    for (const auto& point : points) {
        float x = point.x - centroid.x;
        float y = point.y - centroid.y;
        float z = point.z - centroid.z;

        covariance[0][0] += x * x; covariance[0][1] += x * y; covariance[0][2] += x * z;
        covariance[1][0] += y * x; covariance[1][1] += y * y; covariance[1][2] += y * z;
        covariance[2][0] += z * x; covariance[2][1] += z * y; covariance[2][2] += z * z;
    }

    for (auto& row : covariance) {
        for (auto& value : row) {
            value /= points.size();
        }
    }
    return covariance;
}

// Algoritmo di potenza per trovare l'autovettore principale
inline std::array<float, 3> Point3D::powerIteration(const std::array<std::array<float, 3>, 3>& matrix, int iterations, float tolerance) {
    std::array<float, 3> eigenvector = {1.0f, 1.0f, 1.0f};
    float norm = std::sqrt(eigenvector[0] * eigenvector[0] + eigenvector[1] * eigenvector[1] + eigenvector[2] * eigenvector[2]);

    for (auto& value : eigenvector) {
        value /= norm;
    }

    for (int iter = 0; iter < iterations; ++iter) {
        std::array<float, 3> nextVector = {0.0f, 0.0f, 0.0f};
        for (int i = 0; i < 3; ++i) {
            for (int j = 0; j < 3; ++j) {
                nextVector[i] += matrix[i][j] * eigenvector[j];
            }
        }

        norm = std::sqrt(nextVector[0] * nextVector[0] + nextVector[1] * nextVector[1] + nextVector[2] * nextVector[2]);
        for (auto& value : nextVector) {
            value /= norm;
        }

        float diff = std::abs(nextVector[0] - eigenvector[0]) + std::abs(nextVector[1] - eigenvector[1]) + std::abs(nextVector[2] - eigenvector[2]);
        eigenvector = nextVector;

        if (diff < tolerance) break;
    }

    return eigenvector;
}

// Calcola il segmento proiettato sulla retta di regressione
inline std::pair<Point3D, Point3D> Point3D::calculateRegressionSegment(const std::vector<Point3D>& points) {
    if (points.size() < 2) {
        throw std::invalid_argument("Non ci sono abbastanza punti per calcolare la retta di regressione.");
    }

    Point3D centroid = calculateCentroid(points);
    auto covarianceMatrix = calculateCovarianceMatrix(points, centroid);
    auto direction = powerIteration(covarianceMatrix);

    float minProj = std::numeric_limits<float>::max();
    float maxProj = std::numeric_limits<float>::lowest();
    Point3D minPoint, maxPoint;

    for (const auto& point : points) {
        float x = point.x - centroid.x;
        float y = point.y - centroid.y;
        float z = point.z - centroid.z;

        float projValue = x * direction[0] + y * direction[1] + z * direction[2];

        Point3D projectedPoint = {
            centroid.x + projValue * direction[0],
            centroid.y + projValue * direction[1],
            centroid.z + projValue * direction[2]
        };

        if (projValue < minProj) {
            minProj = projValue;
            minPoint = projectedPoint;
        }
        if (projValue > maxProj) {
            maxProj = projValue;
            maxPoint = projectedPoint;
        }
    }

    return {minPoint, maxPoint};
}


/**
 * @brief Proietta un punto nello spazio tridimensionale su un piano definito dagli autovettori medio e massimo.
 * 
 * La funzione prende come input un punto `P` da cui calcolare un vettore verso un secondo punto `Q`, e due autovettori
 * che definiscono il piano su cui proiettare `Q`. Il processo di proiezione si basa sul calcolo del prodotto scalare
 * tra il vettore PQ e il vettore normale al piano definito dall'autovettore massimo (`autovetMax`).
 * 
 * @param P Il punto di origine utilizzato per creare un vettore verso il punto `Q`.
 * @param autovetMax Autovettore massimo che definisce la normale del piano (coordinate x, y, z).
 * @param Q Il punto da proiettare sul piano.
 * @param autovetMid Autovettore medio che contribuisce a definire il piano di proiezione (non direttamente usato nella funzione).
 * @param autovetMin Autovettore minimo che contribuisce a definire il piano di proiezione (non direttamente usato nella funzione).
 * 
 * @return Point3D Il punto `Q` proiettato sul piano definito dall'autovettore massimo.
 * 
 * @details 
 * - La funzione calcola un vettore `PQ` dalla differenza tra `Q` e `P`.
 * - Utilizza il prodotto scalare tra `PQ` e il vettore normale (`autovetMax`) per determinare quanto `Q` si proietta
 *   lungo la direzione della normale.
 * - Sottrae la componente lungo la normale dal punto `Q` per ottenere la sua proiezione sul piano.
 * - Restituisce un nuovo oggetto `Point3D` che rappresenta la posizione di `Q` proiettata sul piano.
 */
inline Point3D Point3D::projectOntoPlane(const Point3D& P, const float autovetMid[3], const float autovetMin[3], const Point3D& Q) {
    // Converti gli autovettori in oggetti Point3D
    Point3D vMid(autovetMid[0], autovetMid[1], autovetMid[2]);
    Point3D vMin(autovetMin[0], autovetMin[1], autovetMin[2]);

    // Calcola il vettore normale al piano tramite il prodotto vettoriale
    Point3D normalVector = cross(vMid, vMin);  // Assumendo che cross sia un metodo che esegue il prodotto vettoriale

    // Calcola il vettore PQ da P a Q
    Point3D PQ = Q - P;

    // Calcola il prodotto scalare tra PQ e il vettore normale
    float dotProduct = PQ * normalVector;

    // Proietta Q sul piano
    if(normalVector.norm(normalVector)==0) return Q; //se la norma è nulla il punto è già sul piano
    Point3D projectedPoint = Q - normalVector * (dotProduct / normalVector.norm(normalVector));

    return projectedPoint;
}

// inline Point3D Point3D::projectOntoPlane(const Point3D& P, const float autovetMax[3], const Point3D& Q, const float autovetMid[3], const float autovetMin[3]) {
//     // Calcolare il vettore PQ
//     Point3D PQ = Q - P;

//     // Calcolare il prodotto scalare PQ · autovetMax
//     Point3D normalVector(autovetMax[0], autovetMax[1], autovetMax[2]); // Converti l'autovettore massimo in un oggetto Point3D
//     float dotProduct = PQ * normalVector; // Usa il metodo di prodotto scalare della classe Point3D

//     // Calcolare il punto proiettato
//     Point3D projectedPoint = {
//         Q.getX() - dotProduct * normalVector.getX(),
//         Q.getY() - dotProduct * normalVector.getY(),
//         Q.getZ() - dotProduct * normalVector.getZ()
//     };
//     return projectedPoint;
// }

#endif
