#include "libraries/json.hpp"
#include "Point3D.hh"
#include "pixelParams.hh"
#include "Evaluation.hh"
#include <filesystem>
#include <fstream>
#include <vector>
#include <string>
#include <algorithm>
#ifdef USE_STD
#include "configurations/configSTD.hh"
#endif
#ifdef USE_ALT1
#include "configurations/configALT1.hh"
#endif
// Usa lo spazio dei nomi per semplificare la scrittura del codice
using json = nlohmann::json;
namespace fs = filesystem;

// Struttura per rappresentare un tubo
struct TuboJSON {
    //Nodo di iterazione in cui è stato scoperto il percorso
    int epoca;
    //Diametro del cilindro stimato per il tubo
    float diametro;
    float bestScore;
    //statistiche su baricentro.value
    float mediaValueBaricentri;
    float medianaValueBaricentri;
    //statistiche su matrix(bar.x, bar.y, bar.z)
    float mediaMatrixBar;
    float medianMatrixBar;
    //Primo baricentro.value, ridondante
    float valuePartenza;
    //Distanza cumulativa del tubo
    float distanza;
    //Nome del tubo
    std::string nome;
    //Vettore che contiene le intensità degli archi del percorso, ordinate
    std::vector<float> intensitiesEdge;
    //Vettore che contiene i valori dei Baricentri
    std::vector<float> barValues;
    //Distanza massimo-baricentro
    std::vector<float> barmaxDist;
    //Vettore che contiene i valori matriciali dei baricentri
    std::vector<float> matrixValues;
    //Vettore che contiene gli angoli tra le pca dei baricentri
    std::vector<float> pcaAngles;
    //Vettore che contiene gli angoli tra le pca dei baricentri e la direzione iniziale
    std::vector<float> pcaStartAngles;
    //valori massimi locali associati
    std::vector<float> valueMaxLocali;
    //Baricentro di partenza
    Point3D barPartenza;
    //Vettore che contiene i punti ordinati che formano il percorso
    std::vector<Point3D> points;
    //Direzione PCA primo baricentro su cui fare regression
    Point3D direction;
    vector<Point3D> regPoints;

    // Costruttore per inizializzare un tubo con un nome
    TuboJSON(const std::string& n) : nome(n) {}
     // Costruttore di default
    TuboJSON() : epoca(0), diametro(0.0f), bestScore(0.0f), mediaValueBaricentri(0.0f), medianaValueBaricentri(0.0f), valuePartenza(0.0f), distanza(0.0f), nome("Tubo") {}
    // Funzione per aggiungere un'intensità al tubo
    void aggiungiIntensita(const float& intensity) {
        intensitiesEdge.push_back(intensity);
    }
    //Funzione per aggiungere un punto al percorso
    void aggiungiPunto(const Point3D& point){
        points.push_back(point);
    }
};
// Funzione per calcolare la distanza perpendicolare di un punto dalla retta definita da due punti
inline float perpendicularDistance(const Point3D& point, const Point3D& start, const Point3D& end) {
    float dx = end.x - start.x;
    float dy = end.y - start.y;
    float dz = end.z - start.z;

    float numerator = std::abs(dz * (point.x - start.x) - dx * (point.z - start.z));
    float denominator = std::sqrt(dx * dx + dy * dy + dz * dz);
    if (denominator!=0)
    return numerator / denominator;
    return 0;
}
/**
 * @brief Implementazione Algoritmo DouglasPeucker
 * https://en.wikipedia.org/wiki/Ramer%E2%80%93Douglas%E2%80%93Peucker_algorithm
 * 
 */
// Funzione per applicare l'algoritmo di Douglas-Peucker ai punti 3D
inline std::vector<Point3D> douglasPeucker(const std::vector<Point3D>& points, float epsilon) {
    if (points.size() < 2) return points;

    // Trova il punto con la massima distanza perpendicolare dalla retta
    float dmax = 0.0f;
    int index = 0;
    int end = points.size() - 1;

    for (int i = 1; i < end; ++i) {
        float d = perpendicularDistance(points[i], points[0], points[end]);
        if (d > dmax) {
            dmax = d;
            index = i;
        }
    }

    std::vector<Point3D> result;

    // Se la distanza massima è maggiore della soglia epsilon, continua la semplificazione
    if (dmax > epsilon) {
        // Chiamata ricorsiva per i due segmenti
        std::vector<Point3D> recResults1 = douglasPeucker(std::vector<Point3D>(points.begin(), points.begin() + index + 1), epsilon);
        std::vector<Point3D> recResults2 = douglasPeucker(std::vector<Point3D>(points.begin() + index, points.end()), epsilon);

        // Combina i risultati, evitando il punto di sovrapposizione
        result.insert(result.end(), recResults1.begin(), recResults1.end() - 1);
        result.insert(result.end(), recResults2.begin(), recResults2.end());
    } else {
        // Se la distanza è minore della soglia, mantieni solo i due punti estremi
        result.push_back(points[0]);
        result.push_back(points[end]);
    }

    return result;
}

//Funzione che trasforma un oggetto TuboJSON in un json
inline json tuboToJson(const TuboJSON& tubo) {
    json tuboJson;
    tuboJson["Epoca"] = tubo.epoca;
    tuboJson["Diametro"] = tubo.diametro;
    tuboJson["DiametroRicercaRicorsiva"] = tubo.diametro*DIAMETER_INCREASE_FACTOR_RR;
    tuboJson["DiametroOutOfFlag"] = tubo.diametro* DIAMETER_INCREASE_FACTOR_OOF;
    tuboJson["Distanza Percorso"] = tubo.distanza;
    tuboJson["BestScore"] = tubo.bestScore;
    tuboJson["MediaValoriBaricentri"]= tubo.mediaValueBaricentri;
    tuboJson["ValuesBaricentri"]=tubo.barValues;
    tuboJson["MatrixValues"]=tubo.matrixValues;
    tuboJson["BarmaxDistance"]=tubo.barmaxDist;

    // Aggiungi intensità
    tuboJson["EdgeIntensities"] = tubo.intensitiesEdge;
    //Calcolo della mediana
    vector<float> barValues=tubo.barValues;
    tuboJson["MedianaValueBaricentri"]= calculateMedian(barValues);
    // Aggiungi punti
    json pointsJson = json::array();
    for (const auto& point : tubo.points) {
        pointsJson.push_back({point.x, point.y, point.z});
    }
    tuboJson["Points"] = pointsJson;
    //Aggiungi Punti Douglas-Peucker
    vector<Point3D> dpPoints = douglasPeucker(tubo.points, 1.0f);
    json dpPointsJson = json::array();
    for (const auto& point : dpPoints) {
        dpPointsJson.push_back({point.x, point.y, point.z});
    }
    tuboJson["dpPoints"]= dpPointsJson;
    //Aggiungi i due estremi di regressione
    json regPoints= json::array();
    Point3D p3d=Point3D(tubo.barPartenza);
    Point3D direzione_norm=tubo.direction;
    Point3D direzione_inv_norm=Point3D(
        tubo.direction * -1
    );
    Point3D regPoint1= p3d.projection(tubo.points[0], direzione_norm);
    Point3D regPoint2= p3d.projection(tubo.points[tubo.points.size()-1], direzione_inv_norm);
    pair<Point3D, Point3D> pts=regPoint1.calculateRegressionSegment(tubo.points);
    regPoint1=pts.first;
    regPoint2=pts.second;
    regPoints.push_back({regPoint1.x, regPoint1.y, regPoint1.z});
    regPoints.push_back({regPoint2.x, regPoint2.y, regPoint2.z});
    tuboJson["regPoints"]= regPoints;
    //angoli dei baricentri con PCA successiva
    tuboJson["pcaAngles"]=tubo.pcaAngles;
    //angoli dei baricentri con PCA iniziale
    tuboJson["pcaStartAngles"]=tubo.pcaStartAngles;
    tuboJson["maxLocalValues"]=tubo.valueMaxLocali;
    return tuboJson;
}

// // Funzione per salvare più tubi in un unico file JSON
// inline void salvaTubiInJson(const std::vector<TuboJSON>& tubi, const std::string& filename,
// int offsetTop, int offsetBottom, int offsetLeft, int offsetRight, int offsetZ) {
//     json outputJson;
//     // Aggiungi i dati di offset una sola volta all'inizio
//     outputJson["Offset"] = {
//         {"Top", offsetTop},
//         {"Bottom", offsetBottom},
//         {"Left", offsetLeft},
//         {"Right", offsetRight},
//         {"Z", offsetZ}
//     };
//     // Creazione della directory, se non esiste già
//     std::filesystem::create_directories(filename);
//
//     for (const auto& tubo : tubi) {
//         outputJson[tubo.nome] = tuboToJson(tubo);
//         //cout<<tubo.nome<<endl;
//     }
//     try{
//         if (!fs::exists(filename))
//             fs::create_directory(filename);
//         // Tenta di aprire il file
// //         std::ofstream outFile(filename+FILENAME_JSON);
//         // Scrivi il JSON nel file
//         outFile << outputJson.dump(4); // Indentazione di 4 spazi per leggibilità
//         outFile.close();
//     }
//     catch(const exception e){}
//
//
// }
/**
 * @brief Funzione che unisce due semitubi in un unico tubo.
 * Il primo tubo viene invertito e unito al secondo.
 */
inline TuboJSON unisciSemitubi(const TuboJSON& semitubo1, const TuboJSON& semitubo2) {
    if (semitubo1.epoca != semitubo2.epoca) {
        throw std::invalid_argument("I semitubi non appartengono alla stessa epoca, impossibile unirli.");
    }

    // Crea un nuovo tubo unito
    TuboJSON tuboUnito("Tubo_" + std::to_string(semitubo1.epoca));
    tuboUnito.epoca = semitubo1.epoca;

    // Inverto i punti, le intensità e i valori dei baricentri del primo semitubo
    std::vector<Point3D> puntiInvertiti = semitubo1.points;
    std::reverse(puntiInvertiti.begin(), puntiInvertiti.end());

    std::vector<float> intensitaInvertite = semitubo1.intensitiesEdge;
    std::reverse(intensitaInvertite.begin(), intensitaInvertite.end());

    std::vector<float> baricentriInvertiti = semitubo1.barValues;
    std::reverse(baricentriInvertiti.begin(), baricentriInvertiti.end());

    
    std::vector<float> matrixValuesInvertiti = semitubo1.matrixValues;
    std::reverse(matrixValuesInvertiti.begin(), matrixValuesInvertiti.end());

    std::vector<float> matrixLocalMaxInvertiti = semitubo1.valueMaxLocali;
    std::reverse(matrixLocalMaxInvertiti.begin(), matrixLocalMaxInvertiti.end());

    // Unisco i dati invertiti del primo semitubo ai dati del secondo semitubo
    tuboUnito.points = puntiInvertiti;
    tuboUnito.points.insert(tuboUnito.points.end(), semitubo2.points.begin(), semitubo2.points.end());

    tuboUnito.intensitiesEdge = intensitaInvertite;
    tuboUnito.intensitiesEdge.insert(tuboUnito.intensitiesEdge.end(), semitubo2.intensitiesEdge.begin(), semitubo2.intensitiesEdge.end());

    tuboUnito.barValues = baricentriInvertiti;
    tuboUnito.barValues.insert(tuboUnito.barValues.end(), semitubo2.barValues.begin(), semitubo2.barValues.end());
    
    tuboUnito.matrixValues = matrixValuesInvertiti;
    tuboUnito.matrixValues.insert(tuboUnito.matrixValues.end(), semitubo2.matrixValues.begin(), semitubo2.matrixValues.end());
    
    tuboUnito.valueMaxLocali = matrixLocalMaxInvertiti;
    tuboUnito.valueMaxLocali.insert(tuboUnito.valueMaxLocali.end(), semitubo2.valueMaxLocali.begin(), semitubo2.valueMaxLocali.end());
    //Unisco i dati sugli angoli invertiti
    std::vector<float> pcaAnglesInvertiti = semitubo1.pcaAngles;
    std::reverse(pcaAnglesInvertiti.begin(), pcaAnglesInvertiti.end());
    tuboUnito.pcaAngles=pcaAnglesInvertiti;
    tuboUnito.pcaAngles.insert(tuboUnito.pcaAngles.end(), semitubo2.pcaAngles.begin(), semitubo2.pcaAngles.end());
    
    std::vector<float> pcaStartAnglesInvertiti = semitubo1.pcaAngles;
    std::reverse(pcaStartAnglesInvertiti.begin(), pcaStartAnglesInvertiti.end());
    tuboUnito.pcaStartAngles=pcaStartAnglesInvertiti;
    tuboUnito.pcaStartAngles.insert(tuboUnito.pcaStartAngles.end(), semitubo2.pcaStartAngles.begin(), semitubo2.pcaStartAngles.end());
    
    std::vector<float> barmaxDistInvertiti = semitubo1.barmaxDist;
    std::reverse(barmaxDistInvertiti.begin(), barmaxDistInvertiti.end());
    tuboUnito.barmaxDist= barmaxDistInvertiti;
    tuboUnito.barmaxDist.insert(tuboUnito.barmaxDist.end(), semitubo2.barmaxDist.begin(), semitubo2.barmaxDist.end());


    // Calcolo delle statistiche
    tuboUnito.diametro = (semitubo1.diametro + semitubo2.diametro) / 2.0f;
    tuboUnito.bestScore = (semitubo1.bestScore + semitubo2.bestScore) / 2.0f;
    tuboUnito.mediaValueBaricentri = (semitubo1.mediaValueBaricentri + semitubo2.mediaValueBaricentri) / 2.0f;
    tuboUnito.medianaValueBaricentri = calculateMedian(tuboUnito.barValues);
    tuboUnito.valuePartenza=(semitubo1.valuePartenza); 
    tuboUnito.direction=(semitubo1.direction);
    tuboUnito.distanza= semitubo1.distanza + semitubo2.distanza;
    return tuboUnito;
}




/**
 * @brief Salva i tubi in JSON, fondendo quelli con la stessa epoca.
 */
inline void salvaTubiFusiInJson(const std::vector<TuboJSON>& tubi, const std::string& jsonFullPath,
                                int offsetTop, int offsetBottom, int offsetLeft, int offsetRight, int offsetZ) {
    json outputJson;

    // Aggiungi i dati di offset
    outputJson["Offset"] = {
        {"Top", offsetTop},
        {"Bottom", offsetBottom},
        {"Left", offsetLeft},
        {"Right", offsetRight},
        {"Z", offsetZ}
    };

    // Mappa per fondere i tubi con la stessa epoca
    std::map<int, TuboJSON> tubiPerEpoca;

    for (const auto& tubo : tubi) {
        if (tubiPerEpoca.find(tubo.epoca) != tubiPerEpoca.end()) {
            // Fondi il tubo esistente con quello attuale
            tubiPerEpoca[tubo.epoca] = unisciSemitubi(tubiPerEpoca[tubo.epoca], tubo);
        } else {
            // Aggiungi un nuovo tubo per questa epoca
            tubiPerEpoca[tubo.epoca] = tubo;
        }
    }

    // Aggiungi i tubi fusi al JSON
    for (const auto& [epoca, tubo] : tubiPerEpoca) {
        outputJson[tubo.nome] = tuboToJson(tubo);
    }

    // Directory e file
    fs::path p(jsonFullPath);
    fs::path dir = p.parent_path();  // ottieni la cartella
    fs::create_directories(dir);     // crea la cartella se non esiste

    // final output
    std::ofstream outFile(jsonFullPath);
    outFile << outputJson.dump(4);
    outFile.close();
}

