"""
ChangedController - Gestisce e notifica il cambio di:
- survey
- profondità tomografia
- polarizzazione
- swath
- channel

Quando un controller accetta la modifica di questi atttibuti lancerà l'emit di questi signal per notificare gli altri controller.
"""
# PYSIDE6
from PySide6.QtCore import QObject, Signal

# MODELS
from Model.SurveyManager import SurveyManager

# LOG
import logging

class ChangedController(QObject):
    _instance = None  # Qui memorizziamo l'unica istanza

    def __new__(cls):
        if cls._instance is None:
            cls._instance = super(ChangedController, cls).__new__(cls)
            cls._instance._initialized = False
        return cls._instance
    
    # SIGNAL
    survey_changed = Signal(int)
    depth_changed = Signal(int)
    polarization_changed = Signal(int)
    swath_changed = Signal(int)
    channel_changed = Signal(int)

    def __init__(self, parent=None):
        # Evita la doppia inizializzazione del ChangedController
        if hasattr(self, '_initialized') and self._initialized:
            return
            
        super().__init__(parent)
        self._initialized = True

        # SURVEY MANAGER
        self.survey_manager = SurveyManager()
        
        # LOG
        self.changed_logger = logging.getLogger("changed")
    
    def emit_survey_changed(self, survey):
        self.survey_manager.set_selected_survey(survey)
        self.survey_changed.emit(survey)
        self.changed_logger.info(f"Survey changed to: {survey}")

    def emit_depth_changed(self, depth):
        self.survey_manager.set_selected_depth(depth)
        self.depth_changed.emit(depth)
        self.changed_logger.info(f"Depth changed to: {depth}")

    def emit_polarization_changed(self, polarization):
        self.survey_manager.set_current_polarization(polarization)
        self.polarization_changed.emit(polarization)
        self.changed_logger.info(f"Polarization changed to: {polarization}")

    def emit_swath_changed(self, swath):
        self.survey_manager.set_selected_swath(swath)
        self.swath_changed.emit(swath)
        self.changed_logger.info(f"Swath changed to: {swath}")

    def emit_channel_changed(self, channel):
        self.survey_manager.set_selected_channel(channel)
        self.channel_changed.emit(channel)
        self.changed_logger.info(f"Channel changed to: {channel}")

        
        
