# controllers/ElaborationController.py
# IMPORT PYSIDE6
from PySide6.QtCore import QObject, Qt, QTimer
from PySide6.QtWidgets import QWidget, QHBoxLayout, QCheckBox, QPushButton, QProgressBar

# MODELLI
from Model.SurveyManager import SurveyManager

# THREAD
from Thread.ApplyFilter import ApplyFilter
from Thread.RunAI import RunAI
from Thread.RunAIDetection import RunAIDetection
from Thread.Raster_tile import Raster_tile

# LOG
import logging

class ElaborationController(QObject):
    """
    Controller dedicato esclusivamente al lancio dei thread di elaborazione filtri su tutti i dati.
    Non aggiorna mai la GUI e non accede mai ad altri controller.
    Si limita a lanciare ApplyFilter e a gestire i segnali di stato.
    """

    def __init__(self, view, parent=None):
        super().__init__(parent)

        self.view = view
        self.survey_manager = SurveyManager()
        self.elaboration_logger = logging.getLogger('elaboration')
        self.elaboration_logger.info("ElaborationController inizializzato con integrazione filtri manuale")

        # Thread per applicazione filtri (Singleton)
        self.th_apply_filter = ApplyFilter()
        self.th_apply_filter.filter_completed.connect(self._on_filter_completed)
        self.th_apply_filter.filter_error.connect(self._on_filter_error)
        self.th_apply_filter.filter_progress.connect(self._on_filter_progress)

        # Thread AI (placeholder)
        self.th_run_ai = RunAI()
        self.th_run_ai.ai_started.connect(lambda msg: self._on_ai_started(msg))
        self.th_run_ai.ai_progress.connect(lambda msg: self._on_ai_progress(msg))
        self.th_run_ai.ai_completed.connect(lambda msg: self._on_ai_completed(msg))
        # Thread AI Detection (C++ pipeline)
        self.th_ai_detection = RunAIDetection()
        self.th_ai_detection.ai_started.connect(lambda msg: self._on_ai_detection_started(msg))
        self.th_ai_detection.ai_progress.connect(lambda msg: self._on_ai_detection_progress(msg))
        self.th_ai_detection.ai_completed.connect(lambda msg: self._on_ai_detection_completed(msg))
        self.th_ai_detection.ai_error.connect(self._on_ai_detection_error)
        self.th_run_ai.ai_error.connect(lambda msg: self._on_ai_error(msg))

        self.th_raster_tile = Raster_tile()
        self.th_raster_tile.raster_tile_completed.connect(self.raster_tile_finished)
        # Connetto il signal tile_completed al RunAIDetection per processare ogni tile completato
        self.th_raster_tile.tile_completed.connect(self.th_ai_detection.on_tile_completed)

        # Progress bar per feedback utente
        from View.ProgressBar import ProgressBar
        self.progress_bar = ProgressBar(self.view)

        # Connessione menu filtri
        self.view.menu_filter.triggered.connect(self.handle_filter_action)
        
        # Lista filtri attivi in ordine ([(name, params), ...])
        self.active_filters = []

        # Attempt to connect AI checkbox; retry shortly if not yet available
        self._ai_checkbox_connected = False
        self._ai_busy_bar = None
        self._ai_detection_busy_bar = None
        self._ai_detection_busy_timer = None
        self._connect_ai_checkbox_if_available()
        QTimer.singleShot(500, self._connect_ai_checkbox_if_available)
        QTimer.singleShot(600, self._connect_inference_view_checkbox_if_available)
        QTimer.singleShot(650, self._connect_show_AI_data_checkbox)
        QTimer.singleShot(700, self._connect_rebuild_inference_button)

    def raster_tile_finished(self):
        self.progress_bar.close()


    # TUTTA QUESTA PARTE NON VA INSERITA NELLA VIEW MA IN UN CONTROLLER -------------------------------------
    def handle_filter_action(self, action):
        """Gestisce le azioni del menu filtri"""
        action_text = action.text()
        
        if action_text == "Apply Default Sequence":
            # Applica la sequenza predefinita di filtri
            self.apply_default_filter_sequence()
            
        elif action_text == "Fiter pass-band":
            # Applica singolo filtro pass-band
            params = {
                "dt_ns": 0.125,
                "fmin_mhz": 150,
                "fmax_mhz": 850,
                "order": 71
            }
            self.add_filter_checkbox("bandpass_fir", params)
            
        elif action_text == "Background removal":
            # Applica singolo filtro background removal
            params = {"window_traces": 0}
            self.add_filter_checkbox("background_removal", params)
            
        elif action_text == "Gain":
            # Applica singolo filtro gain
            self.add_filter_checkbox("gain", {})
            
        elif action_text == "Start time shifter":
            # Applica singolo filtro start time shifter
            self.add_filter_checkbox("start_time_shifterOGPR", {})
            
        else:
            self.elaboration_logger.warning(f"Azione filtro non riconosciuta: {action_text}")

    def add_filter_checkbox(self, filter_name, params):
        """Aggiunge un checkbox per un filtro"""
        row = QWidget()
        row_layout = QHBoxLayout(row)
        row_layout.setContentsMargins(0, 0, 0, 0)
        row_layout.setSpacing(5)

        checkbox = QCheckBox(filter_name)
        checkbox.setChecked(True)  # Attiva la spunta quando si aggiunge
        checkbox.setStyleSheet("color: white;")
        checkbox.stateChanged.connect(lambda state, name=filter_name: self.toggle_filter(name, state == Qt.Checked))

        remove_btn = QPushButton("✖")
        remove_btn.setFixedSize(20, 20)
        remove_btn.clicked.connect(lambda: self.remove_filter(row, filter_name))

        row_layout.addWidget(checkbox)
        row_layout.addStretch()
        row_layout.addWidget(remove_btn)

        self.view.scroll_layout.addWidget(row)

        # Aggiungi in coda e applica incrementalmente SOLO il nuovo filtro
        self.active_filters.append((filter_name, params))
        self.apply_single_filter_incremental(filter_name, params)

    def toggle_filter(self, filter_name, enabled):
        """Attiva/disattiva (mantiene ordine) e ricalcola sequenza"""
        if enabled:
            # niente: l'abilitazione avviene tramite add_filter_checkbox
            return
        # se disabilitato via checkbox, rimuovo dall'elenco e ricalcolo
        self.active_filters = [(n,p) for (n,p) in self.active_filters if n != filter_name]
        self.recompute_filters_sequence()

    def remove_filter(self, row_widget, filter_name):
        """Rimuove un filtro"""
        # Rimuovi dall'elenco e ricalcola
        self.active_filters = [(n,p) for (n,p) in self.active_filters if n != filter_name]
        self.recompute_filters_sequence()
        # Rimuovi il widget dalla lista
        self.view.scroll_layout.removeWidget(row_widget)
        row_widget.deleteLater()




    def recompute_filters_sequence(self):
        """Ricalcola l'intera sequenza dei filtri attivi, partendo dai dati originali.
        Si ottiene: disattivando un filtro, i successivi restano applicati (perché ricalcolati).
        """
        if self.th_apply_filter is not None and self.th_apply_filter.isRunning():
            self.elaboration_logger.warning("Elaborazione già in corso, attendere...")
            return
        try:
            # Normalizzo: sostituisco None con {}
            seq = [(name, (params or {})) for (name, params) in self.active_filters]
            self.elaboration_logger.info(f"Ricalcolo sequenza filtri: {[n for (n,_) in seq]}")
            msg = "Nessun filtro attivo" if not seq else f"Applicazione {len(seq)} filtri..."
            self.progress_bar.show_progress_dialog(msg)
            # Configura thread con sequenza custom e reset cartelle
            self.th_apply_filter.set_filter_sequence(seq, reset_before=True)
            self.th_apply_filter.start()
        except Exception as e:
            error_msg = f"Errore ricalcolo sequenza filtri: {str(e)}"
            self.elaboration_logger.error(error_msg)
            self.progress_bar.close()

    def apply_single_filter_incremental(self, filter_name, params):
        """Applica SOLO il filtro aggiunto, usando come input i dati già presenti in Data_current_filter."""
        if self.th_apply_filter is not None and self.th_apply_filter.isRunning():
            self.elaboration_logger.warning("Elaborazione già in corso, attendere...")
            return
        try:
            self.elaboration_logger.info(f"Applico incrementalmente filtro: {filter_name}")
            self.progress_bar.show_progress_dialog(f"Applicazione filtro {filter_name}...")
            self.th_apply_filter.set_filter(filter_name, params)
            # In run() il filtro singolo userà come input i dati filtrati precedenti (prefer_filtered_input=True)
            self.th_apply_filter.start()
        except Exception as e:
            error_msg = f"Errore applicazione filtro incrementale {filter_name}: {str(e)}"
            self.elaboration_logger.error(error_msg)
            self.progress_bar.close()

    def apply_default_filter_sequence(self):
        """Applica la sequenza predefinita di filtri (come prima dell'import automatico)"""
        if self.th_apply_filter is not None and self.th_apply_filter.isRunning():
            self.elaboration_logger.warning("Elaborazione già in corso, attendere...")
            return
            
        if len(self.survey_manager) == 0:
            self.elaboration_logger.error("Nessun survey caricato per applicare filtri")
            return
            
        try:
            self.elaboration_logger.info("Avvio sequenza filtri predefinita")
            self.progress_bar.show_progress_dialog("Applicazione sequenza filtri predefinita...")
            
            # Configura per sequenza predefinita
            self.th_apply_filter.default_sequence = True
            self.th_apply_filter.start()
            
        except Exception as e:
            error_msg = f"Errore avvio sequenza filtri: {str(e)}"
            self.elaboration_logger.error(error_msg)
            self.progress_bar.close()

    def _on_filter_completed(self):
        """Gestisce il completamento dell'elaborazione filtri"""
        try:
            self.elaboration_logger.info("Elaborazione filtri completata con successo")
            self.progress_bar.close()
            
            # Notifica altri controller per aggiornamento GUI
            # Il RadargramController è già connesso al segnale filter_completed
            print("ElaborationController: Filtri applicati, GUI verrà aggiornata automaticamente")
            
        except Exception as e:
            error_msg = f"Errore gestione completamento filtro: {str(e)}"
            self.elaboration_logger.error(error_msg)
            self.progress_bar.close()
    
    def _on_filter_error(self, error_msg):
        """Gestisce gli errori durante l'elaborazione filtri"""
        self.elaboration_logger.error(f"Errore durante l'elaborazione filtri: {error_msg}")
        self.progress_bar.close()
        
        # Mostra messaggio all'utente
        from PySide6.QtWidgets import QMessageBox
        QMessageBox.critical(self.view, "Errore Filtri", f"Errore durante l'applicazione dei filtri:\n{error_msg}")

    def _on_filter_progress(self, message):
        """Gestisce i messaggi di progresso durante l'applicazione dei filtri"""
        self.elaboration_logger.info(f"Progresso filtri: {message}")
        # La progress bar viene aggiornata automaticamente dal segnale

    # ------------------------ AI LAUNCHERS ------------------------
    def launch_ai(self, request: dict | None = None):
        """Starts the AI thread with provided request/config."""
        print("ElaborationController: launch_ai() called")
        if self.th_run_ai.isRunning():
            self.elaboration_logger.warning("AI elaboration already running, please wait...")
            return
        try:
            self.th_run_ai.set_request(request)
            print("ElaborationController: starting AI thread...")
            self.th_run_ai.start()
        except Exception as e:
            self._on_ai_error(str(e))

    def launch_ai_detection(self):
        """Starts the AI Detection pipeline (C++ via multiCompilation.py)"""
        if self.th_ai_detection.isRunning():
            self.elaboration_logger.warning("AI detection already running, please wait...")
            return
        try:
            self.progress_bar.show_progress_dialog("AI detection in progress...")
            self.th_ai_detection.start()
            # Ensure tiles generation starts as in Open Project:
            # - If inference tomography is not yet created, create it and start Raster_tile via Rendering.switch_tomography(True)
            # - Otherwise, (re)start the Raster_tile thread if not running
            try:
                from Thread.Rendering import Rendering
                rendering = Rendering()  # singleton
                # If first time, this call creates tomography_inferenza and starts tile thread
                if not getattr(rendering, 'already_created_tomography', False):
                    self.elaboration_logger.info("AI Detection: creating inference tomography and starting tiles...")
                    rendering.switch_tomography(True)
                else:
                    # Tomography already exists; just ensure tile thread is running
                    if hasattr(rendering, 'th_raster_tile') and not rendering.th_raster_tile.isRunning():
                        self.elaboration_logger.info("AI Detection: starting Raster_tile thread...")
                        rendering.th_raster_tile.start()
            except Exception as e:
                # Non-bloccante: la detection può restare in ascolto finché i tile non arriveranno
                self.elaboration_logger.error(f"AI Detection: unable to start tiles generation: {e}")
        except Exception as e:
            self._on_ai_error(str(e))

    def _on_ai_progress(self, message: str):
        print(f"ElaborationController: {message}")
        self.elaboration_logger.info(f"AI: {message}")
    
    def _on_ai_detection_started(self, message: str):
        print(f"[ElaborationController] AI Detection started: {message}")
        self._show_ai_detection_busy()
        self.elaboration_logger.info(f"AI Detection started: {message}")
    
    def _on_ai_detection_progress(self, message: str):
        print(f"[ElaborationController] AI Detection progress: {message}")
        self._show_ai_detection_busy()  # Reset timer su ogni progress
        self.elaboration_logger.info(f"AI Detection: {message}")
    
    def _on_ai_detection_completed(self, message: str):
        print(f"[ElaborationController] AI Detection completed: {message}")
        self.elaboration_logger.info(f"AI Detection completed: {message}")
        # Nascondi dopo 5 secondi di inattività (gestito dal timer)
    
    def _on_ai_detection_error(self, message: str):
        print(f"[ElaborationController] AI Detection error: {message}")
        self._hide_ai_detection_busy()
        self.elaboration_logger.error(f"AI Detection error: {message}")

    def _on_ai_completed(self, message: str):
        print(f"ElaborationController: {message}")
        self.elaboration_logger.info(f"AI completed: {message}")
        self._hide_ai_busy()
        
        # Avvio RunAIDetection per processare i tile completati
        print("[ElaborationController] Avvio RunAIDetection per processare tile completati...")
        try:
            if not self.th_ai_detection.isRunning():
                self.th_ai_detection.start()
                print("[ElaborationController] RunAIDetection avviato")
            else:
                print("[ElaborationController] RunAIDetection già in esecuzione")
                # Mostra comunque la busy bar perché potrebbe esserci attività
                self._show_ai_detection_busy()
        except Exception as e:
            print(f"[ElaborationController] Errore avvio RunAIDetection: {e}")
        
        # Reuse existing filter-completed signal to refresh radargrams across the app
        try:
            self.th_apply_filter.filter_completed.emit()
        except Exception:
            pass

    def _on_ai_error(self, message: str):
        self.elaboration_logger.error(f"AI error: {message}")
        self._hide_ai_busy()
        from PySide6.QtWidgets import QMessageBox
        QMessageBox.critical(self.view, "AI Error", f"AI elaboration failed:\n{message}")

    def _on_ai_checkbox_changed(self, state):
        # Keep minimal behavior to avoid double-launch with toggled
        print(f"ElaborationController: AI checkbox changed -> state={state}")

    def _connect_ai_checkbox_if_available(self):
        if self._ai_checkbox_connected:
            return
        try:
            if hasattr(self.view, 'checkbox_ai_elaboration'):
                self.view.checkbox_ai_elaboration.toggled.connect(self._on_ai_toggled)
                self._ai_checkbox_connected = True
                print("ElaborationController: wired AI elaboration checkbox")
            if hasattr(self.view, 'checkbox_ai_detection'):
                # Launch AI detection on check; ignore uncheck
                self.view.checkbox_ai_detection.stateChanged.connect(lambda s: self.launch_ai_detection() if s else None)
                print("ElaborationController: wired AI detection checkbox")
        except Exception:
            pass

    def _on_ai_toggled(self, checked: bool):
        print(f"ElaborationController: AI checkbox toggled -> {checked}")
        if checked:
            self._show_ai_busy()
            self.launch_ai({})
        else: # dico al map controller di ritornare 
            self.th_run_ai.ai_switch_tomo.emit()

    def _connect_inference_view_checkbox_if_available(self):
        try:
            import GLOBAL as G
            if not getattr(G, 'open_project_mode', False) and hasattr(self.view, 'checkbox_inference_view'):
                # Se non Open Project, nascondi il checkbox
                self.view.checkbox_inference_view.setVisible(False)
                return
            if hasattr(self.view, 'checkbox_inference_view'):
                self.view.checkbox_inference_view.setVisible(True)
                self.view.checkbox_inference_view.toggled.connect(self._on_inference_view_toggled)
        except Exception:
            pass

    def _connect_show_AI_data_checkbox(self):
        """Collega il checkbox per visualizzare dati AI nel radargram - DISATTIVATO
        Il checkbox è ora gestito direttamente da RadargramController
        """
        print("[ElaborationController] Checkbox 'Show AI data' gestito da RadargramController")
        pass

    def _connect_rebuild_inference_button(self):
        """Collega il bottone rebuild inference pipeline"""
        try:
            if hasattr(self.view, 'btn_rebuild_inference'):
                self.view.btn_rebuild_inference.clicked.connect(self.rebuild_inference_pipeline)
                self.elaboration_logger.info("Rebuild inference button connected")
        except Exception as e:
            self.elaboration_logger.warning(f"Failed to connect rebuild inference button: {e}")

    def _stop_all_rebuild_processes(self):
        """Ferma tutti i processi di rebuild attivi (RunAI, Raster_tile, RunAIDetection)"""
        try:
            print("[STOP] Fermo tutti i processi di rebuild...")
            
            # 1. Ferma RunAI
            if self.th_run_ai.isRunning():
                print("[STOP] Interrompo RunAI...")
                self.th_run_ai.terminate()
                self.th_run_ai.wait(5000)  # Attendi fino a 5 secondi
                if self.th_run_ai.isRunning():
                    print("[STOP] RunAI ancora in esecuzione, forzo quit")
                    self.th_run_ai.quit()
                print("[STOP] RunAI fermato")
                
                # PULISCI CONTESTO CUDA per evitare deadlock
                try:
                    import torch
                    if torch.cuda.is_available():
                        print("[STOP] Pulizia cache CUDA...")
                        torch.cuda.empty_cache()
                        print("[STOP] ✓ Cache CUDA pulita")
                except Exception as e:
                    print(f"[STOP] Warning: errore pulizia CUDA: {e}")
            
            # 2. Ferma Raster_tile
            if self.th_raster_tile.isRunning():
                print("[STOP] Interrompo Raster_tile...")
                self.th_raster_tile.terminate()
                self.th_raster_tile.wait(5000)
                if self.th_raster_tile.isRunning():
                    print("[STOP] Raster_tile ancora in esecuzione, forzo quit")
                    self.th_raster_tile.quit()
                print("[STOP] Raster_tile fermato")
            
            # 3. Ferma RunAIDetection
            if self.th_ai_detection.isRunning():
                print("[STOP] Interrompo RunAIDetection...")
                self.th_ai_detection.terminate()
                self.th_ai_detection.wait(5000)
                if self.th_ai_detection.isRunning():
                    print("[STOP] RunAIDetection ancora in esecuzione, forzo quit")
                    self.th_ai_detection.quit()
                print("[STOP] RunAIDetection fermato")
            
            # 4. Disconnetti tutti i signal handler temporanei
            try:
                self.th_run_ai.ai_completed.disconnect()
                print("[STOP] Signal handler RunAI disconnesso")
            except Exception:
                pass
            
            try:
                self.th_raster_tile.raster_tile_completed.disconnect()
                print("[STOP] Signal handler Raster_tile disconnesso")
            except Exception:
                pass
            
            print("[STOP] Tutti i processi fermati e puliti")
            
        except Exception as e:
            print(f"[STOP] Errore durante stop processi: {e}")
            self.elaboration_logger.error(f"Error stopping processes: {e}")

    def rebuild_inference_pipeline(self):
        """
        Rebuild inference tomography and tiles with updated AI models.
        STOPS all running processes and restarts from scratch.
        This method:
        1. Stops all running AI/tiles/detection processes
        2. Runs AI inference with new models on Data_current_filter
        3. Updates Data_AI and refreshes radargram
        4. Regenerates all tiles with new inference data
        5. Reprocesses multiPHT on all tiles
        """
        try:
            self.elaboration_logger.info("=== REBUILD INFERENCE PIPELINE STARTED ===")
            print("\n" + "="*80)
            print("[REBUILD] REBUILD INFERENCE PIPELINE STARTED")
            print("="*80)
            
            # STEP 0: FERMA tutti i processi attivi
            print("[REBUILD] Fermo tutti i processi attivi...")
            self._stop_all_rebuild_processes()
            print("[REBUILD] Tutti i processi fermati")
            
            # Leggi percorsi modelli custom dalla GUI (se forniti)
            custom_paths = {}
            if hasattr(self.view, 'input_model_long') and self.view.input_model_long.text().strip():
                custom_paths['longitudinale'] = self.view.input_model_long.text().strip()
            if hasattr(self.view, 'input_model_trasv') and self.view.input_model_trasv.text().strip():
                custom_paths['trasversale'] = self.view.input_model_trasv.text().strip()
            if hasattr(self.view, 'input_model_manhole') and self.view.input_model_manhole.text().strip():
                custom_paths['manhole'] = self.view.input_model_manhole.text().strip()
            
            # Se ci sono percorsi custom, impostali in RunAI
            if custom_paths:
                print(f"[REBUILD] Imposto percorsi custom: {list(custom_paths.keys())}")
                # Carico i default e sovrascrivo con i custom
                default_paths = self.th_run_ai.default_model_paths.copy()
                default_paths.update(custom_paths)
                self.th_run_ai.set_model_paths(default_paths)
            else:
                print("[REBUILD] Uso percorsi modelli di default")
                # Resetta eventuali custom paths precedenti
                self.th_run_ai.request = None
            
            # Step 1: Run AI inference with new models
            self.elaboration_logger.info("Step 1: Running AI inference with new models...")
            print("[REBUILD] Step 1: Verifico thread RunAI...")
            print(f"[REBUILD] RunAI.isRunning() = {self.th_run_ai.isRunning()}")
            
            # Attendi un po' dopo terminate per permettere a CUDA di stabilizzarsi
            import time
            print("[REBUILD] Attendo 1 secondo per stabilizzare CUDA...")
            time.sleep(1)  # Pausa sincrona per dare tempo a CUDA
            
            if not self.th_run_ai.isRunning():
                print("[REBUILD] RunAI non in esecuzione, procedo con rebuild")
                # Connect to completion to trigger tiles generation
                try:
                    self.th_run_ai.ai_completed.disconnect()  # Remove old connections
                    print("[REBUILD] Disconnessi vecchi handler ai_completed")
                except Exception as e:
                    print(f"[REBUILD] Nessun handler precedente da disconnettere: {e}")
                
                self.th_run_ai.ai_completed.connect(self._on_rebuild_ai_completed)
                print("[REBUILD] Connesso handler _on_rebuild_ai_completed")
                print("[REBUILD] Avvio RunAI thread...")
                self.th_run_ai.start()
                print("[REBUILD] RunAI.start() chiamato")
                print(f"[REBUILD] RunAI.isRunning() dopo start = {self.th_run_ai.isRunning()}")
            else:
                self.elaboration_logger.warning("AI thread already running, cannot rebuild")
                print("[REBUILD] ERRORE: RunAI già in esecuzione, cannot rebuild")
                self.progress_bar.close()
                if hasattr(self.view, 'btn_rebuild_inference'):
                    self.view.btn_rebuild_inference.setEnabled(True)
                
        except Exception as e:
            self.elaboration_logger.error(f"Error in rebuild_inference_pipeline: {e}")
            print(f"[REBUILD] ERRORE ECCEZIONE: {e}")
            import traceback
            traceback.print_exc()
            self.progress_bar.close()
            if hasattr(self.view, 'btn_rebuild_inference'):
                self.view.btn_rebuild_inference.setEnabled(True)

    def _on_rebuild_ai_completed(self, message: str):
        """Called when AI inference completes during rebuild"""
        try:
            self.elaboration_logger.info(f"AI inference completed for rebuild: {message}")
            print("\n" + "="*80)
            print("[REBUILD] AI inference completata!")
            print(f"[REBUILD] Messaggio: {message}")
            print("="*80)
            
            # Ferma la progress bar piccola di AI
            print("[REBUILD] Fermo AI busy bar")
            self._hide_ai_busy()
            
            # AGGIORNA IMMEDIATAMENTE il radargram con i nuovi dati di Data_AI
            print("[REBUILD] Aggiorno radargram con nuovi dati AI...")
            try:
                if self.parent():
                    radargram_ctrl = getattr(self.parent(), 'radargramController', None)
                    if radargram_ctrl:
                        radargram_ctrl._global_abs_max = None
                        radargram_ctrl.on_filter_completed()
                        print("[REBUILD] Radargram aggiornato con Data_AI")
                    else:
                        print("[REBUILD] RadargramController non trovato")
                # Trigger anche via signal per sicurezza
                QTimer.singleShot(100, lambda: self._delayed_radargram_refresh())
            except Exception as e2:
                print(f"[REBUILD] Errore refresh radargram: {e2}")
                QTimer.singleShot(100, lambda: self._delayed_radargram_refresh())
            
            # Step 2: Start RunAIDetection thread (must be running before tiles)
            print("[REBUILD] Step 2: Avvio RunAIDetection...")
            print(f"[REBUILD] RunAIDetection.isRunning() = {self.th_ai_detection.isRunning()}")
            if not self.th_ai_detection.isRunning():
                self.th_ai_detection.start()
                print("[REBUILD] RunAIDetection.start() chiamato")
            else:
                print("[REBUILD] RunAIDetection già in esecuzione, continuo...")
            
            # Step 3: Crea prima la tomografia di inferenza
            print("[REBUILD] Step 3a: Creo tomografia di inferenza...")
            try:
                from Thread.Rendering import Rendering
                rendering = Rendering()
                # Forza la creazione della tomografia di inferenza
                print("[REBUILD] Chiamo rendering.switch_tomography(True)...")
                rendering.switch_tomography(True)
                print("[REBUILD] Tomografia di inferenza creata")
            except Exception as e:
                print(f"[REBUILD] Errore creazione tomografia inferenza: {e}")
                import traceback
                traceback.print_exc()
            
            # Step 3b: Regenerate all tiles
            print("[REBUILD] Step 3b: Regenero tiles...")
            self.elaboration_logger.info("Step 2: Regenerating tiles with new inference data...")
            print(f"[REBUILD] Raster_tile.isRunning() = {self.th_raster_tile.isRunning()}")
            
            if not self.th_raster_tile.isRunning():
                print("[REBUILD] Disconnetti vecchi handler raster_tile_completed")
                try:
                    self.th_raster_tile.raster_tile_completed.disconnect()
                    print("[REBUILD] Disconnessi vecchi handler")
                except Exception as e:
                    print(f"[REBUILD] Nessun handler precedente: {e}")
                
                print("[REBUILD] Connetto handler _on_rebuild_tiles_completed")
                self.th_raster_tile.raster_tile_completed.connect(self._on_rebuild_tiles_completed)
                print("[REBUILD] Avvio Raster_tile thread...")
                self.th_raster_tile.start()
                print("[REBUILD] Raster_tile.start() chiamato")
            else:
                print("[REBUILD] ERRORE: Raster_tile già in esecuzione")
            
            # Reconnect normal AI completion handler
            print("[REBUILD] Ricollego handler AI normale")
            self.th_run_ai.ai_completed.disconnect()
            self.th_run_ai.ai_completed.connect(lambda msg: self._on_ai_completed(msg))
            
        except Exception as e:
            self.elaboration_logger.error(f"Error in _on_rebuild_ai_completed: {e}")
            self.progress_bar.close()
            if hasattr(self.view, 'btn_rebuild_inference'):
                self.view.btn_rebuild_inference.setEnabled(True)

    def _on_rebuild_tiles_completed(self):
        """Called when tiles regeneration completes during rebuild"""
        try:
            self.elaboration_logger.info("=== REBUILD INFERENCE PIPELINE COMPLETED ===")
            print("\n" + "="*80)
            print("[REBUILD] REBUILD INFERENCE PIPELINE COMPLETATO!")
            print("="*80)
            print("[REBUILD] Ricollego handler normale per tiles")
            
            # Reconnect normal tiles completion handler
            try:
                self.th_raster_tile.raster_tile_completed.disconnect()
                print("[REBUILD] Disconnessi handler rebuild")
            except Exception as e:
                print(f"[REBUILD] Errore disconnect: {e}")
            
            self.th_raster_tile.raster_tile_completed.connect(self.raster_tile_finished)
            print("[REBUILD] Handler normale riconnesso")
            
            print("[REBUILD] Rebuild completato")
            
            print("[REBUILD] Bottone già cliccabile (sempre abilitato)")
            
            # Refresh radargram display
            print("[REBUILD] Aggiorno visualizzazione radargram...")
            # Trigger refresh tramite MainController/RadargramController direttamente
            try:
                if self.parent():
                    radargram_ctrl = getattr(self.parent(), 'radargramController', None)
                    if radargram_ctrl:
                        radargram_ctrl._global_abs_max = None
                        radargram_ctrl.on_filter_completed()
                        print("[REBUILD] Radargram aggiornato tramite RadargramController")
                    else:
                        print("[REBUILD] RadargramController non trovato")
            except Exception as e:
                print(f"[REBUILD] Errore refresh radargram: {e}")
            
            # Show completion message
            print("[REBUILD] Mostro messaggio di completamento")
            from PySide6.QtWidgets import QMessageBox
            QMessageBox.information(
                self.view,
                "Rebuild Complete",
                "Inference pipeline rebuilt successfully with new models.\n"
                "All tiles have been regenerated and reprocessed.\n"
                "Radargram display has been refreshed."
            )
            print("[REBUILD] Messaggio mostrato - REBUILD COMPLETATO!")
            print("="*80 + "\n")
            
        except Exception as e:
            self.elaboration_logger.error(f"Error in _on_rebuild_tiles_completed: {e}")
            print(f"[REBUILD] ERRORE in _on_rebuild_tiles_completed: {e}")
            import traceback
            traceback.print_exc()
            if hasattr(self.view, 'btn_rebuild_inference'):
                self.view.btn_rebuild_inference.setEnabled(True)
                print("[REBUILD] Bottone riabilitato dopo errore")

    def _on_inference_view_toggled(self, checked: bool):
        # Toggle tra tomografia originale e inferenza a livello di rendering
        try:
            from Controller.Main.MapController import MapController  # not used directly here
        except Exception:
            pass
        # Reuse Rendering thread switch
        try:
            # Quando checked=True, usa inferenza; False -> originale
            from Controller.Main.MapController import MapController as _
            # Usa il Rendering singleton già istanziato nel MapController
            # Emuliamo lo switch chiamando il metodo sul thread di rendering attraverso MapController se disponibile
            # In mancanza, inviamo direttamente il segnale di switch usato dall'AI
            if checked:
                # abilita tomografia inferenza
                # set flag e chiedi al Rendering di cambiare modalità
                # Non generiamo tomografie qui, solo switch della cache/raster
                pass
            # Emetto il segnale già esistente usato dal toggle AI per forzare lo switch
            self.th_run_ai.ai_switch_tomo.emit()
        except Exception:
            # fallback: nessuna azione critica se non disponibile
            pass

    def _on_ai_started(self, message: str):
        # ensure busy bar visible also if started programmatically
        self._show_ai_busy()

    def _ensure_ai_busy(self):
        if self._ai_busy_bar is not None:
            return
        self._ai_busy_bar = QProgressBar()
        self._ai_busy_bar.setRange(0, 0)  # indeterminate
        self._ai_busy_bar.setTextVisible(False)
        self._ai_busy_bar.setFixedHeight(12)
        self._ai_busy_bar.setFixedWidth(80)
        self._ai_busy_bar.setVisible(False)

    def _show_ai_busy(self):
        self._ensure_ai_busy()
        if hasattr(self.view, 'checkbox_ai_elaboration'):
            cb = self.view.checkbox_ai_elaboration
            parent = cb.parent()
            lay = parent.layout() if parent is not None else None
            if lay is not None:
                idx = lay.indexOf(cb)
                if idx != -1 and lay.indexOf(self._ai_busy_bar) == -1:
                    # Insert right after checkbox to keep label text visible
                    lay.insertWidget(idx + 1, self._ai_busy_bar)
            cb.setEnabled(False)
        self._ai_busy_bar.setVisible(True)

    def _hide_ai_busy(self):
        if self._ai_busy_bar is None:
            return
        self._ai_busy_bar.setVisible(False)
        if hasattr(self.view, 'checkbox_ai_elaboration'):
            cb = self.view.checkbox_ai_elaboration
            cb.setEnabled(True)
    
    def _ensure_ai_detection_busy(self):
        if self._ai_detection_busy_bar is not None:
            return
        self._ai_detection_busy_bar = QProgressBar()
        self._ai_detection_busy_bar.setRange(0, 0)  # indeterminate
        self._ai_detection_busy_bar.setTextVisible(False)
        self._ai_detection_busy_bar.setFixedHeight(12)
        self._ai_detection_busy_bar.setFixedWidth(80)
        self._ai_detection_busy_bar.setVisible(False)

    def _show_ai_detection_busy(self):
        self._ensure_ai_detection_busy()
        # Troviamo la checkbox "AI Detection" e inseriamo la barra dopo
        if hasattr(self.view, 'checkbox_ai_detection'):
            cb = self.view.checkbox_ai_detection
            parent = cb.parent()
            lay = parent.layout() if parent is not None else None
            if lay is not None:
                idx = lay.indexOf(cb)
                if idx != -1 and lay.indexOf(self._ai_detection_busy_bar) == -1:
                    # Insert right after checkbox
                    lay.insertWidget(idx + 1, self._ai_detection_busy_bar)
            cb.setEnabled(False)
        self._ai_detection_busy_bar.setVisible(True)
        print("[ElaborationController] ✓ AI Detection busy bar SHOWN")
        
        # Reset timer: nascondi dopo 5 secondi di inattività
        if self._ai_detection_busy_timer is not None:
            self._ai_detection_busy_timer.stop()
        self._ai_detection_busy_timer = QTimer()
        self._ai_detection_busy_timer.timeout.connect(self._hide_ai_detection_busy)
        self._ai_detection_busy_timer.setSingleShot(True)
        self._ai_detection_busy_timer.start(5000)  # 5 secondi

    def _hide_ai_detection_busy(self):
        if self._ai_detection_busy_bar is None:
            return
        self._ai_detection_busy_bar.setVisible(False)
        if hasattr(self.view, 'checkbox_ai_detection'):
            cb = self.view.checkbox_ai_detection
            cb.setEnabled(True)
        print("[ElaborationController] ✓ AI Detection busy bar HIDDEN")
        
        # Stop timer
        if self._ai_detection_busy_timer is not None:
            self._ai_detection_busy_timer.stop()
            self._ai_detection_busy_timer = None

