# controllers/MainController.py
# IMPORT PYSIDE6
from PySide6.QtCore import QObject
from PySide6.QtWidgets import QFileDialog

# MODELLI
from Model.SurveyManager import SurveyManager

# THREAD
from Thread.ReadFile import ReadFile
from Thread.ApplyFilter import ApplyFilter

# VIEW
from View.ProgressBar import ProgressBar
from PySide6.QtWidgets import QPushButton, QHBoxLayout, QWidget, QSizePolicy
from PySide6.QtWidgets import QMessageBox
import os

from Model.ProjectManifest import write_external_project
from GLOBAL import tempData

class MainController(QObject):

    def __init__(self, view, parent=None):
        super().__init__(parent)

        self.view = view
        self.survey_manager = SurveyManager()  # recupero il gestore dei survey

        # IMPORT SURVEY
        self.view.import_survey_action.triggered.connect(self.import_survey)
        self.th_readFile = ReadFile()
        self.th_readFile.finish_read.connect(self.read_file_finished)

        # Progress bar per operazioni import
        self.progress_bar = ProgressBar(self.view)

        # --- Simple page switcher (Overview / Algorithm) ---
        self._setup_page_switcher()

        # SAVE PROJECT (toolbar button)
        try:
            self.view.save.clicked.connect(self.save_project)
        except Exception:
            pass


    def import_survey(self):
        survey_folder = QFileDialog.getExistingDirectory(self.view, "Select Directory")
        if survey_folder:
            self.th_readFile.original_survey_folder = survey_folder
            self.th_readFile.start()
            # self.progress_bar.show_progress_dialog("")
            self.progress_bar.show_progress_dialog("Caricamento file in corso...")

    # -------------------- PAGES --------------------
    def _setup_page_switcher(self):
        # Create two small buttons in the toolbar to switch pages
        try:
            # Center the page-switcher in the toolbar using expanding spacers
            self.btn_overview = QPushButton("Overview", self.view)
            self.btn_algorithm = QPushButton("Algorithm", self.view)
            self.btn_overview.setCheckable(True)
            self.btn_algorithm.setCheckable(True)
            self.btn_overview.setChecked(True)

            # Container widget for the two buttons
            container = QWidget()
            layout = QHBoxLayout(container)
            layout.setContentsMargins(8, 0, 0, 0)
            layout.setSpacing(6)
            layout.addWidget(self.btn_overview)
            layout.addWidget(self.btn_algorithm)
            # left spacer
            left_spacer = QWidget()
            left_spacer.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
            # right spacer
            right_spacer = QWidget()
            right_spacer.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
            self.view.toolbar.addWidget(left_spacer)
            self.view.toolbar.addWidget(container)
            self.view.toolbar.addWidget(right_spacer)

            # Wire buttons
            self.btn_overview.clicked.connect(self._show_overview)
            self.btn_algorithm.clicked.connect(self._show_algorithm)
        except Exception:
            pass

    def _show_overview(self):
        self.btn_overview.setChecked(True)
        self.btn_algorithm.setChecked(False)
        # Rimaniamo sulla stessa pagina e mostriamo il pannello Filtri (index 0)
        try:
            if hasattr(self.view, 'algorithm_stack'):
                self.view.algorithm_stack.setCurrentIndex(0)
            # Se per qualche motivo esistesse ancora la seconda pagina, forziamo index 0
            if hasattr(self.view, 'pages'):
                self.view.pages.setCurrentIndex(0)
        except Exception:
            pass

    def _show_algorithm(self):
        self.btn_overview.setChecked(False)
        self.btn_algorithm.setChecked(True)
        # Rimaniamo sulla stessa pagina e mostriamo il pannello Elaboration (index 1)
        try:
            if hasattr(self.view, 'algorithm_stack'):
                self.view.algorithm_stack.setCurrentIndex(1)
            # Assicuriamoci comunque che la pagina visibile sia la principale
            if hasattr(self.view, 'pages'):
                self.view.pages.setCurrentIndex(0)
        except Exception:
            pass

    def read_file_finished(self, check_read):
        if check_read:
            print("MainController: File caricato con successo")
            # Seleziona il survey appena importato
            new_survey_id = len(self.survey_manager) - 1
            self.survey_manager.selected = new_survey_id
            self.progress_bar.close()
            
            print("MainController: Survey caricato")
            
        else:
            print("MainController: Errore nel caricamento del file")
            self.progress_bar.close()

    def save_project(self):
        # Ask for destination folder; copy TempData and write project.json
        dest_dir = QFileDialog.getExistingDirectory(self.view, "Select destination folder for project")
        if not dest_dir:
            return
        try:
            project_json = write_external_project(dest_dir, tempData, self.survey_manager)
            QMessageBox.information(self.view, "Save Project", f"Project saved:\n{project_json}")
        except Exception as e:
            QMessageBox.critical(self.view, "Save Project", f"Error while saving project: {e}")




