"""
RadargramController - Gestisce la visualizzazione del radargramma e la selezione swath/channel.

Funzionalità:
- Caricamento e visualizzazione radargrammi tramite SurveyManager
- Gestione combo box swath e channel
- Cambio canale/swath
- Aggiornamento diretto del plot_widget
- Accesso standardizzato ai dati tramite pattern SurveyManager originale
"""
# PYSIDE6
from PySide6.QtCore import QObject, Signal
import pyqtgraph as pg
import numpy as np
import os

# IMPORT CHANGED
from Controller.Main.Changed import ChangedController

# MODELLI
from Model.SurveyManager import SurveyManager

# THREAD
from Thread.ApplyFilter import ApplyFilter

# LOG
import logging

class RadargramController(QObject):
    """
    Controller per gestire la visualizzazione dei radargrammi e la selezione swath/channel.
    Tutti i dati vengono acceduti tramite il pattern SurveyManager originale standardizzato.
    Si occupa solo di visualizzazione e navigazione dei dati originali.
    """
    
    def __init__(self, view, parent=None):
        super().__init__(parent)
        
        # Riferimento alla view per accedere al plot_widget e combo box
        self.view = view
        
        self.survey_manager = SurveyManager()   # recupero il gestore dei survey
        self.radargram_logger = logging.getLogger('radargram')  # recupero il logger corretto

        # Visualizza sempre i dati filtrati se presenti
        self.show_filtered_data = True
        # Cache for global normalization
        self._global_abs_max = None

        # NOTIFY CHANGED
        self.changed_controller = ChangedController()
        self.changed_controller.survey_changed.connect(self.populate_combo_box)

        # THREAD DI APPLICAZIONE FILTRI
        self.th_apply_filter = ApplyFilter()
        self.th_apply_filter.filter_completed.connect(self.on_filter_completed)  # Aggiorna quando il filtro è completato

        # GESTIONE DELLE COMBO BOX PER LA VISUALIZZAZIONE DEL RADARGRAMMA
        self.populate_combo_box() # DI DEFAULT PRIMO SURVEY - PRIMA SWATH
        self.view.combo_swath.currentIndexChanged.connect(self.swath_changed)
        self.view.combo_channel.currentIndexChanged.connect(self.channel_changed)
        # CLIP SMALL (range simmetrico [-v:+v])
        self.view.clip_slider_small.valueChanged.connect(self.clip_small_changed)
        
        # H-SCALE SLIDER (zoom/dezoom orizzontale)
        self.hscale_factor: float = 1.0
        self._last_hscale_factor: float = 1.0
        try:
            self.view.hscale_value_label.setText(f"{self.view.hscale_slider.value()}%")
            self.view.hscale_slider.valueChanged.connect(self.hscale_changed)
        except Exception:
            # In caso lo slider non sia presente nella view
            pass
        
        # GESTIONE DELLO SLIDER PER LE SUBSWATH
        self.view.subswath_slider.valueChanged.connect(self.subswath_changed)
        
        # COLLEGA CHECKBOX "Show AI data" DIRETTAMENTE
        self._connect_show_AI_data_checkbox()
        
        # Carico il primo radargramma
        self.refresh_radargram()

        # CONNESSIONE PER IL CAMBIO DI POLARIZZAZIONE
        self.view.polarization_combo.currentTextChanged.connect(self.polarization_changed)

    # POPOLO I COMBO BOX CON NUMERO DI SWATH E NUMERO DI CANALI DISPONIBILI | CON IL SURVEY SELEZIONATO
    def populate_combo_box(self):
        # ripulisco il combo-box swath e channel
        self.view.combo_swath.clear()
        self.view.combo_channel.clear()
        self.radargram_logger.info("Popolo i combo-box")
        # Modifico il combo-box SWATH
        for s in range(self.survey_manager.get_num_tot_swath()):
            self.view.combo_swath.addItem(f"{s}")
        # Modifico il combo-box CHANNEL
        for ch in range(self.survey_manager.get_num_tot_ch()):
            self.view.combo_channel.addItem(f"{ch}")
        
    # FUNZIONI PER IL CAMBIO RADARGRAMMA -------------------------------------------------------------------
    def swath_changed(self, index):
        self.radargram_logger.info(f"Cambio swath: {index}")
        self.changed_controller.emit_swath_changed(index) # notifica il cambio di swath
        index_channel = self.view.combo_channel.currentIndex()
        self.channel_changed(index_channel)
    def channel_changed(self, index):
        self.radargram_logger.info(f"Cambio channel: {index}")
        self.changed_controller.emit_channel_changed(index) # notifica il cambio di channel
        self.refresh_radargram()    
    
    def subswath_changed(self, index):
        self.radargram_logger.info(f"Cambio subswath: {index}")
        self.survey_manager.set_selected_subswath(index)
        self.refresh_radargram()

    def refresh_radargram(self):
        """Aggiorna la visualizzazione del radargramma usando i dati dal SurveyManager"""
        self.radargram_logger.info(f"Aggiorna il radargramma su: {self.survey_manager.get_id_selected_survey()} - {self.survey_manager.get_id_selected_swath()} - {self.survey_manager.get_id_selected_subswath()} - {self.survey_manager.get_id_selected_channel()}")
        
        try:
            # Allinea la polarizzazione al valore selezionato nella UI, evitando mismatch VV/HH
            try:
                ui_pol_text = self.view.polarization_combo.currentText()
                ui_pol = 1 if ui_pol_text == "HH" else 0
                self.survey_manager.set_current_polarization(ui_pol)
            except Exception:
                pass

            # Carica sempre i dati filtrati se presenti, altrimenti fallback agli originali
            raw_data = self.get_display_data()
            data = raw_data
            # Normalizza globalmente in [-1, 1] considerando tutto il progetto (calcolo locale)
            # gmax = self._get_global_abs_max()
            # if gmax > 0 and np.isfinite(gmax):
            #     data = np.clip(raw_data / gmax, -1.0, 1.0)
            # Fallback robusto: se dopo la normalizzazione i dati risultano tutti zero o non finiti,
            # usa una normalizzazione locale sul valore massimo assoluto del dato corrente
            try:
                max_abs_after = float(np.nanmax(np.abs(data)))
                if not np.isfinite(max_abs_after) or max_abs_after == 0.0:
                    local_abs_max = float(np.nanmax(np.abs(raw_data)))
                    if np.isfinite(local_abs_max) and local_abs_max > 0:
                        data = np.clip(raw_data / local_abs_max, -1.0, 1.0)
            except Exception:
                pass
            #print(f"data originale: {data.shape}")
            # Ruoto di 90° orario per mettere la parte sinistra in alto
            data = np.rot90(data, k=-1)

            num_tot_subswath = self.survey_manager.get_num_tot_subswath()
            current_subswath = self.survey_manager.get_id_selected_subswath()

            # Aggiorna lo slider delle subswath
            self.view.subswath_slider.blockSignals(True)
            self.view.subswath_slider.setMaximum(num_tot_subswath - 1)
            self.view.subswath_slider.setValue(current_subswath)
            self.view.subswath_slider.blockSignals(False)

            # (Rimosso) clipping percentile slider grande
            # Applica clipping simmetrico in ampiezza (slider piccolo)
            if self.view.clip_slider_small.value() > 0:
                max_abs = float(np.nanmax(np.abs(data)))
                v = (self.view.clip_slider_small.value() / 1000.0) * max_abs
                if v > 0:
                    data = np.clip(data, -v, +v)
                self.view.clip_value_label.setText(f"[ { -v: .3f} : { v: .3f} ]")
            else:
                max_abs = float(np.nanmax(np.abs(data)))
                self.view.clip_value_label.setText(f"[ { -max_abs: .3f} : { max_abs: .3f} ]")
            
            # Aggiorna la visualizzazione
            self.view.plot_widget.clear()
            image_item = pg.ImageItem()
            image_item.setAutoDownsample(True)
            image_item.setImage(data)
            
            # Blocca la navigazione e fa occupare tutto lo spazio
            self.view.plot_widget.setMouseEnabled(x=False, y=False)
            self.view.plot_widget.hideButtons()
            self.view.plot_widget.setMenuEnabled(False)

            self.view.plot_widget.addItem(image_item)
            
            # Centra e applica zoom/dezoom orizzontale tramite ViewBox
            vb = self.view.plot_widget.getViewBox()
            vb.setAspectLocked(False)
            width = float(data.shape[0])
            scale = max(0.10, float(self.hscale_factor))
            x_len = width / scale
            x_center = width / 2.0
            x_half = x_len / 2.0
            # Do NOT clamp: allow range to extend beyond [0, width] to achieve centered dezoom
            x_min = x_center - x_half
            x_max = x_center + x_half
            vb.setXRange(x_min, x_max, padding=0.0)
            vb.setYRange(0.0, float(data.shape[1]), padding=0.0)
            
        except Exception as e:
            error_msg = f"Errore durante l'aggiornamento del radargramma: {str(e)}"
            self.radargram_logger.error(error_msg)
            print(f"RadargramController: {error_msg}")

    def on_filter_completed(self, message: str = ""):
        """Gestisce il completamento dell'applicazione dei filtri o rebuild"""
        if "Rebuild" in message:
            print("RadargramController: Rebuild completato, aggiorno radargramma")
            self.radargram_logger.info("Rebuild completato, refresh radargramma")
        else:
            print("RadargramController: Filtri completati, aggiorno radargramma con dati filtrati")
            self.radargram_logger.info("Filtri completati, refresh radargramma")
        
        # Invalida la cache del max globale perché i dati sono cambiati
        self._global_abs_max = None
        
        # Forza l'aggiornamento del radargramma
        # Il refresh_radargram() leggerà automaticamente il checkbox "Show AI data"
        # e caricherà i dati da Data_AI o Data_current_filter di conseguenza
        self.refresh_radargram()
    
    def hscale_changed(self, value: int):
        """Aggiorna lo zoom orizzontale (10% - 100%) senza lavoro inutile."""
        # Sanitize and clamp slider value
        try:
            v_int = int(value)
        except Exception:
            v_int = 100
        v_int = 10 if v_int < 10 else 100 if v_int > 100 else v_int

        # Convert to factor
        factor = float(v_int) / 100.0

        # Update label regardless
        try:
            self.view.hscale_value_label.setText(f"{v_int}%")
        except Exception:
            pass

        # Skip refresh if unchanged
        if abs(factor - self._last_hscale_factor) < 1e-6:
            return

        self.hscale_factor = factor
        self._last_hscale_factor = factor
        self.refresh_radargram()
    
    def clip_small_changed(self, value):
        """Aggiorna la visualizzazione quando cambia il clip simmetrico [-v:+v]."""
        try:
            self.radargram_logger.info(f"Clip range simmetrico slider: {value}")
        except Exception:
            pass
        self.refresh_radargram()
        
        # Verifica se i dati filtrati esistono (controllo file)
        try:
            current_subswath_id = self.survey_manager.get_id_selected_subswath()
            current_channel_id = self.survey_manager.get_id_selected_channel()
            current_survey_id = self.survey_manager.get_id_selected_survey()
            current_swath_id = self.survey_manager.get_id_selected_swath()

            subswath = self.survey_manager[current_survey_id][current_swath_id][current_subswath_id]
            filtered_path = os.path.join(subswath.data_current_filter_folder, f"channel_{current_channel_id:02d}.npy")
            if os.path.exists(filtered_path):
                print(f"RadargramController: Dati filtrati disponibili per canale {current_channel_id}")
            else:
                print(f"RadargramController: Nessun dato filtrato per canale {current_channel_id}")
        except Exception as e:
            print(f"RadargramController: Errore verifica dati filtrati: {str(e)}")

    def get_display_data(self):
        """Carica dati filtrati se presenti, altrimenti dati originali."""
        try:
            current_survey_id = self.survey_manager.get_id_selected_survey()
            current_swath_id = self.survey_manager.get_id_selected_swath()
            current_subswath_id = self.survey_manager.get_id_selected_subswath()
            current_channel_id = self.survey_manager.get_id_selected_channel()

            subswath = self.survey_manager[current_survey_id][current_swath_id][current_subswath_id]

            # Determina quale data folder usare (dal checkbox in MainWindow)
            use_AI_data = False
            checkbox_state = "not found"
            if hasattr(self.view, 'checkbox_show_AI_data'):
                if hasattr(self.view.checkbox_show_AI_data, 'isChecked'):
                    use_AI_data = self.view.checkbox_show_AI_data.isChecked()
                    checkbox_state = "checked" if use_AI_data else "unchecked"
                else:
                    checkbox_state = "no isChecked method"
            else:
                checkbox_state = "view has no checkbox_show_AI_data"
            
            print(f"[RadargramController] Carico dati per canale {current_channel_id}")
            print(f"[RadargramController] Checkbox state: {checkbox_state}")
            print(f"[RadargramController] use_AI_data = {use_AI_data}")
            
            # Se checkbox attivo, prova prima Data_AI, poi Data_current_filter, poi Data_default
            if use_AI_data and hasattr(subswath, 'data_AI_folder'):
                ai_path = os.path.join(subswath.data_AI_folder, f"channel_{current_channel_id:02d}.npy")
                print(f"[RadargramController] Controllo Data_AI: {ai_path}")
                if os.path.exists(ai_path):
                    print(f"[RadargramController] ✓ Carico dati AI: {ai_path}")
                    return np.load(ai_path)
                print(f"[RadargramController] ✗ Data_AI non disponibile, fallback a Data_current_filter")

            # Prova prima i dati filtrati (Data_current_filter)
            filtered_path = os.path.join(subswath.data_current_filter_folder, f"channel_{current_channel_id:02d}.npy")
            print(f"[RadargramController] Controllo Data_current_filter: {filtered_path}")
            if os.path.exists(filtered_path):
                print(f"[RadargramController] ✓ Carico dati FILTRATI: {filtered_path}")
                return np.load(filtered_path)
            # Fallback agli originali (Data_default)
            print(f"RadargramController: Carico dati ORIGINALI per canale {current_channel_id}")
            return subswath.channel(current_channel_id)

        except Exception as e:
            print(f"RadargramController: Errore caricamento dati, fallback a SurveyManager: {str(e)}")
            return self.survey_manager.get_channel_data()
    # -----------------------------------------------------------------------------------------------------

    def _get_global_abs_max(self) -> float:
        """Compute and cache the global absolute max across all radargram files in the project.
        Prefers Data_current_filter files; falls back to Data_default.
        """
        if self._global_abs_max is not None:
            return self._global_abs_max

        max_abs_value: float = 0.0
        try:
            for survey in self.survey_manager.survey_list:
                for pol in (0, 1):  # 0=VV, 1=HH
                    survey.type_polarization = pol
                    try:
                        num_swath = survey.get_num_tot_swath()
                    except Exception:
                        num_swath = 0
                    for sw_idx in range(num_swath):
                        swath = survey[sw_idx]
                        try:
                            num_sub = swath.get_num_tot_subswath()
                        except Exception:
                            num_sub = 0
                        for sub_idx in range(num_sub):
                            subswath = swath[sub_idx]
                            # Prefer filtered
                            filtered_files = []
                            try:
                                for fn in os.listdir(subswath.data_current_filter_folder):
                                    if fn.endswith('.npy'):
                                        filtered_files.append(os.path.join(subswath.data_current_filter_folder, fn))
                            except Exception:
                                pass
                            file_list = filtered_files if filtered_files else getattr(subswath, 'file_subswath_channel_list', [])
                            for file_path in file_list:
                                try:
                                    arr = np.load(file_path, mmap_mode='r')
                                    local_max = float(np.nanmax(np.abs(arr)))
                                    if local_max > max_abs_value:
                                        max_abs_value = local_max
                                except Exception:
                                    continue
        except Exception:
            pass

        if not np.isfinite(max_abs_value) or max_abs_value <= 0:
            max_abs_value = 1.0
        self._global_abs_max = max_abs_value
        return self._global_abs_max

    def polarization_changed(self, polarization):
        self.radargram_logger.info(f"Cambio polarizzazione: {polarization}")
        self.changed_controller.emit_polarization_changed(1 if polarization == "HH" else 0) # notifica il cambio di polarizzazione
        self.survey_manager.set_selected_subswath(0)
        # Blocco i segnali delle combo box
        self.view.combo_swath.blockSignals(True)
        self.view.combo_channel.blockSignals(True)
        # popolo i combo-box con i dati del nuovo survey
        self.populate_combo_box()
        # Riattivo i segnali
        self.view.combo_swath.blockSignals(False)
        self.view.combo_channel.blockSignals(False)
        # Aggiorno manualmente il radargramma
        self.refresh_radargram()

    def _connect_show_AI_data_checkbox(self):
        """Collega il checkbox 'Show AI data' a questo controller per toggle immediato"""
        from PySide6.QtCore import QTimer
        print("[RadargramController] Provo a collegare checkbox 'Show AI data'...")
        try:
            # Usa QTimer per attendere che la view sia completamente inizializzata
            # Prova multipla con timeout crescenti
            def try_connect(retry_count=0):
                try:
                    if hasattr(self.view, 'checkbox_show_AI_data'):
                        self.view.checkbox_show_AI_data.toggled.connect(self._on_show_AI_data_toggled)
                        print("[RadargramController] ✓ Checkbox 'Show AI data' collegato direttamente")
                    else:
                        if retry_count < 5:
                            print(f"[RadargramController] Checkbox non trovato, retry {retry_count+1}/5")
                            QTimer.singleShot(200, lambda: try_connect(retry_count + 1))
                        else:
                            print("[RadargramController] ✗ checkbox_show_AI_data non trovato dopo 5 tentativi")
                except Exception as e:
                    print(f"[RadargramController] Errore collegamento checkbox: {e}")
            
            QTimer.singleShot(500, lambda: try_connect(0))
            
        except Exception as e:
            print(f"[RadargramController] Errore _connect_show_AI_data_checkbox: {e}")
    
    def _on_show_AI_data_toggled(self, checked: bool):
        """Toggle tra Data_current_filter e Data_AI quando cambia checkbox"""
        print(f"\n{'='*60}")
        print(f"[RadargramController] Show AI data toggled: checked={checked}")
        print(f"{'='*60}")
        
        # Invalida cache per forzare ricaricamento
        self._global_abs_max = None
        print("[RadargramController] Cache invalidata, refresh radargram...")
        
        # Refresh immediato
        self.refresh_radargram()
        print("[RadargramController] ✓ Radargram aggiornato")


