# controllers/SurveyController.py
# IMPORT PYSIDE6
from PySide6.QtCore import QObject, Qt
from PySide6.QtWidgets import QTreeWidgetItem
import os
import logging

# IMPORT CHANGED
from Controller.Main.Changed import ChangedController

# MODELLI
from Model.SurveyManager import SurveyManager

# THREAD
from Thread.ReadFile import ReadFile

class SurveyController(QObject):

    def __init__(self, view, parent=None):
        super().__init__(parent)

        self.view = view
        self.survey_manager = SurveyManager()
        self.survey_logger = logging.getLogger('survey')

        # NOTIFY CHANGED
        self.changed_controller = ChangedController()

        # ELENCO SURVEY CON RISPETTIVE SWATH
        self.view.tree_widget.itemChanged.connect(self.handle_item_changed)
        self.populate_tree_widget()
        self.first_populate = True

        self.th_readFile = ReadFile()
        self.th_readFile.finish_read.connect(self.populate_tree_widget)

    def populate_tree_widget(self):
        # DEVO COSTRUIRE UN TREE WIDGET FORMATO COSI:
        # Survey 1
        #   Swath 1
        #   Swath 2
        # Survey 2
        #   Swath 1
        self.view.tree_widget.clear()
        
        first_survey_item = None
        
        # Itera attraverso tutti i survey nel SurveyManager
        for survey_idx in range(len(self.survey_manager)):
            survey = self.survey_manager[survey_idx]
            
            # Crea il nome del survey (puoi personalizzare questo)
            #survey_name = os.path.basename(survey.folder_survey)
            survey_name = survey.survey_name
            self.survey_logger.info(f"Survey: {survey_name}")

            survey_item = QTreeWidgetItem([survey_name])
            survey_item.setFlags(survey_item.flags() | Qt.ItemIsUserCheckable)
            survey_item.setCheckState(0, Qt.CheckState.Unchecked)
            # Salva l'ID del survey nell'item per riferimento futuro
            survey_item.setData(0, Qt.UserRole, survey.ID_survey)
            
            # Salva il riferimento al primo survey
            if survey_idx == 0:
                first_survey_item = survey_item
            
            # Aggiungi le swath del survey corrente
            num_swath = survey.num_tot_swath
            for swath_idx in range(num_swath):
                swath = survey[swath_idx]
                swath_name = swath.swath_name
                #swath_name = os.path.basename(swath.link_swath_data)
                #swath_name = os.path.splitext(swath_name)[0]
                swath_item = QTreeWidgetItem([swath_name])
                swath_item.setFlags(swath_item.flags() | Qt.ItemIsUserCheckable)
                swath_item.setData(0, Qt.UserRole, {'survey_id': survey.ID_survey, 'swath_idx': swath_idx})
                
                survey_item.addChild(swath_item)

            self.view.tree_widget.addTopLevelItem(survey_item)
        
        # Seleziona il primo survey di default
        if first_survey_item:
            first_survey_item.setCheckState(0, Qt.CheckState.Checked)
            self.survey_manager.selected = first_survey_item.data(0, Qt.UserRole)
            self.survey_logger.info(f"Survey selezionato: {self.survey_manager.selected}")
            self.first_populate = False # non seleziono piu il primo survey di default
        # se non è il primo populate, seleziono il survey selezionato

    # NEL TREE_WIDGET LASCIO SELEZIONARE UNA SURVEY PER VOLTA E DEVE SEMPRE ESSERCENE UNO SELEZIONATO
    def handle_item_changed(self, item, column):
        self.survey_logger.info(f"Item cambiato: {item.text(0)}")
        if item.parent() is None:  # Se è un survey (item di primo livello)
            # Blocco temporaneamente i segnali per evitare la ricorsione
            self.view.tree_widget.blockSignals(True)
            
            if item.checkState(column) == Qt.CheckState.Checked:
                # Deseleziona gli altri survey
                for i in range(self.view.tree_widget.topLevelItemCount()):
                    other_item = self.view.tree_widget.topLevelItem(i)
                    if other_item != item:
                        other_item.setCheckState(column, Qt.CheckState.Unchecked)
                
                # Ottieni l'ID del survey selezionato
                survey_id = item.data(0, Qt.UserRole)
                if survey_id is not None:
                    self.changed_controller.emit_survey_changed(survey_id)
                    #self.survey_manager.selected = survey_id
            else:  # Se si sta tentando di deselezionare
                # Impedisci la deselezione rimettendo il check
                item.setCheckState(column, Qt.CheckState.Checked)
            # Riattivo i segnali
            self.view.tree_widget.blockSignals(False)
