# controllers/StartController.py
# IMPORT PYSIDE6
from PySide6.QtWidgets import QFileDialog, QApplication, QMessageBox
from PySide6.QtCore import QObject, Qt, Signal

# MODEL
from Model.SurveyManager import SurveyManager

# THREAD
from Thread.ReadFile import ReadFile
from Thread.ApplyFilter import ApplyFilter
from Thread.RunAI import RunAI

# VIEW
from View.ProgressBar import ProgressBar

# LOG
import logging
import os

from Model.ProjectManifest import open_from_tempdata_manifest, open_external_project_json
from GLOBAL import tempData, open_project_mode
class StartController(QObject):

    # Signal per notificare che deve iniziare la generazione dei tiles
    start_tile_generation = Signal()

    def __init__(self, view, parent=None):
        super().__init__(parent)

        self.view = view
        self.survey_manager = SurveyManager()   # recupero il gestore dei survey
        self.start_logger = logging.getLogger('start')  # recupero il logger corretto

        # Inizializza la progress bar riutilizzabile
        self.progress_bar = ProgressBar(self.view)

        # import survey
        self.view.import_survey_button.clicked.connect(self.select_folder_import)

        # open project
        self.view.open_project_button.clicked.connect(self.select_project)

        # thread read file and load model
        self.th_readFile = ReadFile()
        self.th_readFile.finish_read.connect(self.read_file_finished)
        self.th_readFile.base_tomography_completed.connect(self.auto_trigger_ai)

        # thread per l'applicazione dei filtri standard
        self.th_apply_filter = ApplyFilter()
        self.th_apply_filter.filter_completed.connect(self._on_filter_completed)
        self.th_apply_filter.filter_error.connect(self._on_filter_error)
        self.th_apply_filter.filter_progress.connect(self._on_filter_progress)

        # thread AI: esegue l'inferenza tra base e creazione tomografie di inferenza
        self.th_runAI = RunAI()
        self.th_runAI.ai_started.connect(self._on_ai_started)
        self.th_runAI.ai_progress.connect(self._on_ai_progress)
        self.th_runAI.ai_completed.connect(self._on_ai_completed)
        self.th_runAI.ai_error.connect(self._on_ai_error)



    def select_folder_import(self):
        # CONTROLLO LA TIPOLOGIA SELEZIONATA
        # ...
        # ...

        survey_folder = QFileDialog.getExistingDirectory(self.view, "Select Directory")
        if survey_folder:
            self.th_readFile.original_survey_folder = survey_folder
            self.th_readFile.start()
            self.progress_bar.show_progress_dialog()

    def select_project(self):
        # L'utente seleziona un project.json. Il codice cercherà la cartella TempData a fianco e la userà.
        from PySide6.QtWidgets import QFileDialog as _FD
        project_json = _FD.getOpenFileName(self.view, "Select project.json", os.getcwd(), "JSON Files (*.json)")[0]
        if not project_json:
            return
        try:
            self.progress_bar.show_progress_dialog("Apertura progetto...")
            ok = open_external_project_json(project_json, self.survey_manager)
            if ok:
                self.start_logger.info("Project opened from external project.json")
                # Set runtime flag to start tile thread only for Open Project
                import GLOBAL as G
                G.open_project_mode = True
                self.progress_bar.close()
                self.view.close_by_code()
            else:
                self.start_logger.error("Invalid project.json or TempData not found")
                self.progress_bar.close()
        except Exception as e:
            self.start_logger.error(f"Error opening project: {e}")
            self.progress_bar.close()

    def read_file_finished(self, check_read):
        if check_read:
            print("StartController: File caricato con successo")
            # Assicurati che il survey appena caricato sia selezionato
            self.survey_manager.selected = 0  # o altro ID se necessario
            # FILTRI AUTOMATICI DISATTIVATI
            # self.progress_bar.show_progress_dialog("Applicazione filtri in corso...")
            # self.th_apply_filter.default_sequence = True
            # self.th_apply_filter.start()
            self.progress_bar.close()
            print("StartController: Survey caricato senza applicazione automatica filtri")
            self.view.close_by_code()
        else:
            print("StartController: Errore nel caricamento del file")
            self.progress_bar.close()

    def _on_filter_completed(self):
        """Gestisce il completamento dell'applicazione dei filtri"""
        try:
            self.start_logger.info("Applicazione filtri standard completata")
            self.progress_bar.close()
            self.view.close_by_code()
            
        except Exception as e:
            error_msg = f"Errore durante la gestione del completamento filtri: {str(e)}"
            self.start_logger.error(error_msg)
            QMessageBox.critical(self.view, "Errore", error_msg)

    def _on_filter_error(self, error_msg):
        """Gestisce gli errori durante l'applicazione dei filtri"""
        self.start_logger.error(f"Errore durante l'applicazione dei filtri: {error_msg}")
        self.progress_bar.close()
        QMessageBox.critical(self.view, "Errore", f"Errore durante l'applicazione dei filtri: {error_msg}")

    def _on_filter_progress(self, message):
        """Gestisce i messaggi di progresso durante l'applicazione dei filtri"""
        self.start_logger.info(message)

    def show_error_import(self):
        msg_box = QMessageBox()
        msg_box.setIcon(QMessageBox.Critical)
        msg_box.setWindowTitle("Error")
        msg_box.setText("Import survey failed")
        msg_box.setStandardButtons(QMessageBox.Ok)
        msg_box.exec()

    def auto_trigger_ai(self):
        """Attiva automaticamente l'AI dopo il completamento della tomografia base"""
        print("[StartController] Tomografia base completata, avvio automatico AI")
        print("[DEBUG] auto_trigger_ai called")
        # Avvio il thread di inferenza AI. La creazione della tomografia di
        # inferenza e dei tile partirà nel callback _on_ai_completed
        try:
            self.th_runAI.start()
            print("[DEBUG] AI thread started")
        except Exception as e:
            print(f"[ERROR] Errore avvio AI: {e}")
            self.start_logger.error(f"Errore avvio AI: {e}")

    def _on_ai_started(self, message: str):
        print(f"[StartController] AI started: {message}")

    def _on_ai_progress(self, message: str):
        print(f"[StartController] AI progress: {message}")

    def _on_ai_completed(self, message: str):
        print(f"[StartController] AI completata: {message}")
        # Ora creo la tomografia di inferenza dai dati filtrati aggiornati
        try:
            print("[DEBUG] Creazione tomografia di inferenza...")
            self.survey_manager.create_tomography_inferenza()
            print("[DEBUG] Tomografia di inferenza creata con successo")

            # Avvio la generazione dei tiles
            print("[DEBUG] Avvio generazione tiles...")
            from Thread.Raster_tile import Raster_tile
            th_raster_tile = Raster_tile()
            th_raster_tile.start()
            print("[DEBUG] Generazione tiles avviata")
        except Exception as e:
            print(f"[ERROR] Errore creazione tomografia inferenza: {e}")
            self.start_logger.error(f"Errore creazione tomografia inferenza: {e}")

    def _on_ai_error(self, message: str):
        print(f"[StartController] AI error: {message}")
        self.start_logger.error(f"AI error: {message}")
