import os
import json
import time
import shutil
from typing import Dict, Any

# MODELS
from Model.SurveyManager import SurveyManager


def _relpath_safe(path: str, start: str) -> str:
    try:
        return os.path.relpath(path, start)
    except Exception:
        return path


def write_tempdata_manifest(tempdata_root: str, survey_manager: SurveyManager) -> str:
    """Write a lightweight manifest in the TempData folder to allow quick reopen.
    Returns the manifest absolute path.
    """
    manifest: Dict[str, Any] = {
        "schema_version": "0.1.0",
        "app": {"name": "XMAP"},
        "project": {
            "created_at": time.strftime('%Y-%m-%dT%H:%M:%SZ', time.gmtime()),
        },
        "tempdata_root": tempdata_root,
        "surveys": [],
    }

    for survey in survey_manager.survey_list:
        survey_entry: Dict[str, Any] = {
            "name": survey.survey_name,
            "path": _relpath_safe(survey.local_survey_folder, tempdata_root),
            "swaths": [],
        }

        def _collect_swaths(swath_list, pol: str):
            for swath in swath_list:
                swath_path = swath.polarization_folder_local
                entry = {
                    "name": swath.swath_name,
                    "pol": pol,
                    "path": _relpath_safe(swath_path, tempdata_root),
                    "header": _relpath_safe(swath.header_json_file, tempdata_root),
                    "lat": _relpath_safe(swath.path_georef_matrix_lat, tempdata_root),
                    "lon": _relpath_safe(swath.path_georef_matrix_lon, tempdata_root),
                    "subswath_index": _relpath_safe(swath.subswath_json_file, tempdata_root),
                    "polygon": _relpath_safe(swath.path_polygon_total_area, tempdata_root),
                    "subswaths": []
                }
                for subswath in swath.subswath_list:
                    ss_entry = {
                        "name": subswath.subswath_name,
                        "path": _relpath_safe(subswath.subSwath_folder, tempdata_root),
                        "georef": {
                            "lat": _relpath_safe(subswath.path_georef_matrix_lat, tempdata_root),
                            "lon": _relpath_safe(subswath.path_georef_matrix_lon, tempdata_root),
                            "polygon": _relpath_safe(subswath.path_polygon_subswath_area, tempdata_root),
                        },
                        "data_default": _relpath_safe(subswath.data_default_folder, tempdata_root),
                        "data_current_filter": _relpath_safe(subswath.data_current_filter_folder, tempdata_root),
                        "tomography": _relpath_safe(subswath.tomography_folder, tempdata_root),
                        "tomography_inferenza": _relpath_safe(os.path.join(subswath.subSwath_folder, 'tomography_inferenza'), tempdata_root),
                    }
                    entry["subswaths"].append(ss_entry)
                survey_entry["swaths"].append(entry)

        _collect_swaths(survey.swath_list, "VV")
        _collect_swaths(survey.swath_list_HH, "HH")

        manifest["surveys"].append(survey_entry)

    manifest_path = os.path.join(tempdata_root, "project_tempdata.json")
    try:
        with open(manifest_path, "w", encoding="utf-8") as f:
            json.dump(manifest, f, indent=2)
    except Exception:
        # As fallback, attempt writing inside the first survey folder
        if manifest["surveys"]:
            alt_dir = os.path.join(tempdata_root, manifest["surveys"][0]["path"])
            os.makedirs(alt_dir, exist_ok=True)
            manifest_path = os.path.join(alt_dir, "project_tempdata.json")
            with open(manifest_path, "w", encoding="utf-8") as f:
                json.dump(manifest, f, indent=2)
    return manifest_path


def open_from_tempdata_manifest(manifest_path: str, survey_manager: SurveyManager) -> bool:
    """Rebuild the in-memory structures from an existing TempData manifest.
    This avoids raw import by indexing existing files.
    """
    if not os.path.isfile(manifest_path):
        return False
    try:
        with open(manifest_path, "r", encoding="utf-8") as f:
            manifest = json.load(f)
    except Exception:
        return False

    tempdata_root = os.path.abspath(os.path.dirname(manifest_path))
    if os.path.isdir(manifest.get("tempdata_root", "")):
        # If manifest stored absolute root, prefer that
        tempdata_root = manifest["tempdata_root"]

    # Build structures
    import numpy as np
    import json as _json
    import geopandas as gpd
    from shapely.geometry import shape

    # Track overall bounds (EPSG:32632 meters)
    overall_max_x = None
    overall_min_x = None
    overall_max_y = None
    overall_min_y = None

    for s in manifest.get("surveys", []):
        # Use the survey path to preserve the original survey_name and folder layout
        survey_root = s.get("path")
        if survey_root:
            survey_abs = os.path.join(tempdata_root, survey_root)
        else:
            survey_abs = tempdata_root
        survey_manager.ADD_SURVEY(survey_abs, "OGPR")

        # Add swaths VV then HH
        for swath_entry in s.get("swaths", []):
            pol = swath_entry.get("pol", "VV")
            polarization = 0 if pol == "VV" else 1
            survey_manager.set_current_polarization(polarization)

            # Load header and indices
            header_path = os.path.join(tempdata_root, swath_entry["header"]) if swath_entry.get("header") else None
            subswath_index_path = os.path.join(tempdata_root, swath_entry["subswath_index"]) if swath_entry.get("subswath_index") else None
            lat_path = os.path.join(tempdata_root, swath_entry["lat"]) if swath_entry.get("lat") else None
            lon_path = os.path.join(tempdata_root, swath_entry["lon"]) if swath_entry.get("lon") else None
            polygon_path = os.path.join(tempdata_root, swath_entry["polygon"]) if swath_entry.get("polygon") else None

            header_json = {}
            if header_path and os.path.isfile(header_path):
                with open(header_path, "r", encoding="utf-8") as hf:
                    header_json = _json.load(hf)

            subswath_index = {"subswath_details": []}
            if subswath_index_path and os.path.isfile(subswath_index_path):
                with open(subswath_index_path, "r", encoding="utf-8") as sf:
                    subswath_index = _json.load(sf)

            georef_lat = np.load(lat_path) if lat_path and os.path.isfile(lat_path) else None
            georef_lon = np.load(lon_path) if lon_path and os.path.isfile(lon_path) else None

            # Update overall bounds
            try:
                if georef_lon is not None and georef_lat is not None:
                    max_x = float(np.nanmax(georef_lon))
                    min_x = float(np.nanmin(georef_lon))
                    max_y = float(np.nanmax(georef_lat))
                    min_y = float(np.nanmin(georef_lat))
                    if overall_max_x is None or max_x > overall_max_x:
                        overall_max_x = max_x
                    if overall_min_x is None or min_x < overall_min_x:
                        overall_min_x = min_x
                    if overall_max_y is None or max_y > overall_max_y:
                        overall_max_y = max_y
                    if overall_min_y is None or min_y < overall_min_y:
                        overall_min_y = min_y
            except Exception:
                pass

            # coords_perimeter from polygon geojson
            coords_perimeter = []
            try:
                if polygon_path and os.path.isfile(polygon_path):
                    gdf = gpd.read_file(polygon_path)
                    if len(gdf) > 0:
                        geom = gdf.iloc[0].geometry
                        if geom and hasattr(geom, "exterior"):
                            coords_perimeter = list(geom.exterior.coords)
            except Exception:
                pass

            name_swath = swath_entry.get("name", os.path.basename(os.path.dirname(swath_entry.get("path", ""))))
            original_swath_path = os.path.join(tempdata_root, swath_entry.get("path", ""))
            survey_manager.ADD_SWATH(original_swath_path, name_swath, header_json, subswath_index, coords_perimeter, georef_lat, georef_lon)

            # Add subswaths and index files
            for info_subswath in subswath_index.get("subswath_details", []):
                survey_manager.ADD_SUBSWATH(info_subswath)
                # georef per subswath
                try:
                    ss_name = f"SubSwath{info_subswath['id_subswath']:02d}"
                    ss_folder = os.path.join(original_swath_path, ss_name)
                    ss_lat = os.path.join(ss_folder, "georef", "georef_matrix_lat.npy")
                    ss_lon = os.path.join(ss_folder, "georef", "georef_matrix_lon.npy")
                    ss_poly = os.path.join(ss_folder, "georef", "Polygon_SubswathArea.json")
                    ss_lat_arr = np.load(ss_lat) if os.path.isfile(ss_lat) else None
                    ss_lon_arr = np.load(ss_lon) if os.path.isfile(ss_lon) else None
                    ss_coords = []
                    try:
                        gdf_ss = gpd.read_file(ss_poly)
                        if len(gdf_ss) > 0:
                            geom_ss = gdf_ss.iloc[0].geometry
                            if geom_ss and hasattr(geom_ss, "exterior"):
                                ss_coords = list(geom_ss.exterior.coords)
                    except Exception:
                        pass
                    survey_manager.ADD_SUBSWATH_GEOREF(info_subswath['id_subswath'], ss_lat_arr, ss_lon_arr, ss_coords)
                    # Index existing files to avoid re-writing
                    try:
                        current_survey_idx = survey_manager.current_survey if survey_manager.current_survey is not None else 0
                        survey_obj = survey_manager[current_survey_idx]
                        swath_obj = survey_obj[survey_manager.current_swath]
                        swath_obj[info_subswath['id_subswath']].index_existing_files()
                    except Exception:
                        pass
                except Exception:
                    # continue even if a subswath fails
                    pass

            survey_manager.confirm_add_subswath()
            survey_manager.confirm_add_swath()

        survey_manager.confirm_add_survey()

    # Initialize tiles if bounds available
    try:
        if None not in (overall_max_x, overall_min_x, overall_max_y, overall_min_y):
            from Model.TileManager import TileManager
            tile_manager = TileManager()
            tile_manager.set_bounds_zone(overall_max_x, overall_min_x, overall_max_y, overall_min_y)
    except Exception:
        pass

    return True


def write_external_project(dest_dir: str, tempdata_src: str, survey_manager: SurveyManager) -> str:
    """Create a portable project folder:
    - Copies TempData into dest_dir/TempData
    - Writes dest_dir/project.json with minimal metadata and reference to sibling TempData
    Returns the path to project.json
    """
    os.makedirs(dest_dir, exist_ok=True)
    dest_tempdata = os.path.join(dest_dir, 'TempData')
    # Fresh copy of TempData
    if os.path.isdir(dest_tempdata):
        shutil.rmtree(dest_tempdata, ignore_errors=True)
    shutil.copytree(tempdata_src, dest_tempdata)

    # Ensure inner manifest exists (write if missing)
    inner_manifest = os.path.join(dest_tempdata, 'project_tempdata.json')
    if not os.path.isfile(inner_manifest):
        write_tempdata_manifest(dest_tempdata, survey_manager)

    project_name = None
    if survey_manager.survey_list:
        project_name = survey_manager.survey_list[0].survey_name
    manifest = {
        "schema_version": "0.1.0",
        "app": {"name": "XMAP"},
        "project": {
            "name": project_name or "XMAP_Project",
            "created_at": time.strftime('%Y-%m-%dT%H:%M:%SZ', time.gmtime()),
        },
        "tempdata_rel": "TempData",
        "notes": "This is the entry manifest. The detailed index is TempData/project_tempdata.json",
    }
    project_json_path = os.path.join(dest_dir, 'project.json')
    with open(project_json_path, 'w', encoding='utf-8') as f:
        json.dump(manifest, f, indent=2)
    return project_json_path


def open_external_project_json(project_json_path: str, survey_manager: SurveyManager) -> bool:
    """Open a project by selecting an external project.json.
    Assumes a sibling TempData folder exists beside the JSON.
    Copies that TempData into the application's local TempData, then loads models from it.
    """
    if not os.path.isfile(project_json_path):
        return False
    try:
        with open(project_json_path, 'r', encoding='utf-8') as f:
            manifest = json.load(f)
    except Exception:
        manifest = {}

    base_dir = os.path.dirname(project_json_path)
    tempdata_name = manifest.get('tempdata_rel', 'TempData')
    external_tempdata = os.path.join(base_dir, tempdata_name)
    if not os.path.isdir(external_tempdata):
        return False

    # Copy external TempData into local TempData
    from GLOBAL import tempData as local_tempdata
    if os.path.isdir(local_tempdata):
        shutil.rmtree(local_tempdata, ignore_errors=True)
    shutil.copytree(external_tempdata, local_tempdata)

    local_manifest = os.path.join(local_tempdata, 'project_tempdata.json')
    return open_from_tempdata_manifest(local_manifest, survey_manager)


