import os
import json
import numpy as np
import geopandas as gpd
from shapely.geometry import Polygon

# LOGGING
import logging

class SubSwath:
    def __init__(self, subswath_info, swath_folder):
        # LOGGING
        self.logger = logging.getLogger('survey_manager')

        # INFORMAZIONI SUBSWATH
        self.id_subswath = subswath_info["id_subswath"]
        self.subswath_name = f"SubSwath{self.id_subswath:02d}"
        self.start_index = subswath_info["start_index"]
        self.end_index = subswath_info["end_index"]
        self.size = subswath_info["size"]

        # FOLDER
        self.subSwath_folder = os.path.join(swath_folder, self.subswath_name) 
        os.makedirs(self.subSwath_folder, exist_ok=True) # creo la cartella locale(TempData) della subSwath

        # CREAZIONE DELLE CARTELLE CONTENENTI I DATI DELLA SUBSWATH
        self.georef_folder = os.path.join(self.subSwath_folder, "georef") # cartella locale(TempData) della subSwath Georef
        os.makedirs(self.georef_folder, exist_ok=True) # creo la cartella locale della subSwath Georef

        self.tomography_folder = os.path.join(self.subSwath_folder, "tomography") # cartella locale(TempData) della subSwath Tomography
        os.makedirs(self.tomography_folder, exist_ok=True) # creo la cartella locale della subSwath Tomography

        self.data_default_folder = os.path.join(self.subSwath_folder, "Data_default") # cartella locale(TempData) della subSwath Data_default
        os.makedirs(self.data_default_folder, exist_ok=True) # creo la cartella locale della subSwath Data_default

        self.data_current_filter_folder = os.path.join(self.subSwath_folder, "Data_current_filter") # cartella locale(TempData) della subSwath Data_current_filter
        os.makedirs(self.data_current_filter_folder, exist_ok=True) # creo la cartella locale della subSwath Data_current_filter

        self.data_AI_folder = os.path.join(self.subSwath_folder, "Data_AI") # cartella locale(TempData) della subSwath Data_AI (inferenza)
        os.makedirs(self.data_AI_folder, exist_ok=True) # creo la cartella locale della subSwath Data_AI

        # PATH FILE GEOREF
        self.path_polygon_subswath_area = os.path.join(self.georef_folder, "Polygon_SubswathArea.json")
        self.path_georef_matrix_lat = os.path.join(self.georef_folder, "georef_matrix_lat.npy")
        self.path_georef_matrix_lon = os.path.join(self.georef_folder, "georef_matrix_lon.npy")

        # TRACCIA DEI RADARGRAMMI E DELLE TOMOGRAFIE SALVATE IN LOCALE
        self.file_subswath_channel_list = []  # lista che conterrà il path al file .npy dei canali [.NPY]
        self.file_subswath_channel_default_list = []  # lista che conterrà il path al file .npy dei canali senza elaborazioni [.NPY]
        self.selected_subswath_channel = 0 # 0 di default

        # DATI RELATIVI ALLA TOMOGRFIA
        self.file_tomography_list = []  # lista che conterrà il path al file .npy delle tomografie [.NPY]
        self.file_tomography_inferenza_list = []  # lista che conterrà il path al file .npy delle tomografie inferenza [.NPY]
        self.selected_tomography = 0 # 0 di default
        self.tot_depth = 0

        # LISTA DI TUTTI I TILE CHE ATTRAVERSA LA SUBSWATH
        self.path_list_tile_where_subswath_is = os.path.join(self.subSwath_folder, "list_index_tiles.json")
        self.list_tile_where_subswath_is = []

        # Creazione maschera raster
        self.mask_raster = False

        self.logger.info(f"|| SubSwath initialized successfully ||")
        self.logger.info(f"[SUBSWATH] -> ID: {self.id_subswath}")
        self.logger.info(f"[SUBSWATH] -> Name: {self.subswath_name}")
    
    def index_existing_files(self):
        """Populate internal file lists by indexing existing files on disk.
        Avoids re-writing arrays when opening an existing project.
        """
        try:
            # Index default channel data
            if os.path.isdir(self.data_default_folder):
                npy_files = [f for f in os.listdir(self.data_default_folder) if f.endswith('.npy')]
                npy_files.sort()
                self.file_subswath_channel_list = [os.path.join(self.data_default_folder, f) for f in npy_files]
                self.num_tot_ch = len(self.file_subswath_channel_list)
                # Determine original sample count from first channel if available
                if self.file_subswath_channel_list:
                    try:
                        first = np.load(self.file_subswath_channel_list[0], mmap_mode='r')
                        self.original_sample_count = first.shape[0]
                    except Exception:
                        pass

            # Index tomography files
            if os.path.isdir(self.tomography_folder):
                tomo_files = [f for f in os.listdir(self.tomography_folder) if f.endswith('.npy')]
                # Prefer ordering by depth_NNNcm
                def _depth_key(name: str):
                    try:
                        base = os.path.splitext(name)[0]
                        n = base.split('_')[-1]
                        return int(n.replace('cm', ''))
                    except Exception:
                        return 0
                tomo_files.sort(key=_depth_key)
                self.file_tomography_list = [os.path.join(self.tomography_folder, f) for f in tomo_files]
                self.tot_depth = len(self.file_tomography_list)

            # Index tomography_inferenza if present
            tomography_inferenza_folder = os.path.join(self.subSwath_folder, 'tomography_inferenza')
            if os.path.isdir(tomography_inferenza_folder):
                def _depth_key(name: str):
                    try:
                        base = os.path.splitext(name)[0]
                        n = base.split('_')[-1]
                        return int(n.replace('cm', ''))
                    except Exception:
                        return 0
                tomo_inf_files = [f for f in os.listdir(tomography_inferenza_folder) if f.endswith('.npy')]
                tomo_inf_files.sort(key=lambda n: _depth_key(n))
                self.file_tomography_inferenza_list = [os.path.join(tomography_inferenza_folder, f) for f in tomo_inf_files]
        except Exception as e:
            self.logger.error(f"[SUBSWATH] index_existing_files error: {e}")

    def ADD_SUBSWATH_GEOREF(self, georef_matrix_lat, georef_matrix_lon, coords_perimeter):
        self.logger.info(f"[SUBSWATH] ----------------- ADD SUBSWATH GEOREF -----------------")
        # salvo la georef nella cartella locale Georef
        np.save(self.path_georef_matrix_lat, georef_matrix_lat)
        np.save(self.path_georef_matrix_lon, georef_matrix_lon)
        # Polygon subswath area
        polygon = Polygon(coords_perimeter)
        gdf = gpd.GeoDataFrame({
            'geometry': [polygon]
        })
        gdf.set_crs(epsg=32632, inplace=True)
        gdf.to_file(self.path_polygon_subswath_area, driver='GeoJSON')

    def ADD_SUBSWATH_CHANNEL(self, channel_data, name_channel):
        self.logger.info(f"[SUBSWATH] ----------------- ADD SUBSWATH CHANNEL -----------------")
        # salvo il radargramma nella cartella locale Data_default
        file_path = os.path.join(self.data_default_folder, f"{name_channel}.npy")
        self.logger.info(f"[SUBSWATH] -> File channel: {file_path}")
        np.save(file_path, channel_data)
        self.file_subswath_channel_list.append(file_path)
        self.num_tot_ch = len(self.file_subswath_channel_list)
        self.original_sample_count = channel_data.shape[0]
        self.logger.info(f"[SUBSWATH] -> Original sample count: {self.original_sample_count}")
    def ADD_SUBSWATH_TOMOGRAPHY(self, tomography_data, name_tomography):
        self.logger.info(f"[SUBSWATH] ----------------- ADD SUBSWATH TOMOGRAPHY -----------------")
        # salvo la tomografia nella cartella locale Tomography
        file_path = os.path.join(self.tomography_folder, f"{name_tomography}.npy")
        self.logger.info(f"[SUBSWATH] -> File tomography: {file_path}")
        np.save(file_path, tomography_data)
        self.file_tomography_list.append(file_path)
        self.tot_depth = len(self.file_tomography_list)

    # Mi prendo il canale della subswath selezionata
    def channel(self, channel_idx):
        return np.load(self.file_subswath_channel_list[channel_idx])

    def tomography(self, tomography_idx):
        return np.load(self.file_tomography_list[tomography_idx])
    
    # Mi prendo la georef della subswath selezionata
    def get_subswath_area(self):
        return self.path_polygon_subswath_area

    # get/set della lista dei tile che attraversa la subswath
    def get_list_tile_where_subswath_is(self):
        return self.list_tile_where_subswath_is
    def set_list_tile_where_subswath_is(self, list_tile_where_subswath_is):
        self.list_tile_where_subswath_is = list_tile_where_subswath_is
        with open(self.path_list_tile_where_subswath_is, "w") as f:
            json.dump(self.list_tile_where_subswath_is, f)

    # funzione temporanea per salvare il mat3d_sub in locale
    def funct_temp_blocchettino(self, mat3d_sub):
        self.path_mat3d_sub = os.path.join(self.subSwath_folder, "mat3d_sub.npy")
        np.save(self.path_mat3d_sub, mat3d_sub)
        self.mat3d_sub = mat3d_sub

    # CREAZIONE TOMOGRAFIA INFERENZA
    def create_tomography_inferenza(self):
        self.logger.info(f"[SUBSWATH] ----------------- CREATE TOMOGRAPHY INFERENZA -----------------")
        # creo la cartella tomography_inferenza
        self.tomography_inferenza_folder = os.path.join(self.subSwath_folder, "tomography_inferenza")
        os.makedirs(self.tomography_inferenza_folder, exist_ok=True)

        # Prendo i canali AI e li carico in una lista (da Data_AI, non Data_current_filter)
        channel_list = []
        # Mi prendo la lista dei file .npy della cartella Data_AI (SOLO .npy, non JSON!)
        data_AI_folder_list = [f for f in os.listdir(self.data_AI_folder) if f.endswith('.npy')]
        # Ordino per garantire ordine corretto dei canali
        data_AI_folder_list.sort()
        self.logger.info(f"[SUBSWATH] -> Carico {len(data_AI_folder_list)} canali AI da {self.data_AI_folder}")
        for ch in data_AI_folder_list:
            chnnellino_aperto = np.load(os.path.join(self.data_AI_folder, ch), allow_pickle=True)
            self.logger.info(f"[SUBSWATH] -> Canale AI aperto: {chnnellino_aperto.shape}")
            channel_list.append(chnnellino_aperto)
        # prendo le dimensioni massime delle matrici
        max_rows = max(channel_list, key=lambda x: x.shape[0]).shape[0]
        max_cols = max(channel_list, key=lambda x: x.shape[1]).shape[1]

        # creo la matrice 3D con le dimensioni massime trovate
        num_channels = len(channel_list)
        mat3d_sub = np.zeros((max_cols, num_channels, max_rows))

        self.logger.info(f"[SUBSWATH] -> Max rows: {max_rows}")
        self.logger.info(f"[SUBSWATH] -> Max cols: {max_cols}")
        self.logger.info(f"[SUBSWATH] -> Num channels: {num_channels}")
        

        self.logger.info(f"[SUBSWATH] -> Mat3d sub shape: {mat3d_sub.shape}")
        # per ogni canale, posiziono la matrice nell'angolo in alto a sinistra e riempio il resto con zeri
        for ch in range(num_channels):
            current_channel = channel_list[ch]
            current_channel = current_channel.T
            rows, cols = current_channel.shape
            # posiziono la matrice nell'angolo in alto a sinistra
            mat3d_sub[:rows, ch, :cols] = current_channel
        self.logger.info(f"[SUBSWATH] -> Mat3d sub shape: {mat3d_sub.shape}")
        # aggiorno sample_count con le nuove dimensioni
        
        sample_count = mat3d_sub.shape[2]
        self.logger.info(f"[SUBSWATH] -> Sample count: {sample_count}")

        # depth
        # la depth la ricavo andando a leggere il nome dell'ultimo file della tomografia
        # so che il nome è così "depth_NNNcm.npy"
        last_file = self.file_tomography_list[-1]
        max_depth_str = last_file.split("_")[-1].split(".")[0]
        max_depth_cm = int(max_depth_str.replace("cm", ""))
        self.logger.info(f"[SUBSWATH] -> Max depth cm: {max_depth_cm}")

        # devo trovare la proporzione
        layer_cm = max_depth_cm / self.original_sample_count # mi calcolo quante vale un riga della matrice in cm
        self.logger.info(f"[SUBSWATH] -> Layer cm: {layer_cm}  --> {self.original_sample_count} / {max_depth_cm}")
        new_depth_cm = int(layer_cm * sample_count) # trovo la nuova profodnita in cm
        new_depth_m = new_depth_cm / 100.0
        self.logger.info(f"[SUBSWATH] -> New depth cm: {new_depth_cm}  --> {layer_cm} * {sample_count}")
        self.logger.info(f"[SUBSWATH] -> New depth m: {new_depth_m}  --> {new_depth_cm} / 100.0")

        target_depths_cm = np.arange(0, new_depth_cm + 1)  # da 0 a N cm
        #self.logger.info(f"[SUBSWATH] -> Target depths cm: {target_depths_cm}")
        for z in range(len(target_depths_cm)):
            depth_m = target_depths_cm[z] / 100.0  # converti cm in metri
            z_original = int(depth_m * (sample_count / new_depth_m))
            if z_original >= sample_count:
                z_original = sample_count - 1
            tomography = mat3d_sub[:, :, z_original].T # Carico la tomografia all'indice relativo alla profondità
            name_tomography = f"depth_{z:03d}cm"
            file_path = os.path.join(self.tomography_inferenza_folder, f"{name_tomography}.npy")
            np.save(file_path, tomography)
            self.file_tomography_inferenza_list.append(file_path)

             # SALVO IL RADARGRAMMA COME IMMAGINE NELLA CARTELLA TRASH
            if False:
                try:
                    # Normalizza l'array a 8-bit unsigned integer (0-255)
                    normalized = np.interp(tomography, (tomography.min(), tomography.max()), (0, 255))
                    img_array = normalized.astype(np.uint8)
                    # Crea l'immagine
                    from PIL import Image
                    from GLOBAL import moment_dir
                    img = Image.fromarray(img_array, mode='L')  # 'L' mode for grayscale
                    # assicuro che la cartella di destinazione esista e imposto un'estensione corretta
                    os.makedirs(moment_dir, exist_ok=True)
                    img_path = os.path.join(moment_dir, f"{name_tomography}.png")
                    img.save(img_path, format="PNG")
                    #self.start_logger.info(f"[SAVE] ---> Saved tomography image: {img_path}")
                except Exception as e:
                    print(f"Error saving tomography image: {str(e)}")
                    continue

    def tomography_inferenza(self, tomography_idx):
        #print(f"Tomography inferenza: {self.file_tomography_inferenza_list[tomography_idx]}")
        return np.load(self.file_tomography_inferenza_list[tomography_idx])

    def save_manholes(self, manholes):
        # salvo i manholes nella cartella della subswath
        file_path = os.path.join(self.subSwath_folder, "manholes.json")
        self.logger.info(f"[SUBSWATH] -> Saving manholes to: {file_path}")
        self.manholes_path = file_path
        with open(self.manholes_path, "w") as f:
            json.dump(manholes, f)
