# CLASSE PER LA GESTIONE DEI TILE
# - Classe per la gestione dei tile
# - Logger: tile.log

# LOGGING
import logging
import os
import json

# MODEL
from Model.SurveyManager import SurveyManager
from Model.Tile import Tile

# GLOBAL
from GLOBAL import tiles_dir

class TileManager:
    _instance = None

    def __new__(cls):
        if cls._instance is None:
            cls._instance = super(TileManager, cls).__new__(cls)
            cls._instance._initialized = False
        return cls._instance

    def __init__(self):
        if self._initialized:
            return
        
        self.survey_manager = SurveyManager()  # recupero il gestore dei survey

        # BOUNDS ZONE
        self.max_x = None # [LONGITUDE]
        self.min_x = None # [LONGITUDE]
        self.max_y = None # [LATITUDE]
        self.min_y = None # [LATITUDE]
                                                                                                                                    # 1 -- 2
        self.bounds_zone = [(self.max_y, self.min_x), (self.max_y, self.max_x), (self.min_y, self.max_x), (self.min_y, self.min_x)] # |    |
                                                                                                                                    # 4 -- 3
        # TILE SIZE
        self.tile_size_m = 30.0 # [METERS]

        # Totale tile x,y
        self.total_tiles_x = None
        self.total_tiles_y = None

        # TILES
        self.dic_tile = {}

        self.total_tiles = None
        
        # CURRENT BOUNDS MAP VISUALIZATION  --> EPSG:32632
        self.lat_max_map = None
        self.lat_min_map = None
        self.lon_min_map = None
        self.lon_max_map = None  
        
        self._initialized = True

    def set_bounds_zone(self, max_x, min_x, max_y, min_y):
        self.max_x = max_x
        self.min_x = min_x
        self.max_y = max_y
        self.min_y = min_y

        # Calcola il numero totale di tile
        self.total_tiles = self._calculate_total_tiles()

        self.create_tiles()
        print("Popolo tile")
        self.popolo_tile()

         # Esporta automaticamente i tile come shapefile
        if True:
            self.export_tiles_to_shapefile("tiles_bounds")

    # DATO UN PUNTO GEOREFERENZIATO RICAVO L'INDICE DEL TILE ALL'INTERNO DEL DIZIONARIO 'dic_tile'
    def from_point_to_index_tile(self, point):
        # INPUT: punto in metri
        # OUTPUT: indice del tile
        x_tile_index = int((point[0] - self.min_x) / self.tile_size_m) 
        y_tile_index = int((point[1] - self.min_y) / self.tile_size_m)
        #print(f"x_tile_index: {x_tile_index}, y_tile_index: {y_tile_index}")
        #print(f"self.total_tiles_x: {self.total_tiles_x}, self.total_tiles_y: {self.total_tiles_y}")

        # ora ricavo l'indice nel dizionario        
        index_tile = x_tile_index * self.total_tiles_y + y_tile_index
        
        #print(f"index_tile: {index_tile}")

        return index_tile

    # DATO L'INDICE DEL TILE NEL DIZIONARIO RICAVO UN PUNTO DI RIFERIMENTO (x, y)
    # Restituisco il corner basso-sinistra del tile in metri (EPSG:32632)
    def from_index_to_point(self, index_tile):
        # INPUT: indice del tile
        # OUTPUT: punto (x, y) in metri corrispondente al corner basso-sinistra del tile
        if self.total_tiles_y is None or self.min_x is None or self.min_y is None:
            raise ValueError("Parametri non inizializzati: assicurarsi di aver chiamato set_bounds_zone/create_tiles")

        if index_tile < 0 or (self.total_tiles is not None and index_tile >= self.total_tiles):
            raise ValueError(f"Indice tile fuori range: {index_tile}")

        # Ricavo gli indici (i, j) nella griglia a partire dall'indice lineare
        x_tile_index = index_tile // self.total_tiles_y
        y_tile_index = index_tile % self.total_tiles_y

        # Calcolo il punto (corner basso-sinistra) del tile
        x = self.min_x + (x_tile_index * self.tile_size_m)
        y = self.min_y + (y_tile_index * self.tile_size_m)

        return x, y

    def rendering_tile_in_bounds(self, min_x, min_y, max_x, max_y):
        # INPUT: coordinate x,y(corner basso sinistro) e x2,y2(corner alto destro) in metri della visualizzazione della mappa
        # OUTPUT: array di tutte le subswath che cadono all'iterno della visualizzazione

        # aggiorno i bounds globali della mappa in visualizzazione
        self.lat_min_map = min_y
        self.lat_max_map = max_y
        self.lon_min_map = min_x
        self.lon_max_map = max_x

        # LISTA SUBSWATH CHE SONO IN VISUALIZZAZIONE
        subswath_in_bounds = set()
        # LISTA INDICI TILE CHE SONO IN VISUALIZZAZIONE
        list_index_tile_x = set()
        list_index_tile_y = set()
        list_index_tile = set()

        # mi creo la funzione per la trasformazione 
        # (x - min_x_globale) / tile_size_m | floor e poi int() == per y ---> x_tile_min
        # (x2 - min_x_globale) / tile_size_m | floor e poi int() == per y2 ---> x_tile_max
        #index_tile_min = self.from_point_to_index_tile((min_x, min_y))
        #index_tile_max = self.from_point_to_index_tile((max_x, max_y))
        #print(f"index_tile_min: {index_tile_min}, index_tile_max: {index_tile_max}")

        # controllo l'asse x --> se ci ricadono tutti i tile lascio tutti gli indici
        #if index_tile_min < 0:
        #    index_tile_min = 0
        #if index_tile_max > self.total_tiles-1:
        #    index_tile_max = self.total_tiles-1

        # mi calcolo gli indici per la y e la x
        x_map_index = int((min_x - self.min_x) / self.tile_size_m) 
        y_map_index = int((min_y - self.min_y) / self.tile_size_m)
        index_map_y_min = y_map_index * self.total_tiles_y + x_map_index
        index_map_x_min = x_map_index * self.total_tiles_y + y_map_index
        print(f"MAP --> index_tile y min : {index_map_y_min}, index_tile x min : {index_map_x_min}")
        x_map_index = int((max_x - self.min_x) / self.tile_size_m) 
        y_map_index = int((max_y - self.min_y) / self.tile_size_m)
        index_map_y_max = y_map_index * self.total_tiles_y + x_map_index
        index_map_x_max = x_map_index * self.total_tiles_y + y_map_index
        print(f"MAP --> index_tile y max : {index_map_y_max}, index_tile x max : {index_map_x_max}")

        # trovo gli indici x e y max per la mia area --> il min è lo 0
        x_area_index = int((self.max_x - self.min_x) / self.tile_size_m) 
        y_area_index = int((self.max_y - self.min_y) / self.tile_size_m)
        index_area_x_max = x_area_index * self.total_tiles_y + y_area_index
        index_area_y_max = y_area_index * self.total_tiles_y + x_area_index
        print(f"AREA --> index_area_y_min: {0}, index_area_x_min: {0}")
        print(f"AREA --> index_area_y_max: {index_area_y_max}, index_area_x_max: {index_area_x_max}")

        # FILTRO SULL'ASSE X
        # 1. controlle se non abbiamo nessun tile all'interno del range
        if index_map_x_max < 0:
            return subswath_in_bounds, list_index_tile
        if index_map_x_min > index_area_x_max:
            return subswath_in_bounds, list_index_tile
        
        # 2. se arrivo qua vuol dire che i tile o ci sono tutti o in parte
        # 2.1 controllo se ci sono tutti i tile
        if index_map_x_min < 0 and index_map_x_max > index_area_x_max: # --> tutti i tile sono in visualizzazione
            # aggiungo tutti gli indici nella lista
            for i in range(self.total_tiles):
                list_index_tile_x.add(i)
        else:
            # 2.2 i tile ci sono in parte e mi calcolo gli indici dei tile che sono in visualizzazione
            # - trovo l'index che taglia i tile sulla x(se quello max o min)
            for c in range(self.total_tiles_x):
                # controllo se il tile è in visualizzazione
                c_i = c * self.total_tiles_y
                #print(f"ANALIZZO tile sulla x: {c_i}")
                i_x, i_y = self.from_index_to_point(c_i)
                i_x_t = int((i_x - self.min_x) / self.tile_size_m) 
                i_y_t = int((i_y - self.min_y) / self.tile_size_m)
                i_x_tile = i_x_t * self.total_tiles_y + i_y_t
                #print(f"TILE --> Indice tile sulla x: {i_x_tile}")
                if i_x_tile >= index_map_x_min and i_x_tile <= index_map_x_max:
                    # aggiungo tutti gli indici nella lista c e la sua line di indici sulla x
                    # se cado all'interno prendo dall'indice x tutta la y
                    #print("La collona ricade nella visualizzazione")
                    for y in range(self.total_tiles_y):
                        ind_valido = c_i + y
                        #print(f"INDEX VALIDO: {ind_valido}")
                        list_index_tile_x.add(ind_valido)
        #print(f">>>>> list_index_tile_x: {list_index_tile_x}")
        # FILTRO SULL'ASSE Y
        # 1. controlle se non abbiamo nessun tile all'interno del range
        if index_map_y_max < 0:
            return subswath_in_bounds, list_index_tile
        if index_map_y_min > index_area_y_max:
            return subswath_in_bounds, list_index_tile
        
        # 2. se arrivo qua vuol dire che i tile o ci sono tutti o in parte
        # 2.1 controllo se ci sono tutti i tile
        if index_map_y_min < 0 and index_map_y_max > index_area_y_max: # --> tutti i tile sono in visualizzazione
            # aggiungo tutti gli indici nella lista
            for i in range(self.total_tiles):
                list_index_tile_y.add(i)
        else:
            # 2.2 i tile ci sono in parte e mi calcolo gli indici dei tile che sono in visualizzazione
            # - trovo l'index che taglia i tile sulla y(se quello max o min)
            for c in range(self.total_tiles_y):
                # controllo se il tile è in visualizzazione
                i_x, i_y = self.from_index_to_point(c)
                i_x_t = int((i_x - self.min_x) / self.tile_size_m) 
                i_y_t = int((i_y - self.min_y) / self.tile_size_m)
                i_y_tile = i_y_t * self.total_tiles_y + i_x_t
                #print(f"Indice tile sulla y: {i_y_tile}")
                if i_y_tile >= index_map_y_min and i_y_tile <= index_map_y_max:
                    # aggiungo tutti gli indici nella lista c e la sua line di indici sulla x
                    for x in range(self.total_tiles_x):
                        ind_valido = c + self.total_tiles_y * x
                        list_index_tile_y.add(ind_valido)
        #print(f">>>>> list_index_tile_y: {list_index_tile_y}")


        # for(range(x_tile_min, x_tile_max+1))  ---> i_x
        #    for(range(y_tile_min, y_tile_max+1)) ---> i_y
        #        id = i_x + i_y * tiles_x_total

        #        prendi il dizionario: self.dic_tile[id] e mi prendo le subswath(list) all'interno del tile
        #        append alla lista
        #for index_tile in range(index_tile_min, index_tile_max+1):
        
        # prendo gli indici che ci sono in entrambe le liste
        list_index_tile = list_index_tile_x.intersection(list_index_tile_y)
        print(f">>>>> list_index_tile: {list_index_tile}")
        for index_tile in list_index_tile:
            for subswath in self.dic_tile[index_tile].all_id_subswath:
                subswath_in_bounds.add(subswath)
                #subswath_in_bounds.append(subswath)

        #LISTA -set- : insieme delle subswath che cadono all'interno della nostra visualizzazione

        print(f"subswath_in_bounds: {subswath_in_bounds}")  # LISTA DELLE MIE SUBSWATH CHE CADONO ALL'INTERNO DELLA VISUALIZZAZIONE
        return subswath_in_bounds, list_index_tile
    
    # MI CREO UNA STRUTTURA DOVE SE PRENDO 1 TILE CONOSCO TUTTE LE SUBSWATH CHE CADONO ALL'INTERNO DI ESSO
    # E SE PRENDO 1 SUBSWATH CONOSCO TUTTI I TILE CHE CADONO ALL'INTERNO DI ESSA
    def popolo_tile(self):
        # CICLO SURVEY MANAGER
        n_survey = self.survey_manager.get_num_tot_survey()
        for i_survey in range(n_survey):
            n_swath = self.survey_manager[i_survey].get_num_tot_swath()
            for i_swath in range(n_swath):
                n_subswath = self.survey_manager[i_survey][i_swath].get_num_tot_subswath()
                for i_subswath in range(n_subswath):
                    #print(f"Ciclo subswath {i_subswath}:")

                    # lista subswath che cattura i tile
                    list_tile_where_subswath_is = [] 

                    # APRO IL POLYGON DELLO SubSwath  e recupero i punti
                    geojson_path = self.survey_manager[i_survey][i_swath][i_subswath].get_subswath_area() # da sostituire con il path del subswath
                    with open(geojson_path, 'r') as f:
                        geojson = json.load(f)
                    polygon = geojson['features'][0]['geometry']['coordinates'][0]  # EPSG:32632

                    # Contemporaneamenti creo i due array 

                    # ciclo la lista di punti georef e guardo a quale tile appartiene e mi prendo l'id
                    for point in polygon:

                        index_tile = self.from_point_to_index_tile(point)
                        list_tile_where_subswath_is.append(index_tile)

                    # array per tutte le tile coperte del subswath
                    # lista in set e evito i duplicati
                    list_tile_where_subswath_is = list(set(list_tile_where_subswath_is))
                    # questo array lo salvo nella subswath
                    self.survey_manager[i_survey][i_swath][i_subswath].set_list_tile_where_subswath_is(list_tile_where_subswath_is)
                        
                    # ciclo sull'array di tile e per ogni tile vado nel dizionare e faccio l'append dell'id della subswath (cambiare metodo id subswath per non avere duplicati)
                    # ciclo sull'array di tile e per ogni tile vado nel dizionare e faccio l'append dell'id della swath (cambiare metodo id swath per non avere duplicati)
                    # [(n,n,n), (n,n,n), ..., (n,n,n)] -- indicano survey, swath, subswath
                    for index_tile in list_tile_where_subswath_is:
                        self.dic_tile[index_tile].add_id_subswath(i_survey, i_swath, i_subswath)

        # salvo le liste dei tile --> DA FARE 
        # ciclo il dizionario e prendo all_id_subswath 
        # mi creo il dizionario con id Tile preso dall'indice dal dizionario e all_id_subswath come valore
        # devo creare un unico dizionario
        dic_tile_id_subswath = {}
        for index_tile in self.dic_tile:
            dic_tile_id_subswath[index_tile] = self.dic_tile[index_tile].all_id_subswath
        # salvo il dizionario in un file json
        with open(os.path.join(tiles_dir, "list_subswath_each_tile.json"), "w") as f:
            json.dump(dic_tile_id_subswath, f)

    # PARTENDO DAI PUNTI BOUNDS GLOBALI MI CREO LA GRIGLIA DI TILE CON QUADRATI DI NXN METRI 
    # COSTRUISCO IL MIO DIZIONARIO CON ALL'INTERNO LE TILE 'id_tile'   
    def create_tiles(self):
        """
        Crea tutti i tile con i loro bounds specifici.
        Ogni tile riceve i suoi vertici calcolati in base alla posizione nella griglia.
        """
        # Calcola le dimensioni totali
        width_m = abs(self.max_x - self.min_x)
        height_m = abs(self.max_y - self.min_y)
        
        # Calcola quanti tile completi ci stanno per dimensione
        tiles_x_complete = int(width_m / self.tile_size_m)
        tiles_y_complete = int(height_m / self.tile_size_m)
        
        # Calcola le dimensioni residue
        remainder_x = width_m % self.tile_size_m
        remainder_y = height_m % self.tile_size_m
        
        # Calcola il numero totale di tile per dimensione
        tiles_x_total = tiles_x_complete + (1 if remainder_x > 0 else 0)
        tiles_y_total = tiles_y_complete + (1 if remainder_y > 0 else 0)
        
        tile_id = 0
        
        # Crea tutti i tile con i loro bounds specifici
        for i in range(tiles_x_total):
            for j in range(tiles_y_total):
                # Calcola le coordinate di inizio del tile
                x_start = self.min_x + (i * self.tile_size_m)
                y_start = self.min_y + (j * self.tile_size_m)
                
                # Calcola le coordinate di fine del tile
                #if i < tiles_x_complete:
                    # Tile completo in larghezza
                    #x_end = x_start + self.tile_size_m
                #else:
                    # Tile residuo in larghezza
                    #x_end = self.max_x
                
                #if j < tiles_y_complete:
                    # Tile completo in altezza
                    #y_end = y_start + self.tile_size_m
                #else:
                    # Tile residuo in altezza
                    #y_end = self.max_y

                x_end = x_start + self.tile_size_m #  per la larghezza
                y_end = y_start + self.tile_size_m #  per l'altezza
                
                # Crea i bounds del tile specifico
                # Formato: [(y_max, x_min), (y_max, x_max), (y_min, x_max), (y_min, x_min)]
                #          [    1    ,     2    ,     3    ,     4    ]
                tile_bounds = [
                    (y_end, x_start),    # Vertice 1: alto-sinistra
                    (y_end, x_end),      # Vertice 2: alto-destra  
                    (y_start, x_end),    # Vertice 3: basso-destra
                    (y_start, x_start)   # Vertice 4: basso-sinistra
                ]
                
                # Crea il tile con i suoi bounds specifici
                import numpy as np
                rgb = np.random.rand(3)*255  # matrice 3x1
                self.dic_tile[tile_id] = Tile(tile_bounds, rgb)
                #self.dic_tile[tile_id].save_point_bounds() --> MI DISEGNA I TILE   
                tile_id += 1
        
        print(f"Creati {len(self.dic_tile)} tile con bounds specifici:")
        print(f"- Griglia: {tiles_x_total} x {tiles_y_total}")
        self.total_tiles_x = tiles_x_total
        self.total_tiles_y = tiles_y_total
        print(f"- Tile completi: {tiles_x_complete * tiles_y_complete}")
        if remainder_x > 0 or remainder_y > 0:
            print(f"- Tile parziali: {len(self.dic_tile) - (tiles_x_complete * tiles_y_complete)}")
    def _calculate_total_tiles(self):
        """
        Calcola il numero totale di tile che coprono completamente i bounds_zone
        Include tile 10x10m complete e tile parziali per le aree residue
        Le coordinate sono già in metri (sistema UTM 32632)
        """
        # Le coordinate sono già in metri, calcola direttamente le dimensioni
        width_m = abs(self.max_x - self.min_x)  # Larghezza in metri
        height_m = abs(self.max_y - self.min_y)  # Altezza in metri
        
        # Calcola il numero di tile complete per ogni dimensione
        tiles_x_complete = int(width_m / self.tile_size_m)
        tiles_y_complete = int(height_m / self.tile_size_m)
        
        # Calcola se ci sono aree residue che richiedono tile parziali
        remainder_x = width_m % self.tile_size_m  # Area residua in X
        remainder_y = height_m % self.tile_size_m  # Area residua in Y
        
        # Calcola il numero totale di tile necessarie (incluse quelle parziali)
        tiles_x_total = tiles_x_complete + (1 if remainder_x > 0 else 0)
        tiles_y_total = tiles_y_complete + (1 if remainder_y > 0 else 0)
        
        total = tiles_x_total * tiles_y_total
        
        print(f"Bounds zone: {width_m:.2f}m x {height_m:.2f}m")
        print(f"Tile complete 10x10m: {tiles_x_complete} x {tiles_y_complete} = {tiles_x_complete * tiles_y_complete}")
        
        if remainder_x > 0 or remainder_y > 0:
            print(f"Aree residue: X={remainder_x:.2f}m, Y={remainder_y:.2f}m")
            print(f"Tile totali (incluse parziali): {tiles_x_total} x {tiles_y_total} = {total}")
        else:
            print(f"Nessuna area residua - Tile totali: {total}")
        
        return total
    

    def export_tiles_to_shapefile(self, filename="tiles_bounds"):
        """
        Esporta i bounds di tutti i tile come shapefile nella cartella trash.
        Crea i file .shp, .shx, .dbf, .prj manualmente.
        
        Args:
            filename (str): Nome base del file shapefile (senza estensione)
        """
        if not self.dic_tile:
            print("Errore: Nessun tile creato. Esegui prima create_tiles()")
            return False
        
        # Assicurati che la cartella trash esista
        trash_dir = "trash"
        if not os.path.exists(trash_dir):
            os.makedirs(trash_dir)
        
        # Calcola le dimensioni totali per ricreare la griglia
        width_m = abs(self.max_x - self.min_x)
        height_m = abs(self.max_y - self.min_y)
        tiles_x_complete = int(width_m / self.tile_size_m)
        tiles_y_complete = int(height_m / self.tile_size_m)
        
        # Lista per contenere i dati dei tile
        tiles_data = []
        
        # Ricreo i bounds per ogni tile basandomi sulla griglia
        tile_id = 0
        for i in range(tiles_x_complete + (1 if width_m % self.tile_size_m > 0 else 0)):
            for j in range(tiles_y_complete + (1 if height_m % self.tile_size_m > 0 else 0)):
                # Calcola le coordinate di inizio del tile
                x_start = self.min_x + (i * self.tile_size_m)
                y_start = self.min_y + (j * self.tile_size_m)
                
                # Calcola le coordinate di fine del tile
                #if i < tiles_x_complete:
                    #x_end = x_start + self.tile_size_m
                #else:
                    #x_end = self.max_x
                
                #if j < tiles_y_complete:
                    #y_end = y_start + self.tile_size_m
                #else:
                    #y_end = self.max_y

                x_end = x_start + self.tile_size_m
                y_end = y_start + self.tile_size_m
                
                # Calcola le dimensioni del tile
                tile_width = abs(x_end - x_start)
                tile_height = abs(y_end - y_start)
                tile_area = tile_width * tile_height
                
                # Determina il tipo di tile
                is_complete_x = abs(tile_width - self.tile_size_m) < 0.001
                is_complete_y = abs(tile_height - self.tile_size_m) < 0.001
                
                if is_complete_x and is_complete_y:
                    tile_type = "completo"
                elif is_complete_x and not is_complete_y:
                    tile_type = "residuo_y"
                elif not is_complete_x and is_complete_y:
                    tile_type = "residuo_x"
                else:
                    tile_type = "residuo_xy"
                
                # Aggiungi i dati del tile
                tiles_data.append({
                    'tile_id': tile_id,
                    'x_min': x_start,
                    'y_min': y_start,
                    'x_max': x_end,
                    'y_max': y_end,
                    'width_m': round(tile_width, 2),
                    'height_m': round(tile_height, 2),
                    'area_m2': round(tile_area, 2),
                    'tile_type': tile_type,
                    'tile_size_m': self.tile_size_m,
                    'grid_i': i,
                    'grid_j': j
                })
                
                tile_id += 1
        
        # Salva come GeoJSON (più semplice da implementare)
        self._save_as_geojson(tiles_data, os.path.join(trash_dir, f"{filename}.geojson"))
        
        # Salva anche come CSV per compatibilità
        self._save_as_csv(tiles_data, os.path.join(trash_dir, f"{filename}.csv"))
        
        print(f"File esportati con successo in {trash_dir}/:")
        print(f"- {filename}.geojson (formato GeoJSON)")
        print(f"- {filename}.csv (dati tabulari)")
        print(f"- {len(tiles_data)} tile esportati")
        print(f"- Sistema di coordinate: UTM Zone 32N (EPSG:32632)")
        print(f"- Tile size: {self.tile_size_m}m x {self.tile_size_m}m")
        
        return True    
    def _save_as_geojson(self, tiles_data, filepath):
        """
        Salva i dati dei tile come file GeoJSON
        """
        features = []
        
        for tile in tiles_data:
            # Crea il poligono del tile
            coordinates = [[
                [tile['x_min'], tile['y_min']],  # Basso-sinistra
                [tile['x_max'], tile['y_min']],  # Basso-destra
                [tile['x_max'], tile['y_max']],  # Alto-destra
                [tile['x_min'], tile['y_max']],  # Alto-sinistra
                [tile['x_min'], tile['y_min']]   # Chiudi il poligono
            ]]
            
            feature = {
                "type": "Feature",
                "properties": {
                    "tile_id": tile['tile_id'],
                    "x_min": tile['x_min'],
                    "y_min": tile['y_min'],
                    "x_max": tile['x_max'],
                    "y_max": tile['y_max'],
                    "width_m": tile['width_m'],
                    "height_m": tile['height_m'],
                    "area_m2": tile['area_m2'],
                    "tile_type": tile['tile_type'],
                    "tile_size_m": tile['tile_size_m'],
                    "grid_i": tile['grid_i'],
                    "grid_j": tile['grid_j']
                },
                "geometry": {
                    "type": "Polygon",
                    "coordinates": coordinates
                }
            }
            features.append(feature)
        
        geojson = {
            "type": "FeatureCollection",
            "crs": {
                "type": "name",
                "properties": {
                    "name": "EPSG:32632"
                }
            },
            "features": features
        }
        
        with open(filepath, 'w', encoding='utf-8') as f:
            json.dump(geojson, f, indent=2, ensure_ascii=False) 
    def _save_as_csv(self, tiles_data, filepath):
        """
        Salva i dati dei tile come file CSV
        """
        with open(filepath, 'w', encoding='utf-8') as f:
            # Header
            f.write("tile_id,x_min,y_min,x_max,y_max,width_m,height_m,area_m2,tile_type,tile_size_m,grid_i,grid_j\n")
            
            # Data
            for tile in tiles_data:
                f.write(f"{tile['tile_id']},{tile['x_min']},{tile['y_min']},{tile['x_max']},{tile['y_max']},"
                       f"{tile['width_m']},{tile['height_m']},{tile['area_m2']},{tile['tile_type']},"
                       f"{tile['tile_size_m']},{tile['grid_i']},{tile['grid_j']}\n")
    
    def create_volume_raster(self, on_tile_completed=None):
        # in questa funzione devo creare un volume raster dei miei tile contenenti le porzioni di subswath
        # 1) ciclo sui tile
        # 2) per ogni tile mi prendo gli indici delle subswath che cadono all'interno del tile
        # 3) creo una griglia fissa per tile con GSD = 0.2 m e rasterizzo le subswath clippando ai 4 vertici del tile
        import os
        import numpy as np
        from shapely.geometry import Polygon
        from shapely.geometry import Point
        from rasterio.transform import from_bounds
        from GLOBAL import tiles_dir
        from Algorithm.create_rastert import create_raster
        from pyproj import Transformer
        from rasterio.warp import reproject, Resampling
        from scipy.ndimage import binary_closing, distance_transform_edt

        # Parametri
        gsd_m = 0.02  # GSD fisso in metri --> 2 cm

        # Cache precomputazioni per subswath (riutilizzate su tutti i tile e depth)
        sub_precomp_cache = {}
        to_lonlat = Transformer.from_crs("EPSG:32632", "EPSG:4326", always_xy=True)

        def get_sub_precomp(id_survey, id_swath, id_subswath):
            key = (id_survey, id_swath, id_subswath)
            if key in sub_precomp_cache:
                return sub_precomp_cache[key]
            # polygon
            with open(self.survey_manager[id_survey][id_swath][id_subswath].get_subswath_area(), 'r') as f:
                gj = json.load(f)
            coords_xy = gj['features'][0]['geometry']['coordinates'][0]
            subs_poly = Polygon(coords_xy)
            # georef
            y_path = self.survey_manager[id_survey][id_swath][id_subswath].path_georef_matrix_lat
            x_path = self.survey_manager[id_survey][id_swath][id_subswath].path_georef_matrix_lon
            y_m = np.load(y_path)
            x_m = np.load(x_path)
            # manholes
            manholes_path = self.survey_manager[id_survey][id_swath][id_subswath].manholes_path
            with open(manholes_path, 'r') as f:
                manholes = json.load(f)
            # ricreo una lista che dato un manhols x= righe di x_m e y_m e z= colonne di x_m e y_m
            # La lista sarà una lista di coordinate (lat, lon) dei miei manholes
            manholes_list = []
            for manhole in manholes:
                manholes_list.append((float(x_m[manhole['x']][manhole['z']]), float(y_m[manhole['x']][manhole['z']])))
            print(f"Manholes list: {manholes_list}")
            # depth count
            try:
                depth_count = len(self.survey_manager[id_survey][id_swath][id_subswath].file_tomography_inferenza_list)
            except Exception:
                depth_count = 0
            # transform del raster subswath in EPSG:4326 (coerente con create_raster)
            lon, lat = to_lonlat.transform(x_m.ravel(), y_m.ravel())
            lon = lon.reshape(x_m.shape)
            lat = lat.reshape(y_m.shape)
            delta_lon = float(lon.max() - lon.min())
            delta_lat = float(lat.max() - lat.min())
            risoluzione = 800  # per inferenza
            if delta_lon > delta_lat:
                res_x = int(risoluzione)
                res_y = int((delta_lat / (delta_lon + 1e-12)) * risoluzione)
            else:
                res_x = int((delta_lon / (delta_lat + 1e-12)) * risoluzione)
                res_y = int(risoluzione)
            res_x = max(res_x, 1)
            res_y = max(res_y, 1)
            src_transform = from_bounds(float(lon.min()), float(lat.min()), float(lon.max()), float(lat.max()), res_x, res_y)
            sub_precomp_cache[key] = {
                "polygon": subs_poly,
                "x_m": x_m,
                "y_m": y_m,
                "src_transform": src_transform,
                "depth_count": depth_count,
                "manholes_list": manholes_list
            }
            return sub_precomp_cache[key]

        # Pre-calcolo: funzione per bounds tile e poligono tile (EPSG:32632)
        def tile_bounds_xy(bounds_zone):
            # bounds_zone: [(y_max,x_min),(y_max,x_max),(y_min,x_max),(y_min,x_min)]
            xs = [pt[1] for pt in bounds_zone]
            ys = [pt[0] for pt in bounds_zone]
            return min(xs), min(ys), max(xs), max(ys)

        def tile_polygon(bounds_zone):
            x_min, y_min, x_max, y_max = tile_bounds_xy(bounds_zone)
            # shapely vuole [(x,y), ...] con anello chiuso
            return Polygon([(x_min, y_min), (x_max, y_min), (x_max, y_max), (x_min, y_max), (x_min, y_min)])

        # Ciclo su tutti i tile
        for tile_id, tile in self.dic_tile.items():
            subs_list = tile.all_id_subswath
            if not subs_list:
                continue

            # Bounds e griglia del tile
            x_min, y_min, x_max, y_max = tile_bounds_xy(tile.bounds_zone)
            width_m = x_max - x_min
            height_m = y_max - y_min

            # Assumo che la dimensione tile sia multiplo intero del GSD
            nx = int(round(width_m / gsd_m))
            ny = int(round(height_m / gsd_m))
            if nx <= 0 or ny <= 0:
                continue

            # Trasform per la maschera raster
            transform = from_bounds(x_min, y_min, x_max, y_max, nx, ny)

            # Cartella di output per il tile e salvataggio trasformazione pixel->CRS (una volta sola)
            out_dir = os.path.join(tiles_dir, f"tile_{tile_id:06d}")
            os.makedirs(out_dir, exist_ok=True)
            transform_info = {
                "crs": "EPSG:32632",
                "affine": list(transform),  # [a,b,c,d,e,f] per x_map=a*col+b*row+c; y_map=d*col+e*row+f
                "nx": int(nx),
                "ny": int(ny),
                "bounds": {"x_min": float(x_min), "y_min": float(y_min), "x_max": float(x_max), "y_max": float(y_max)},
                "gsd_m": float(gsd_m)
            }
            with open(os.path.join(out_dir, "transform_px_to_crs.json"), "w", encoding="utf-8") as f:
                json.dump(transform_info, f, indent=2, ensure_ascii=False)

            tile_poly = tile_polygon(tile.bounds_zone)

            # Determino quante depth processare (uso il minimo comune tra le subs presenti nel tile)
            depth_counts = []
            subs_candidates = []
            manholes_candidates = []
            for (id_survey, id_swath, id_subswath) in subs_list:
                try:
                    pre = get_sub_precomp(id_survey, id_swath, id_subswath)
                except Exception:
                    continue
                if tile_poly.intersects(pre["polygon"]) is False:
                    continue
                if pre["depth_count"] > 0:
                    depth_counts.append(pre["depth_count"])
                    subs_candidates.append((id_survey, id_swath, id_subswath))
                    manholes_candidates += pre["manholes_list"]
                    print(f"Manholes candidates: {manholes_candidates}")
            if not depth_counts:
                continue
            max_depth = min(depth_counts)
            # Limite di profondità a 250 cm
            max_depth = min(max_depth, 200)

            # FILTRO DALLA LISTA DEI MANHOLES CI SONO MANHOLES CHE CADONO ALL'INTERNO DEL TILE
            for manhole in manholes_candidates:
                if not tile_poly.contains(Point(manhole[0], manhole[1])):
                    # TOLGO LA COORD DA MANHOLES CANDIDATES
                    manholes_candidates.remove(manhole)
            print(f"Manholes candidates filtrato: {manholes_candidates}")

            # Loop sulle profondità
            for current_depth_idx in range(max_depth):
                # Prepara immagine del tile per questa depth
                tile_img = np.full((ny, nx), np.nan, dtype=np.float32)

                for (id_survey, id_swath, id_subswath) in subs_candidates:
                    pre = sub_precomp_cache[(id_survey, id_swath, id_subswath)]
                    # Rasterizzazione subswath con pipeline esistente (veloce e corretta)
                    try:
                        tomo = self.survey_manager[id_survey][id_swath][id_subswath].tomography_inferenza(current_depth_idx)
                    except Exception:
                        continue
                    sub_arr = create_raster(tomo, pre["x_m"], pre["y_m"], self.survey_manager[id_survey][id_swath][id_subswath].path_polygon_subswath_area, splat='nearest', tomo_inferenza=True)
                    if sub_arr is None:
                        continue

                    # Riproiezione su griglia del tile (nearest). Last-wins: sovrascrive dove il nuovo ha valore.
                    tmp = np.full((ny, nx), np.nan, dtype=np.float32)
                    reproject(
                        source=sub_arr.astype(np.float32),
                        destination=tmp,
                        src_transform=pre["src_transform"],
                        src_crs="EPSG:4326",
                        dst_transform=transform,
                        dst_crs="EPSG:32632",
                        src_nodata=np.nan,
                        dst_nodata=np.nan,
                        resampling=Resampling.nearest
                    )
                    m = ~np.isnan(tmp)
                    if np.any(m):
                        tile_img[m] = tmp[m]

                # Se tutto NaN, salto salvataggio
                if np.isnan(tile_img).all():
                    continue

                # Post-processing: riempimento buchi piccoli tramite closing binario
                # Kernel 4x4 (≈ 3/4 px). Parametrizzabile se serve.
                mask_valid = ~np.isnan(tile_img)
                structure = np.ones((12, 12), dtype=bool)
                mask_closed = binary_closing(mask_valid, structure=structure)
                fill_mask = mask_closed & ~mask_valid
                if np.any(fill_mask):
                    # Copia del valore del vicino valido più vicino (nearest) solo per i pixel da riempire
                    _, (iy, ix) = distance_transform_edt(~mask_valid, return_distances=True, return_indices=True)
                    tile_img[fill_mask] = tile_img[iy[fill_mask], ix[fill_mask]]

                # Salvataggio su disco
                out_raster = os.path.join(out_dir, f"depth_{current_depth_idx:03d}.npy")
                np.save(out_raster, tile_img)

                # opzionale: GeoTIFF
                # if False:
                if True:
                    import rasterio
                    # Converti a uint8 per compatibilità
                    tile_img_uint8 = np.full((ny, nx), 0, dtype=np.uint8)
                    valid_mask = ~np.isnan(tile_img)
                    if np.any(valid_mask):
                        # Normalizza i valori validi a [0, 255]
                        valid_values = tile_img[valid_mask]
                        min_val = np.min(valid_values)
                        max_val = np.max(valid_values)
                        if max_val > min_val:
                            normalized = (valid_values - min_val) / (max_val - min_val)
                            tile_img_uint8[valid_mask] = (normalized * 255).astype(np.uint8)

                    with rasterio.open(os.path.join(out_dir, f"depth_{current_depth_idx:03d}cm.tiff"), "w", driver="GTiff", height=ny, width=nx, count=1, dtype=np.uint8, crs="EPSG:32632", transform=transform, nodata=0) as dst:
                        dst.write(tile_img_uint8, 1)

            # ORA ATTRAVERSO LA TRANSFORM DEL TIF CHE STO CREANDO, RIPORTO I MANHOLES CON COORDINATE PIXEL DEL TILE
            # quindi da una lista di (lat, lon) mi creo una lista di (x, y)
            manholes_list_pixel = []
            for manhole in manholes_candidates:
                x_pixel = int((manhole[0] - transform.c) / transform.a)
                y_pixel = int((manhole[1] - transform.f) / transform.e)
                manholes_list_pixel.append((x_pixel, y_pixel))
            print(f"Manholes candidates pixel: {manholes_list_pixel}")
            # ora salvo il file json nel formato richiesto
            json_tombini = {
                "Offset": {
                    "Left": 0,
                    "Top": 0,
                    "Z": 0
                }
            }
            # manholes numerati con index e key "centerPoint": [[x,y]]
            for idx, (px, py) in enumerate(manholes_list_pixel):
                json_tombini[str(idx)] = {"centerPoint": [[int(px), int(py)]]}
            # salvo il json accanto al raster del tile
            manholes_json_path = os.path.join(out_dir, "final_tombini.json")
            with open(manholes_json_path, "w", encoding="utf-8") as f:
                json.dump(json_tombini, f, indent=2, ensure_ascii=False)



            # Fine tile: scrivo marker di completamento e invoco callback
            print(f"[TileManager] Tile {tile_id} completato: {max_depth} depth salvate in {out_dir}")
            try:
                complete_meta = {
                    "tile_id": int(tile_id),
                    "expected_depth": int(max_depth),
                    "written_depth": int(max_depth),
                }
                marker_path = os.path.join(out_dir, "tile_complete.json")
                with open(marker_path, "w", encoding="utf-8") as f:
                    json.dump(complete_meta, f, indent=2, ensure_ascii=False)
                print(f"[TileManager] Marker salvato: {marker_path}")
            except Exception as e:
                print(f"[TileManager] Errore salvataggio marker tile {tile_id}: {e}")
            
            if callable(on_tile_completed):
                try:
                    print(f"[TileManager] Invoco callback per tile {tile_id}")
                    on_tile_completed(tile_id, out_dir, max_depth)
                    print(f"[TileManager] Callback completato per tile {tile_id}")
                except Exception as e:
                    print(f"[TileManager] Errore callback tile {tile_id}: {e}")
            else:
                print(f"[TileManager] Nessun callback per tile {tile_id}")
        print(f"END")








    def singola_depth_funzionante(self):
        # in questa funzione devo creare un volume raster dei miei tile contenenti le porzioni di subswath
        # 1) ciclo sui tile
        # 2) per ogni tile mi prendo gli indici delle subswath che cadono all'interno del tile
        # 3) creo una griglia fissa per tile con GSD = 0.2 m e rasterizzo le subswath clippando ai 4 vertici del tile
        import os
        import numpy as np
        from shapely.geometry import Polygon
        from rasterio.transform import from_bounds
        from GLOBAL import tiles_dir
        from Algorithm.create_rastert import create_raster
        from pyproj import Transformer
        from rasterio.warp import reproject, Resampling

        # Parametri
        gsd_m = 0.02  # GSD fisso in metri -->2 cm

        # ciclo su tutti i depth della tomografia dell'inferenza
        len(file_tomography_inferenza_list)

        current_depth_idx = self.survey_manager.get_depth()
        current_depth_idx = 45

        # Pre-calcolo: funzione per bounds tile e poligono tile (EPSG:32632)
        def tile_bounds_xy(bounds_zone):
            # bounds_zone: [(y_max,x_min),(y_max,x_max),(y_min,x_max),(y_min,x_min)]
            xs = [pt[1] for pt in bounds_zone]
            ys = [pt[0] for pt in bounds_zone]
            return min(xs), min(ys), max(xs), max(ys)

        def tile_polygon(bounds_zone):
            x_min, y_min, x_max, y_max = tile_bounds_xy(bounds_zone)
            # shapely vuole [(x,y), ...] con anello chiuso
            return Polygon([(x_min, y_min), (x_max, y_min), (x_max, y_max), (x_min, y_max), (x_min, y_min)])

        # Ciclo su tutti i tile
        for tile_id, tile in self.dic_tile.items():
            subs_list = tile.all_id_subswath
            if not subs_list:
                continue

            # Bounds e griglia del tile
            x_min, y_min, x_max, y_max = tile_bounds_xy(tile.bounds_zone)
            width_m = x_max - x_min
            height_m = y_max - y_min

            # Assumo che la dimensione tile sia multiplo intero del GSD
            nx = int(round(width_m / gsd_m))
            ny = int(round(height_m / gsd_m))
            if nx <= 0 or ny <= 0:
                continue

            # Trasform per la maschera raster
            transform = from_bounds(x_min, y_min, x_max, y_max, nx, ny)

            # Prepara immagine del tile: verrà composta sovrascrivendo per ogni subswath (last wins)
            tile_img = np.full((ny, nx), np.nan, dtype=np.float32)

            tile_poly = tile_polygon(tile.bounds_zone)

            # Reproject helper: dal raster subswath (EPSG:4326) alla griglia del tile (EPSG:32632)
            transformer = Transformer.from_crs("EPSG:32632", "EPSG:4326", always_xy=True)

            for (id_survey, id_swath, id_subswath) in subs_list:
                # Intersezione veloce per evitare lavori inutili
                try:
                    with open(self.survey_manager[id_survey][id_swath][id_subswath].get_subswath_area(), 'r') as f:
                        gj = json.load(f)
                    coords_xy = gj['features'][0]['geometry']['coordinates'][0]
                    subs_poly = Polygon(coords_xy)
                except Exception:
                    continue

                if tile_poly.intersects(subs_poly) is False:
                    continue

                # Carico georef e tomografia alla profondità corrente
                try:
                    y_path = self.survey_manager[id_survey][id_swath][id_subswath].path_georef_matrix_lat
                    x_path = self.survey_manager[id_survey][id_swath][id_subswath].path_georef_matrix_lon
                    y_m = np.load(y_path)
                    x_m = np.load(x_path)
                    tomo = self.survey_manager[id_survey][id_swath][id_subswath].tomography_inferenza(current_depth_idx)
                except Exception:
                    continue

                # Rasterizzazione subswath con pipeline esistente
                sub_arr = create_raster(tomo, x_m, y_m, self.survey_manager[id_survey][id_swath][id_subswath].path_polygon_subswath_area, splat='nearest', tomo_inferenza=True)
                if sub_arr is None:
                    continue

                # Ricavo il transform del raster subswath (EPSG:4326) replicando la stessa logica
                lon, lat = transformer.transform(x_m.ravel(), y_m.ravel())
                lon = lon.reshape(x_m.shape)
                lat = lat.reshape(y_m.shape)
                delta_lon = float(lon.max() - lon.min())
                delta_lat = float(lat.max() - lat.min())
                risoluzione = 800  # tomo_inferenza=True
                if delta_lon > delta_lat:
                    res_x = int(risoluzione)
                    res_y = int((delta_lat / (delta_lon + 1e-12)) * risoluzione)
                else:
                    res_x = int((delta_lon / (delta_lat + 1e-12)) * risoluzione)
                    res_y = int(risoluzione)
                res_x = max(res_x, 1)
                res_y = max(res_y, 1)
                src_transform = from_bounds(float(lon.min()), float(lat.min()), float(lon.max()), float(lat.max()), res_x, res_y)

                # Riproiezione su griglia del tile (nearest). Last-wins: sovrascrive dove il nuovo ha valore.
                tmp = np.full((ny, nx), np.nan, dtype=np.float32)
                reproject(
                    source=sub_arr.astype(np.float32),
                    destination=tmp,
                    src_transform=src_transform,
                    src_crs="EPSG:4326",
                    dst_transform=transform,
                    dst_crs="EPSG:32632",
                    src_nodata=np.nan,
                    dst_nodata=np.nan,
                    resampling=Resampling.nearest
                )
                m = ~np.isnan(tmp)
                if np.any(m):
                    tile_img[m] = tmp[m]

            # Salvataggio su disco
            out_dir = os.path.join(tiles_dir, f"tile_{tile_id:06d}")
            os.makedirs(out_dir, exist_ok=True)

            # Salvo il raster come .npy (float32 valori originali) e metadati JSON
            out_raster = os.path.join(out_dir, f"depth_{current_depth_idx:03d}_cm.npy")
            np.save(out_raster, tile_img)

            meta = {
                "tile_id": int(tile_id),
                "gsd_m": gsd_m,
                "nx": int(nx),
                "ny": int(ny),
                "bounds": {
                    "x_min": float(x_min),
                    "y_min": float(y_min),
                    "x_max": float(x_max),
                    "y_max": float(y_max)
                },
                "depth_index": int(current_depth_idx)
            }
            with open(os.path.join(out_dir, f"depth_{current_depth_idx:03d}_cm.json"), "w", encoding="utf-8") as f:
                json.dump(meta, f, indent=2, ensure_ascii=False)
        
            # lo salvo anche come raster tiff
            if True:
                import rasterio
                # Converti a uint8 per compatibilità
                tile_img_uint8 = np.full((ny, nx), 0, dtype=np.uint8)
                valid_mask = ~np.isnan(tile_img)
                if np.any(valid_mask):
                    # Normalizza i valori validi a [0, 255]
                    valid_values = tile_img[valid_mask]
                    min_val = np.min(valid_values)
                    max_val = np.max(valid_values)
                    if max_val > min_val:
                        normalized = (valid_values - min_val) / (max_val - min_val)
                        tile_img_uint8[valid_mask] = (normalized * 255).astype(np.uint8)

                # salvo il raster come tiff
                with rasterio.open(os.path.join(out_dir, f"depth_{current_depth_idx:03d}_cm.tiff"), "w", driver="GTiff", height=ny, width=nx, count=1, dtype=np.uint8, crs="EPSG:32632", transform=transform, nodata=0) as dst:
                    dst.write(tile_img_uint8, 1)
                # # salvo il raster come tiff (DISATTIVATO)
                # if False:
                #     with rasterio.open(os.path.join(out_dir, f"depth_{current_depth_idx:03d}_cm.tiff"), "w", driver="GTiff", height=ny, width=nx, count=1, dtype=np.uint8, crs="EPSG:32632", transform=transform, nodata=0) as dst:
                #         dst.write(tile_img_uint8, 1)
        return True
