# IMPORT PYSIDE6
from PySide6.QtWidgets import (QMainWindow, QMessageBox, QWidget, QSplitter, QLabel, QMenu, QScrollArea,
                               QPushButton, QToolBar, QCheckBox, QWidgetAction, QComboBox, QFrame, QDoubleSpinBox,
                               QTreeWidget, QTreeWidgetItem, QDialogButtonBox, QDialog, QSlider, QStackedWidget)  # oggetti
from PySide6.QtWidgets import QHBoxLayout, QVBoxLayout, QSizePolicy  # Layout
from PySide6.QtCore import Qt, QSize, Signal
from PySide6.QtGui import QIcon
# ALTRI IMPORT
import pyqtgraph as pg

# STYLE
from style.MainWindow.MainWindow_css import *
from style.MainWindow.Toolbar_css import *
from style.MainWindow.Menu_style import *
from style.MainWindow.Widget_css import *
from style.MainWindow.Radargram_css import *
from style.MainWindow.Filtri_css import *
from style.MainWindow.Select_survey_css import *

# GLOBAL
from GLOBAL import *

class MainWindow(QMainWindow):

    def __init__(self):
        super().__init__()

        # Impostazioni finestra
        self.setWindowTitle("XMAP")
        self.setWindowIcon(QIcon(icon_path+'/logo_xplora.png'))
        self.setMinimumSize(900, 540)
        self.setStyleSheet(window_style())

        # MENU
        self.createMenu()
        self.menu.setStyleSheet(menu_style())

        # TOOLBAR
        self.toolbar = QToolBar(self)
        self.createToolBar()
        self.toolbar.setStyleSheet(toolbar_style())

        # PAGES (Overview, Algorithm)
        self.pages = QStackedWidget()

        # Overview page (existing UI)
        self.page_overview = QWidget()
        overview_layout = QVBoxLayout(self.page_overview)

        # 1) Container UP
        self.container_up = QWidget()
        self.create_container_up()
        self.container_up.setStyleSheet(widget_style())

        # 2) Parte Radargramma
        self.radargram = QWidget()
        self.radargram.setStyleSheet(widget_style())
        self.create_radargram()

        splitter_vertical = QSplitter(Qt.Vertical)
        splitter_vertical.setStyleSheet(splitter_style())
        splitter_vertical.addWidget(self.container_up)
        splitter_vertical.addWidget(self.radargram)
        splitter_vertical.setSizes([600, 200])
        overview_layout.addWidget(splitter_vertical)

        # (Rimosso) Pagina Algorithm separata: useremo un unico layout e cambieremo solo il pannello destro

        self.pages.addWidget(self.page_overview)   # index 0

        # Set Overview as the initial page
        self.setCentralWidget(self.pages)

        self.showMaximized()

        # Collega le azioni del menu
        #self.menu_filter.triggered.connect(self.handle_filter_action)

    def createMenu(self):
        # OGGETTO MENU BAR CHE CREA IL MENU
        self.menu = self.menuBar()
        # 1. File
        self.file = self.menu.addMenu("File")
        self.import_survey_action = self.file.addAction("Import survey") 
        self.open_project_action = self.file.addAction("Open project")

        # 2. Export
        self.export = self.menu.addMenu("Export")
        tif_checkbox = QCheckBox("Esporta tif")
        tif_action = QWidgetAction(self)
        tif_action.setDefaultWidget(tif_checkbox)
        self.export.addAction(tif_action)

        json_checkbox = QCheckBox("Esporta json")
        json_action = QWidgetAction(self)
        json_action.setDefaultWidget(json_checkbox)
        self.export.addAction(json_action)
        self.export.addSeparator()

        self.export.addAction("Export Selected")

    def createToolBar(self):
        # set toolbar
        self.toolbar.setMovable(False)
        self.toolbar.setFloatable(False)
        self.toolbar.setFixedHeight(50)

        # Aggiunta dei pulsanti alla toolbar
        self.save = QPushButton("Save")
        self.save.setIcon(QIcon(icon_path + '/save_as.svg'))
        self.save.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Expanding)

        self.toolbar.addWidget(self.save)
        self.toolbar.addSeparator()
        self.addToolBar(Qt.TopToolBarArea, self.toolbar)

    def create_container_up(self):
        container_layout = QHBoxLayout()
        container_layout.setContentsMargins(0, 0, 0, 0)
        # SPLITTER CON I 3 QWIDGET (SURVEY - MAPPA - ALGORITMO)
        splitter_up = QSplitter(Qt.Horizontal)
        splitter_up.setStyleSheet(splitter_style())

        # SURVEY
        self.container_up_left()

        # MAPPA
        self.container_up_map()

        # ALGORTIMO (pannello destro con stack interno Overview/Algorithm)
        self.container_up_algorithm()

        # Aggiungiamo i pulsanti al QSplitter
        splitter_up.addWidget(self.leftWidget)
        splitter_up.addWidget(self.map_container)
        splitter_up.addWidget(self.algorithm)
        # imposto una dimensione iniziale
        splitter_up.setSizes([200, 1000, 200])

        # Aggiungi il QSplitter al layout orizzontale
        container_layout.addWidget(splitter_up)
        self.container_up.setLayout(container_layout)

    def container_up_left(self):
        self.leftWidget = QWidget()
        self.leftWidget.setStyleSheet(widget_style())
        self.leftWidget.setMaximumWidth(300)
        leftWidget_layout = QVBoxLayout(self.leftWidget)

        # SELECT SURVEY
        self.select_survey = QWidget()
        self.select_survey.setObjectName("select_survey_widget")
        self.select_survey.setStyleSheet(select_survey_style())
        select_survey_layout = QVBoxLayout(self.select_survey)

        self.title_survey_widget = QWidget()
        self.title_survey_widget.setObjectName("title_survey")
        self.title_survey_widget.setStyleSheet(title_survey_widget_style())
        title_layout = QHBoxLayout(self.title_survey_widget)
        title_layout.setContentsMargins(5, 4, 5, 0)  # left, top, right, bottom
        self.title_survey_widget.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        self.title_survey_widget.setMaximumHeight(40)
        self.title_survey_widget.setMinimumHeight(40)

        self.title_survey = QLabel("Imported survey")
        self.title_survey.setObjectName("title_survey")
        self.title_survey.setStyleSheet(title_survey_style())
        self.title_survey.setAlignment(Qt.AlignCenter | Qt.AlignVCenter)
        self.title_survey.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

        title_layout.addWidget(self.title_survey)

        separator = QFrame()
        separator.setFrameShape(QFrame.HLine)
        separator.setFrameShadow(QFrame.Sunken)
        separator.setStyleSheet("background-color: #7851A0;")  # opzionale: puoi personalizzare lo stile

        # Widget per la selezione della polarizzazione
        self.polarization_widget = QWidget()
        self.polarization_widget.setObjectName("polarization_widget")
        polarization_layout = QHBoxLayout(self.polarization_widget)
        polarization_layout.setContentsMargins(5, 5, 5, 5)

        polarization_label = QLabel("Polarization:")
        polarization_label.setStyleSheet("color: white;")
        self.polarization_combo = QComboBox()
        self.polarization_combo.addItems(["VV", "HH"])
        self.polarization_combo.setStyleSheet("""
            QComboBox {
                background-color: #444444;
                color: white;
                border: 1px solid #7851A0;
                border-radius: 3px;
                padding: 3px;
            }
            QComboBox::drop-down {
                border: none;
                background-color: #7851A0;
            }
            QComboBox::down-arrow {
                image: none;
                border-width: 0px;
            }
            QComboBox QAbstractItemView {
                background-color: #444444;
                color: white;
                selection-background-color: #7851A0;
            }
        """)

        polarization_layout.addWidget(polarization_label)
        polarization_layout.addWidget(self.polarization_combo)

        self.manage_survey = QWidget()
        self.manage_survey.setObjectName("manage_survey")
        manage_survey_layout = QVBoxLayout(self.manage_survey)
        self.manage_survey.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

        # Scroll Area
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        scroll_area.setStyleSheet("border: none;")

        # QTreeWidget statico
        self.tree_widget = QTreeWidget()
        self.tree_widget.setHeaderHidden(True)
        self.tree_widget.setStyleSheet("color: white;")
        self.tree_widget.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

        scroll_area.setWidget(self.tree_widget)
        manage_survey_layout.addWidget(scroll_area)

        select_survey_layout.addWidget(self.title_survey_widget)
        select_survey_layout.addWidget(separator)
        select_survey_layout.addWidget(self.polarization_widget)
        select_survey_layout.addWidget(self.manage_survey)

        leftWidget_layout.setContentsMargins(5, 5, 5, 5)
        leftWidget_layout.addWidget(self.select_survey)

        # CORREZIONE STRISCIATE
        self.correct_alignment_widget = QWidget()
        self.correct_alignment_widget.setObjectName("correct_alignment_widget")
        self.correct_alignment_widget.setStyleSheet(correct_alignment_widget_style())
        self.correct_alignment_widget.setFixedHeight(180)
        correct_alignment_layout = QVBoxLayout(self.correct_alignment_widget)

        self.title_correct_alignment_widget = QWidget()
        self.title_correct_alignment_widget.setObjectName("title_correct_alignment")
        self.title_correct_alignment_widget.setStyleSheet(title_correct_alignment_widget_style())
        title_correct_alignment_layout = QHBoxLayout(self.title_correct_alignment_widget)
        title_correct_alignment_layout.setContentsMargins(5, 4, 5, 0)  # left, top, right, bottom
        self.title_correct_alignment_widget.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        self.title_correct_alignment_widget.setMaximumHeight(40)
        self.title_correct_alignment_widget.setMinimumHeight(40)

        self.title_correct_alignment = QLabel("Correct alignment swath")
        self.title_correct_alignment.setObjectName("title_correct_alignment")
        self.title_correct_alignment.setStyleSheet(title_correct_alignment_style())
        self.title_correct_alignment.setAlignment(Qt.AlignCenter | Qt.AlignVCenter)
        self.title_correct_alignment.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        title_correct_alignment_layout.addWidget(self.title_correct_alignment)

        separator = QFrame()
        separator.setFrameShape(QFrame.HLine)
        separator.setFrameShadow(QFrame.Sunken)
        separator.setStyleSheet("background-color: #7851A0;")

        self.choose_correction_widget = QWidget()
        choose_correction_layout = QVBoxLayout(self.choose_correction_widget)

        self.spinbox = QDoubleSpinBox()
        self.spinbox.setRange(-1.0, 1.0)
        self.spinbox.setSingleStep(0.1)
        self.spinbox.setValue(0.0)
        self.spinbox.setFixedWidth(90)

        self.applay_correction = QPushButton("Applay")
        self.applay_correction.setObjectName("applay")
        self.applay_correction.setFixedSize(200, 30)
        self.applay_correction.setStyleSheet(applay_correction())

        choose_correction_layout.addWidget(self.spinbox, alignment=Qt.AlignCenter)
        choose_correction_layout.addWidget(self.applay_correction, alignment=Qt.AlignCenter)

        correct_alignment_layout.addWidget(self.title_correct_alignment_widget)
        correct_alignment_layout.addWidget(separator)
        correct_alignment_layout.addWidget(self.choose_correction_widget)

        leftWidget_layout.addWidget(self.correct_alignment_widget)
    def container_up_map(self):
        self.map_container = QWidget()
        self.map_layout = QVBoxLayout(self.map_container)
        self.map_layout.setContentsMargins(0, 0, 0, 0)
    def container_up_algorithm(self):
        self.algorithm = QWidget()
        algorithm_layout = QVBoxLayout(self.algorithm)
        algorithm_layout.setContentsMargins(10, 10, 10, 10)

        # Stack interno per commutare tra pannello Filtri (overview) e pannello Elaboration (algorithm)
        self.algorithm_stack = QStackedWidget()

        # --- View 0: pannello Filtri (quello esistente) ---
        self.filter = QWidget()
        self.filter.setObjectName("filtri_widget")
        self.filter.setStyleSheet(filtri_style())
        filter_layout = QVBoxLayout(self.filter)

        self.title_filter_widget = QWidget()
        self.title_filter_widget.setObjectName("title_filtri")
        self.title_filter_widget.setStyleSheet(title_filter_widget_style())
        title_layout = QHBoxLayout(self.title_filter_widget)
        title_layout.setContentsMargins(5, 4, 5, 0)
        self.title_filter_widget.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        self.title_filter_widget.setMaximumHeight(40)
        self.title_filter_widget.setMinimumHeight(40)

        self.title_filter = QLabel("Select filters")
        self.title_filter.setObjectName("title_filtri")
        self.title_filter.setStyleSheet(title_filter_style())
        self.title_filter.setAlignment(Qt.AlignCenter | Qt.AlignVCenter)
        self.title_filter.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

        self.add_filter = QPushButton("Add")
        self.add_filter.setIcon(QIcon(icon_path + '/add.png'))
        self.add_filter.setIconSize(QSize(10, 10))
        self.add_filter.setStyleSheet(add_filter_button())
        self.add_filter.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        self.add_filter.setCursor(Qt.PointingHandCursor)
        self.add_filter.setMinimumSize(50, 25)
        self.add_filter.setMaximumSize(50, 25)

        self.menu_filter = QMenu(self.add_filter)
        self.menu_filter.addAction("Fiter pass-band")
        self.menu_filter.addAction("Start time shifter")
        self.menu_filter.addAction("Background removal")
        self.menu_filter.addAction("Gain")
        self.menu_filter.addAction("Time-domain migration")
        self.menu_filter.addAction("Absolute value")
        self.menu_filter.addAction("Clip")
        self.menu_filter.setStyleSheet(add_filter_menu())
        self.add_filter.clicked.connect(self.open_filters_menu)

        title_layout.addWidget(self.title_filter)
        title_layout.addWidget(self.add_filter, alignment=Qt.AlignVCenter)

        separator = QFrame()
        separator.setFrameShape(QFrame.HLine)
        separator.setFrameShadow(QFrame.Sunken)
        separator.setStyleSheet("background-color: #7851A0;")

        self.manage_filter = QWidget()
        self.manage_filter.setObjectName("manage_filter")
        manage_filter_layout = QVBoxLayout(self.manage_filter)
        self.manage_filter.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        scroll_area.setStyleSheet("border: none;")

        scroll_content = QFrame()
        scroll_content.setObjectName("scroll_frame")
        scroll_content_layout = QVBoxLayout(scroll_content)
        scroll_content_layout.setContentsMargins(0, 0, 0, 0)
        scroll_content_layout.setSpacing(5)
        self.scroll_layout = scroll_content_layout
        scroll_area.setWidget(scroll_content)
        manage_filter_layout.addWidget(scroll_area)

        filter_layout.addWidget(self.title_filter_widget)
        filter_layout.addWidget(separator)
        filter_layout.addWidget(self.manage_filter)

        # --- View 1: pannello Elaboration con 2 checkbox ---
        elaboration_panel = self.create_elaboration_widget()

        self.algorithm_stack.addWidget(self.filter)          # index 0 -> overview/filters
        self.algorithm_stack.addWidget(elaboration_panel)    # index 1 -> algorithm/elaboration

        algorithm_layout.addWidget(self.algorithm_stack)

    def create_elaboration_widget(self):
        elaboration = QWidget()
        # Usa gli stessi objectName del pannello Filtri per ereditare esattamente lo stile
        elaboration.setObjectName("filtri_widget")
        elaboration.setStyleSheet(filtri_style())
        layout = QVBoxLayout(elaboration)

        title_widget = QWidget()
        title_widget.setObjectName("title_filtri")
        title_widget.setStyleSheet(title_filter_widget_style())
        title_layout = QHBoxLayout(title_widget)
        title_layout.setContentsMargins(5, 4, 5, 0)
        title_widget.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        title_widget.setMaximumHeight(40)
        title_widget.setMinimumHeight(40)

        title_label = QLabel("Elaboration")
        title_label.setObjectName("title_filtri")
        title_label.setStyleSheet(title_filter_style())
        title_label.setAlignment(Qt.AlignCenter | Qt.AlignVCenter)
        title_label.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        title_layout.addWidget(title_label)

        separator = QFrame()
        separator.setFrameShape(QFrame.HLine)
        separator.setFrameShadow(QFrame.Sunken)
        separator.setStyleSheet("background-color: #7851A0;")

        content = QWidget()
        content_layout = QVBoxLayout(content)
        content.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

        self.checkbox_ai_elaboration = QCheckBox("AI elaboration")
        self.checkbox_ai_elaboration.setStyleSheet("color: white;")
        self.checkbox_ai_detection = QCheckBox("AI Detection")
        self.checkbox_ai_detection.setStyleSheet("color: white;")
        # Checkbox per visualizzare tomografia inferenza (solo Open Project)
        self.checkbox_inference_view = QCheckBox("Show inference tomography")
        self.checkbox_inference_view.setStyleSheet("color: white;")
        # Checkbox per visualizzare dati AI nel radargram
        self.checkbox_show_AI_data = QCheckBox("Show AI data in radargram")
        self.checkbox_show_AI_data.setStyleSheet("color: white;")
        self.checkbox_show_AI_data.setToolTip(
            "Toggle between:\n"
            "☐ Unchecked: Data_current_filter (filtrati senza AI)\n"
            "☑ Checked: Data_AI (maschere AI)"
        )
        
        # Configurazione percorsi modelli
        from PySide6.QtWidgets import QLineEdit
        model_config_label = QLabel("AI Model Paths (optional):")
        model_config_label.setStyleSheet("color: white; font-weight: bold; margin-top: 10px;")
        
        self.input_model_long = QLineEdit()
        self.input_model_long.setPlaceholderText("Longitudinale (default if empty)")
        self.input_model_long.setStyleSheet("color: white; background-color: #444; border: 1px solid #666; padding: 4px;")
        
        self.input_model_trasv = QLineEdit()
        self.input_model_trasv.setPlaceholderText("Trasversale (default if empty)")
        self.input_model_trasv.setStyleSheet("color: white; background-color: #444; border: 1px solid #666; padding: 4px;")
        
        self.input_model_manhole = QLineEdit()
        self.input_model_manhole.setPlaceholderText("Manhole (default if empty)")
        self.input_model_manhole.setStyleSheet("color: white; background-color: #444; border: 1px solid #666; padding: 4px;")
        
        # Bottone per rebuild inference pipeline con nuovi modelli
        self.btn_rebuild_inference = QPushButton("Rebuild with new models")
        self.btn_rebuild_inference.setStyleSheet("""
            QPushButton {
                background-color: #9673a6;
                color: white;
                border: 1px solid #7851A0;
                border-radius: 5px;
                padding: 8px;
                font-weight: bold;
                margin-top: 5px;
            }
            QPushButton:hover {
                background-color: #7851A0;
            }
            QPushButton:pressed {
                background-color: #5a3d6e;
            }
            QPushButton:disabled {
                background-color: #555555;
                color: #888888;
            }
        """)
        self.btn_rebuild_inference.setToolTip(
            "Rebuild inference tomography and tiles with updated AI models.\n"
            "Leave paths empty to use defaults.\n"
            "This will:\n"
            "1. Run AI inference with new models on Data_current_filter\n"
            "2. Regenerate all tiles with new inference data\n"
            "3. Reprocess multiPHT on all tiles"
        )

        content_layout.addWidget(self.checkbox_ai_elaboration)
        content_layout.addWidget(self.checkbox_ai_detection)
        content_layout.addWidget(self.checkbox_inference_view)
        content_layout.addWidget(self.checkbox_show_AI_data)
        content_layout.addWidget(model_config_label)
        content_layout.addWidget(self.input_model_long)
        content_layout.addWidget(self.input_model_trasv)
        content_layout.addWidget(self.input_model_manhole)
        content_layout.addWidget(self.btn_rebuild_inference)
        content_layout.addStretch(1)

        layout.addWidget(title_widget)
        layout.addWidget(separator)
        layout.addWidget(content)
        return elaboration

    # CREAZIONE DELLA PARTE INFERIORE DELL'INTERFACCIA -----------------------------------------------------------------
    def create_radargram(self):
        radar_layout = QHBoxLayout(self.radargram)

        self.radar_combobox = QWidget()
        self.radar_combobox.setObjectName("radar_combobox")
        radar_combobox_layout = QVBoxLayout(self.radar_combobox)
        radar_combobox_layout.setSpacing(10)

        self.title_choose = QLabel("Select swath and channels")
        self.title_choose.setObjectName("title_choose")
        self.title_choose.setAlignment(Qt.AlignCenter)
        self.title_choose.setStyleSheet(title_combobox())

        self.swath_widget = QWidget()
        self.swath_widget.setObjectName("swath_widget")
        swath_widget_layout = QHBoxLayout(self.swath_widget)
        self.label_swath = QLabel("Swath: ")
        self.label_swath.setStyleSheet(swath_label_style())
        self.combo_swath = QComboBox()
        swath_widget_layout.addWidget(self.label_swath)
        swath_widget_layout.addWidget(self.combo_swath)
        self.swath_widget.setStyleSheet(swath_widget_style())

        self.channel_widget = QWidget()
        self.channel_widget.setObjectName("channel_widget")
        channel_widget_layout = QHBoxLayout(self.channel_widget)
        self.label_channel = QLabel("Channel: ")
        self.label_channel.setStyleSheet(channel_label_style())
        self.combo_channel = QComboBox()
        channel_widget_layout.addWidget(self.label_channel)
        channel_widget_layout.addWidget(self.combo_channel)
        self.channel_widget.setStyleSheet(channel_widget_style())

        # Prevent left controls from stretching too much
        self.label_swath.setFixedWidth(60)
        self.combo_swath.setFixedWidth(120)
        self.label_channel.setFixedWidth(60)
        self.combo_channel.setFixedWidth(120)
        swath_widget_layout.setContentsMargins(0, 0, 0, 0)
        channel_widget_layout.setContentsMargins(0, 0, 0, 0)

        radar_combobox_layout.addWidget(self.title_choose)
        radar_combobox_layout.addWidget(self.swath_widget)
        radar_combobox_layout.addWidget(self.channel_widget)

        # Values Range widget (moved beside swath/channel as part of Radargram area)
        self.clip_widget_small = QWidget()
        self.clip_widget_small.setObjectName("clip_widget_small")
        self.clip_widget_small.setStyleSheet("#clip_widget_small { border: 1px solid #9ECB78; border-radius: 6px; }")
        clip_layout = QVBoxLayout(self.clip_widget_small)
        clip_layout.setContentsMargins(8, 6, 8, 6)
        clip_layout.setSpacing(6)

        clip_title = QLabel("Values Range")
        clip_title.setStyleSheet("color: white;")
        clip_title.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)

        clip_row = QHBoxLayout()
        clip_row.setContentsMargins(0, 0, 0, 0)
        clip_row.setSpacing(8)

        self.clip_slider_small = QSlider(Qt.Horizontal)
        self.clip_slider_small.setMinimum(0)
        self.clip_slider_small.setMaximum(1000)
        self.clip_slider_small.setValue(0)
        self.clip_slider_small.setMaximumWidth(150)
        self.clip_slider_small.setStyleSheet("""
            QSlider::groove:horizontal { height: 6px; background: #444; border: 1px solid #7851A0; border-radius: 3px; }
            QSlider::handle:horizontal { background: #7851A0; width: 14px; margin: -4px 0; border-radius: 7px; }
        """)

        self.clip_value_label = QLabel("[ -0.000 : 0.000 ]")
        self.clip_value_label.setStyleSheet("color: #BBBBBB;")
        self.clip_value_label.setMinimumWidth(120)

        clip_row.addWidget(self.clip_slider_small)
        clip_row.addWidget(self.clip_value_label)

        clip_layout.addWidget(clip_title)
        clip_layout.addLayout(clip_row)

        # Imposta un'altezza fissa di 56 pixel per il widget dei valori di range
        self.clip_widget_small.setFixedHeight(56)
        # Aggiunge il widget dei valori di range al layout principale dei controlli radar
        radar_combobox_layout.addWidget(self.clip_widget_small)
        # Aggiunge uno spazio di 10 pixel dopo il widget dei valori di range
        radar_combobox_layout.addSpacing(10)
        
        # Horizontal Scale widget (reduce width visually)
        self.hscale_widget = QWidget()
        self.hscale_widget.setObjectName("hscale_widget")
        self.hscale_widget.setStyleSheet("#hscale_widget { border: 1px solid #9ECB78; border-radius: 6px; }")
        hscale_layout = QVBoxLayout(self.hscale_widget)
        hscale_layout.setContentsMargins(8, 6, 8, 6)
        hscale_layout.setSpacing(6)

        hscale_title = QLabel("Horizontal scale")
        hscale_title.setStyleSheet("color: white;")
        hscale_title.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)

        hscale_row = QHBoxLayout()
        hscale_row.setContentsMargins(0, 0, 0, 0)
        hscale_row.setSpacing(8)

        self.hscale_slider = QSlider(Qt.Horizontal)
        self.hscale_slider.setMinimum(10)  # 10%
        self.hscale_slider.setMaximum(100) # 100%
        self.hscale_slider.setValue(100)
        self.hscale_slider.setMaximumWidth(150)
        self.hscale_slider.setStyleSheet("""
            QSlider::groove:horizontal { height: 6px; background: #444; border: 1px solid #7851A0; border-radius: 3px; }
            QSlider::handle:horizontal { background: #7851A0; width: 14px; margin: -4px 0; border-radius: 7px; }
        """)

        self.hscale_value_label = QLabel("100%")
        self.hscale_value_label.setStyleSheet("color: #BBBBBB;")
        self.hscale_value_label.setMinimumWidth(50)

        hscale_row.addWidget(self.hscale_slider)
        hscale_row.addWidget(self.hscale_value_label)

        hscale_layout.addWidget(hscale_title)
        hscale_layout.addLayout(hscale_row)

        self.hscale_widget.setMinimumHeight(56)
        radar_combobox_layout.addWidget(self.hscale_widget)
        radar_combobox_layout.setContentsMargins(10, 10, 10, 10)
        self.radar_combobox.setStyleSheet(radar_combobox_style())
        self.radar_combobox.setFixedWidth(220)
        self.radar_combobox.setMaximumHeight(300)
        self.radar_combobox.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Preferred)
        radar_combobox_layout.setAlignment(Qt.AlignTop)

        radar_layout.addWidget(self.radar_combobox)
        radar_layout.setContentsMargins(5, 5, 5, 5)

        # CONTAINER PER PLOT E SLIDER
        plot_container = QWidget()
        plot_container_layout = QVBoxLayout(plot_container)
        plot_container_layout.setContentsMargins(0, 0, 0, 0)
        plot_container_layout.setSpacing(5)
        plot_container.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

        # PLOT
        self.plot_widget = pg.PlotWidget()
        self.plot_widget.setBackground('#3F3F3F')
        self.plot_widget.setTitle("Radargramma")
        self.plot_widget.showGrid(x=False, y=False)
        self.plot_widget.setLabel('left', 'Depth (z)')
        self.plot_widget.setLabel('bottom', 'Distance (m)')
        self.image_item = pg.ImageItem()
        self.plot_widget.addItem(self.image_item)
        
        # SLIDER PER LE SUBSWATH
        self.subswath_slider = QSlider(Qt.Horizontal)
        self.subswath_slider.setMaximumHeight(30)
        self.subswath_slider.setStyleSheet("""
            QSlider::groove:horizontal {
                border: 1px solid #7851A0;
                height: 8px;
                background: #444444;
                border-radius: 4px;
            }
            QSlider::handle:horizontal {
                background: #7851A0;
                border: 1px solid #7851A0;
                width: 18px;
                margin: -5px 0;
                border-radius: 9px;
            }
            QSlider::handle:horizontal:hover {
                background: #9ECB78;
            }
        """)
        self.subswath_slider.setMinimum(0)
        self.subswath_slider.setMaximum(0)
        self.subswath_slider.setValue(0)
        
        # Aggiungo plot e slider al container
        plot_container_layout.addWidget(self.plot_widget)
        plot_container_layout.addWidget(self.subswath_slider)
        
        # (Rimosso) slider clipping grande non più necessario
        
        radar_layout.addWidget(plot_container)
        radar_layout.setStretch(0, 0)
        radar_layout.setStretch(1, 1)

    
    # FUNZIONE PER FAR APPARIRE IL MENU A TENDINA DEI FILTRI (BRUTTO MA RIMANE COSI PER ORA)
    def open_filters_menu(self):
        pos = self.add_filter.mapToGlobal(self.add_filter.rect().bottomLeft())
        self.menu_filter.exec_(pos)
    
    # SET DEL WIDGET DELLA MAPPA
    def setMap(self, map):
        self.map = map
        self.map_layout.addWidget(self.map)

    def closeEvent(self, event):
        reply = QMessageBox.question(self, 'Confirm',
                                     "Are you sure to quit the application?",
                                     QMessageBox.Yes | QMessageBox.No, QMessageBox.No)
        if reply == QMessageBox.Yes:
            # Cleanup threads and CUDA before closing
            try:
                import torch
                if torch.cuda.is_available():
                    print("[MainWindow] Pulizia cache CUDA durante chiusura...")
                    torch.cuda.empty_cache()
                    print("[MainWindow] ✓ Cache CUDA pulita")
            except Exception as e:
                print(f"[MainWindow] Errore pulizia CUDA: {e}")
            
            event.accept()
        else:
            event.ignore()

